/**
 * Off-Plan Search AJAX Functionality
 */
(function($) {
    'use strict';

    var HouzezOffplanSearch = {
        
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.initSelectpicker();
            this.updateFilterClearVisibility();
        },
        
        /**
         * Bind events
         */
        bindEvents: function() {
            var self = this;
            
            // Search button click
            $(document).on('click', '#offplan-search-btn', function(e) {
                e.preventDefault();
                self.performSearch(1);
            });
            
            // Per-field clear (x) click
            $(document).on('click', '.offplan-filter-clear', function(e) {
                e.preventDefault();
                var targetId = $(this).data('target');
                var $select = $('#' + targetId);
                if (!$select.length) return;
                if (typeof $.fn.selectpicker !== 'undefined' && $select.hasClass('selectpicker')) {
                    if ($select.attr('multiple') || ($select.attr('name') && $select.attr('name').indexOf('[]') !== -1)) {
                        $select.selectpicker('val', []);
                    } else {
                        $select.selectpicker('val', '');
                    }
                    $select.selectpicker('refresh');
                } else {
                    if ($select.attr('multiple') || ($select.attr('name') && $select.attr('name').indexOf('[]') !== -1)) {
                        $select.val([]);
                    } else {
                        $select.val('');
                    }
                }
                self.updateFilterClearVisibility();
                var formData = $('#offplan-search-form').serialize();
                var sortBy = $('#offplan_sort_properties').val() || '';
                if (sortBy) formData += '&sortby=' + encodeURIComponent(sortBy);
                self.updateUrl(formData, 1);
                self.performSearch(1);
            });
            
            // Dropdown change events (debounced for smoothness)
            var searchTimeout;
            $(document).on('change', '.offplan-filter', function() {
                clearTimeout(searchTimeout);
                setTimeout(function() { self.updateFilterClearVisibility(); }, 80);
                searchTimeout = setTimeout(function() { self.performSearch(1); }, 280);
            });
            $(document).on('changed.bs.select', '.offplan-filter', function() {
                clearTimeout(searchTimeout);
                setTimeout(function() { self.updateFilterClearVisibility(); }, 50);
                searchTimeout = setTimeout(function() { self.performSearch(1); }, 280);
            });
            
            // Also listen to the rendered.bs.select event (when dropdown closes)
            $(document).on('rendered.bs.select', '.offplan-filter', function() {
                setTimeout(function() {
                    self.updateFilterClearVisibility();
                }, 50);
            });
            
            // Listen to hidden.bs.select (when dropdown is closed) - run visibility after close so clear icon shows for new selection
            $(document).on('hidden.bs.select', '.offplan-filter', function() {
                setTimeout(function() {
                    self.updateFilterClearVisibility();
                }, 100);
            });
            
            // Sort dropdown change event
            $(document).on('changed.bs.select', '#offplan_sort_properties', function() {
                self.performSearch(1);
            });
            
            // Pagination clicks
            $(document).on('click', '.offplan-pagination a', function(e) {
                e.preventDefault();
                var href = $(this).attr('href');
                var page = self.getPageFromUrl(href);
                if (page) {
                    self.performSearch(page);
                }
            });
            
            // Handle browser back/forward buttons
            $(window).on('popstate', function(e) {
                if (e.originalEvent.state) {
                    self.performSearch(e.originalEvent.state.page, false);
                }
            });
        },
        
        /**
         * Initialize selectpicker
         */
        initSelectpicker: function() {
            if (typeof $.fn.selectpicker !== 'undefined') {
                $('.selectpicker.offplan-filter').selectpicker();
                $('#offplan_sort_properties').selectpicker();
            }
        },
        
        /**
         * Perform AJAX search
         */
        performSearch: function(page, updateUrl) {
            var self = this;
            if (typeof houzezOffplan === 'undefined') {
                return;
            }
            page = page || 1;
            updateUrl = (updateUrl !== false);
            
            var $form = $('#offplan-search-form');
            var $container = $('#offplan-results-container');
            if (!$container.length) {
                return;
            }
            if (self._pendingRequest && self._pendingRequest.abort) {
                self._pendingRequest.abort();
            }
            // Find pagination wrapper relative to container
            var $pagination = $container.siblings('.offplan-pagination-wrapper');
            if (!$pagination.length) {
                $pagination = $container.parent().find('.offplan-pagination-wrapper');
            }
            var cardStyle = $container.data('card-style') || 'v7';
            var pageId = $container.data('page-id') || 0;
            var columns = $container.data('columns') || 2;
            var layoutCss = $container.data('layout-css') || $container.attr('class') || 'listing-view grid-view row row-cols-1 row-cols-md-2 gy-4 gx-4';
            
            // Get form data
            var formData = $form.serialize();
            
            // Get sort value
            var sortBy = $('#offplan_sort_properties').val() || '';
            if (sortBy) {
                formData += '&sortby=' + encodeURIComponent(sortBy);
            }
            
            // Get posts per page from container
            var postsPerPage = $container.data('posts-per-page') || 9;
            
            // Prepare AJAX data
            var ajaxData = {
                action: 'houzez_offplan_listings',
                nonce: houzezOffplan.nonce,
                paged: page,
                card_style: cardStyle,
                page_id: pageId,
                layout_css: layoutCss,
                posts_per_page: postsPerPage,
                columns: columns
            };
            
            // Merge form data
            var finalData = formData + '&' + $.param(ajaxData);
            
            self.showLoading($container);
            
            self._pendingRequest = $.ajax({
                type: 'GET',
                dataType: 'json',
                url: houzezOffplan.ajaxurl,
                data: finalData,
                success: function(response) {
                    if (response.success) {
                        // Update results container HTML (response.html contains just the property cards)
                        $container.html(response.html);
                        
                        // Ensure container maintains correct classes and data attributes
                        $container.attr('class', layoutCss);
                        $container.attr('data-columns', columns);
                        $container.attr('data-layout-css', layoutCss);
                        $container.attr('data-card-style', cardStyle);
                        $container.attr('data-view', 'grid');
                        
                        // Update pagination
                        if (response.pagination) {
                            if ($pagination.length) {
                                // Replace existing pagination
                                $pagination.html(response.pagination).show();
                            } else {
                                // Create pagination wrapper if it doesn't exist (shouldn't happen, but safety check)
                                $container.after('<div class="offplan-pagination-wrapper">' + response.pagination + '</div>');
                                // Update $pagination reference for future use
                                $pagination = $container.siblings('.offplan-pagination-wrapper');
                            }
                        } else {
                            // If no pagination in response, hide the wrapper
                            if ($pagination.length) {
                                $pagination.hide();
                            }
                        }
                        
                        // Update URL without page reload
                        if (updateUrl) {
                            self.updateUrl(formData, page);
                        }
                        
                        // Scroll to results (short duration for snappy feel)
                        $('html, body').animate({
                            scrollTop: $container.offset().top - 100
                        }, 280);
                        
                        // Reinitialize selectpicker if needed
                        self.initSelectpicker();
                        
                        // Reinitialize sort dropdown
                        if (typeof $.fn.selectpicker !== 'undefined') {
                            $('#offplan_sort_properties').selectpicker('refresh');
                        }
                        
                        // Ensure grid gallery loads: theme only outputs data-images when property has gallery images.
                        // For cards with only featured image, inject data-images so the theme's gridImageGallery can build the carousel.
                        self.ensureGalleryDataAndInit($container);
                        
                        // Update per-field clear (x) visibility after search
                        self.updateFilterClearVisibility();
                    } else {
                        self.showError($container, response.data || 'An error occurred');
                    }
                },
                error: function(xhr, status, error) {
                    if (status !== 'abort') {
                        self.showError($container, 'AJAX Error: ' + error);
                    }
                },
                complete: function(xhr, status) {
                    if (self._pendingRequest === xhr) self._pendingRequest = null;
                    if (status !== 'abort') self.hideLoading();
                }
            });
        },
        
        /**
         * Show loading indicator
         */
        showLoading: function($container) {
            $container.addClass('loading');
            if (!$('#offplan-loading').length) {
                $container.before('<div id="offplan-loading" class="text-center p-4"><div class="spinner-border spinner-border-sm" role="status"><span class="visually-hidden">Loading...</span></div></div>');
            }
        },
        
        hideLoading: function() {
            $('#offplan-results-container').removeClass('loading');
            $('#offplan-loading').remove();
        },
        
        /**
         * Show error message
         */
        showError: function($container, message) {
            $container.html('<div class="alert alert-danger">' + message + '</div>');
        },
        
        /**
         * Update URL without page reload
         */
        updateUrl: function(formData, page) {
            var params = new URLSearchParams(formData);
            if (page > 1) {
                params.set('paged', page);
            } else {
                params.delete('paged');
            }
            
            var newUrl = window.location.pathname + '?' + params.toString();
            window.history.pushState({page: page}, '', newUrl);
        },
        
        /**
         * Get page number from URL
         */
        getPageFromUrl: function(url) {
            var match = url.match(/[?&]paged=(\d+)/);
            return match ? parseInt(match[1]) : 1;
        },
        
        /**
         * Ensure listing cards have gallery data (data-images) and run theme grid gallery init.
         * Theme only outputs data-images when property has gallery images; for featured-only we inject it from the img.
         */
        ensureGalleryDataAndInit: function($container) {
            var self = this;
            if (!$container || !$container.length) return;
            $container.find('.hz-item-gallery-js').each(function() {
                var $card = $(this);
                $card.removeClass('houzez-gallery-loaded');
                if ($card.data('images') && $card.attr('data-images')) return;
                var $img = $card.find('.listing-image-wrap .listing-featured-thumb img, .listing-image-wrap .listing-thumb a img').first();
                if (!$img.length) $img = $card.find('.listing-thumb img, .listing-image-wrap img').first();
                if ($img.length) {
                    var src = $img.attr('src');
                    if (src) {
                        var imgEl = $img[0];
                        var single = [{
                            image: src,
                            alt: $img.attr('alt') || '',
                            width: imgEl.naturalWidth || $img.attr('width') || 800,
                            height: imgEl.naturalHeight || $img.attr('height') || 600
                        }];
                        if ($img.attr('srcset')) single[0].srcset = $img.attr('srcset');
                        if ($img.attr('sizes')) single[0].sizes = $img.attr('sizes');
                        $card.attr('data-images', JSON.stringify(single));
                    }
                }
            });
            setTimeout(function() {
                if (typeof window.houzez_grid_image_gallery === 'function') {
                    window.houzez_grid_image_gallery();
                } else if (typeof houzez !== 'undefined' && houzez.Sliders && typeof houzez.Sliders.gridImageGallery === 'function') {
                    houzez.Sliders.gridImageGallery();
                }
            }, 80);
        },
        
        /**
         * Update per-field clear (x) visibility: toggle has-value on each .offplan-filter-wrap
         */
        updateFilterClearVisibility: function() {
            var $form = $('#offplan-search-form');
            $form.find('select.offplan-filter').each(function() {
                var $select = $(this);
                var $wrap = $select.closest('.offplan-filter-wrap');
                if (!$wrap.length) return;
                var hasValue = false;
                var value = null;
                try {
                    if (typeof $.fn.selectpicker !== 'undefined' && $select.hasClass('selectpicker')) {
                        try {
                            value = $select.selectpicker('val');
                        } catch (e) {}
                        if (!value || (Array.isArray(value) && value.length === 0)) {
                            var selectedOptions = $select.find('option:selected').filter(function() {
                                return $(this).val() !== '';
                            });
                            if (selectedOptions.length > 0) hasValue = true;
                        }
                        if (Array.isArray(value)) {
                            var filtered = value.filter(function(v) { return v !== '' && v !== null && v !== undefined; });
                            if (filtered.length > 0) hasValue = true;
                        } else if (value && value !== '' && value !== null && value !== undefined) {
                            hasValue = true;
                        }
                    } else {
                        value = $select.val();
                        if (value !== '' && value !== null) {
                            if (Array.isArray(value)) {
                                var filtered = value.filter(function(v) { return v !== '' && v !== null && v !== undefined; });
                                if (filtered.length > 0) hasValue = true;
                            } else {
                                hasValue = true;
                            }
                        }
                    }
                    if (!hasValue) {
                        var $selected = $select.find('option:selected').filter(function() { return $(this).val() !== ''; });
                        if ($selected.length > 0) hasValue = true;
                    }
                } catch (e) {
                    try {
                        value = $select.val();
                        if (value && value !== '' && value !== null) hasValue = true;
                    } catch (e2) {}
                }
                $wrap.toggleClass('has-value', !!hasValue);
            });
        }
    };
    
    // Expose for inline script (e.g. template) to call updateFilterClearVisibility after load
    window.HouzezOffplanSearch = HouzezOffplanSearch;
    
    $(document).ready(function() {
        if ($('#offplan-search-form').length) {
            HouzezOffplanSearch.init();
            setTimeout(function() { HouzezOffplanSearch.updateFilterClearVisibility(); }, 150);
        }
    });
    
})(jQuery);

