<?php
/**
 * Plugin Name: Houzez Off-Plan Search
 * Plugin URI: https://example.com/houzez-offplan-search
 * Description: Adds an off-plan property search shortcode with AJAX filtering for Houzez theme. Includes three taxonomy filters (Community, Projects, Property Type) and selectable card styles. Use [houzez_offplan_search] shortcode anywhere.
 * Version: 1.0.3
 * Author: Dev Mizan
 * Author URI: https://devmizan.com
 * Text Domain: houzez-offplan-search
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('HOUZEZ_OFFPLAN_VERSION', '1.0.3');
define('HOUZEZ_OFFPLAN_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('HOUZEZ_OFFPLAN_PLUGIN_URL', plugin_dir_url(__FILE__));
define('HOUZEZ_OFFPLAN_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Check if Houzez theme is active
 */
function houzez_offplan_check_theme() {
    $theme = wp_get_theme();
    $theme_name = $theme->get('Name');
    $theme_template = $theme->get('Template');
    
    // Check if Houzez theme is active
    if (strpos(strtolower($theme_name), 'houzez') !== false || 
        strpos(strtolower($theme_template), 'houzez') !== false ||
        function_exists('houzez_option')) {
        return true;
    }
    
    return false;
}

/**
 * Display admin notice if Houzez theme is not active
 */
function houzez_offplan_admin_notice() {
    if (!houzez_offplan_check_theme()) {
        ?>
        <div class="notice notice-error">
            <p><?php esc_html_e('Houzez Off-Plan Search plugin requires the Houzez theme to be installed and active.', 'houzez-offplan-search'); ?></p>
        </div>
        <?php
    }
}
add_action('admin_notices', 'houzez_offplan_admin_notice');

/**
 * Check required Houzez functions exist
 */
function houzez_offplan_check_dependencies() {
    $required_functions = array(
        'houzez_option',
        'houzez_get_search_taxonomies',
    );
    
    $missing = array();
    foreach ($required_functions as $function) {
        if (!function_exists($function)) {
            $missing[] = $function;
        }
    }
    
    return empty($missing);
}

/**
 * Initialize plugin - Load classes and register shortcode early
 */
function houzez_offplan_load_classes() {
    // Load plugin classes (don't check dependencies yet, do it later)
    require_once HOUZEZ_OFFPLAN_PLUGIN_DIR . 'includes/class-offplan-search.php';
    require_once HOUZEZ_OFFPLAN_PLUGIN_DIR . 'includes/class-offplan-filters.php';
    require_once HOUZEZ_OFFPLAN_PLUGIN_DIR . 'includes/class-offplan-ajax.php';
    require_once HOUZEZ_OFFPLAN_PLUGIN_DIR . 'includes/class-offplan-shortcode.php';
    require_once HOUZEZ_OFFPLAN_PLUGIN_DIR . 'includes/class-status-forms-shortcode.php';
    
    // Initialize shortcode immediately (register shortcode early)
    if (class_exists('Houzez_Offplan_Shortcode')) {
        new Houzez_Offplan_Shortcode();
    }
    if (class_exists('Houzez_Status_Forms_Shortcode')) {
        new Houzez_Status_Forms_Shortcode();
    }
}
// Load classes on plugins_loaded but before init
add_action('plugins_loaded', 'houzez_offplan_load_classes', 5);

/**
 * Initialize plugin - Initialize other components
 */
function houzez_offplan_init() {
    // Check if Houzez theme is active
    if (!houzez_offplan_check_theme()) {
        return;
    }
    
    // Initialize main class
    if (class_exists('Houzez_Offplan_Search')) {
        $houzez_offplan = new Houzez_Offplan_Search();
        $houzez_offplan->init();
    }
}

// Hook into plugins_loaded to ensure theme is loaded first
add_action('plugins_loaded', 'houzez_offplan_init');

/**
 * Activation hook
 */
function houzez_offplan_activate() {
    // Check if Houzez theme is active
    if (!houzez_offplan_check_theme()) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('Houzez Off-Plan Search plugin requires the Houzez theme to be installed and active.', 'houzez-offplan-search'),
            esc_html__('Plugin Activation Error', 'houzez-offplan-search'),
            array('back_link' => true)
        );
    }
    
    // Flush rewrite rules if needed
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'houzez_offplan_activate');

/**
 * Deactivation hook
 */
function houzez_offplan_deactivate() {
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'houzez_offplan_deactivate');

