<?php
/**
 * AJAX handler class
 *
 * @package Houzez_Offplan_Search
 */

if (!defined('ABSPATH')) {
    exit;
}

class Houzez_Offplan_Ajax {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_ajax_houzez_offplan_listings', array($this, 'offplan_listings'));
        add_action('wp_ajax_nopriv_houzez_offplan_listings', array($this, 'offplan_listings'));
    }
    
    /**
     * Handle AJAX requests for off-plan listings
     */
    public function offplan_listings() {
        // Verify nonce (allow GET requests for AJAX)
        if (isset($_GET['nonce'])) {
            if (!wp_verify_nonce($_GET['nonce'], 'houzez_offplan_nonce')) {
                wp_send_json_error(array('message' => 'Invalid nonce'));
                return;
            }
        }
        
        $tax_query = array();
        $meta_query = array();
        $keyword_array = '';
        
        $keyword_field = function_exists('houzez_option') ? houzez_option('keyword_field') : '';
        
        // Get posts per page from GET or use default
        $number_of_prop = isset($_GET['posts_per_page']) ? absint($_GET['posts_per_page']) : 
                         (function_exists('houzez_option') ? absint(houzez_option('search_num_posts')) : 9);
        if (!$number_of_prop) {
            $number_of_prop = 9;
        }
        
        $paged = absint($_GET['paged'] ?? 1);
        $page_id = absint($_GET['page_id'] ?? 0);
        
        // Get card style from GET parameter (shortcode attribute)
        $card_style = 'v7'; // Default
        if (isset($_GET['card_style']) && !empty($_GET['card_style'])) {
            $card_style = sanitize_text_field($_GET['card_style']);
        }
        if (empty($card_style)) {
            $card_style = 'v7';
        }
        
        // Validate card style
        $valid_card_styles = array('v1', 'v2', 'v3', 'v4', 'v5', 'v6', 'v7');
        if (!in_array($card_style, $valid_card_styles, true)) {
            $card_style = 'v7';
        }
        
        $search_qry = array(
            'post_type' => 'property',
            'posts_per_page' => $number_of_prop,
            'paged' => $paged,
            'post_status' => 'publish'
        );
        
        // Apply sold status filter if Houzez function exists
        if (function_exists('houzez_prop_sort')) {
            $search_qry = apply_filters('houzez_sold_status_filter', $search_qry);
        }
        
        // Process keyword search
        if (isset($_GET['keyword']) && !empty($_GET['keyword']) && $keyword_field) {
            $keyword = sanitize_text_field($_GET['keyword']);
            
            if ($keyword_field == 'prop_address') {
                // Handle address keyword if Houzez function exists
                if (class_exists('Houzez_Property_Search')) {
                    $keyword_array = Houzez_Property_Search::get_address_keyword_filter($keyword);
                }
            } else if ($keyword_field == 'prop_city_state_county') {
                if (class_exists('Houzez_Property_Search')) {
                    $tax_query[] = Houzez_Property_Search::get_location_keyword_filter($keyword, array());
                }
            } else {
                if (class_exists('Houzez_Property_Search')) {
                    $search_qry = Houzez_Property_Search::keyword_search($search_qry);
                }
            }
        }
        
        // Apply taxonomy filters (includes developer filter from our class)
        $tax_query = apply_filters('houzez_taxonomy_search_filter', $tax_query);
        $tax_count = count($tax_query);
        
        if ($tax_count > 1) {
            $tax_query['relation'] = 'AND';
        }
        
        if ($tax_count > 0) {
            $search_qry['tax_query'] = $tax_query;
        }
        
        // Apply meta filters
        $meta_query = apply_filters('houzez_meta_search_filter', $meta_query);
        $meta_count = count($meta_query);
        
        if ($meta_count > 0 || !empty($keyword_array)) {
            $search_qry['meta_query'] = array(
                'relation' => 'AND',
                $keyword_array,
                array(
                    'relation' => 'AND',
                    $meta_query
                ),
            );
        }
        
        // Force off-plan status filter
        if (!isset($search_qry['tax_query'])) {
            $search_qry['tax_query'] = array();
        }
        
        $has_offplan = false;
        foreach ($search_qry['tax_query'] as $tax_query) {
            if (isset($tax_query['taxonomy']) && $tax_query['taxonomy'] === 'property_status') {
                if (isset($tax_query['terms']) && 
                    ((is_array($tax_query['terms']) && in_array('off-plan', $tax_query['terms'])) ||
                    $tax_query['terms'] === 'off-plan')) {
                    $has_offplan = true;
                    break;
                }
            }
        }
        
        if (!$has_offplan) {
            $search_qry['tax_query'][] = array(
                'taxonomy' => 'property_status',
                'field' => 'slug',
                'terms' => 'off-plan',
            );
            
            if (count($search_qry['tax_query']) > 1) {
                $search_qry['tax_query']['relation'] = 'AND';
            }
        }
        
        // Apply sorting
        if (function_exists('houzez_prop_sort')) {
            $search_qry = houzez_prop_sort($search_qry);
        }
        
        // Get columns and build layout CSS
        $columns = isset($_GET['columns']) ? absint($_GET['columns']) : 2;
        $valid_columns = array(2, 3, 4);
        if (!in_array($columns, $valid_columns, true)) {
            $columns = 2;
        }
        
        // Build layout CSS based on columns
        $layout_css = 'listing-view grid-view row';
        if ($columns == 4) {
            $layout_css .= ' row-cols-1 row-cols-xl-4 row-cols-lg-3 row-cols-md-2 gy-4 gx-4';
        } elseif ($columns == 3) {
            $layout_css .= ' row-cols-lg-3 row-cols-md-2 row-cols-sm-1 gy-4 gx-4';
        } else {
            $layout_css .= ' row-cols-1 row-cols-md-2 gy-4 gx-4';
        }
        
        // Allow override from GET parameter
        if (isset($_GET['layout_css']) && !empty($_GET['layout_css'])) {
            $layout_css = sanitize_text_field($_GET['layout_css']);
        }
        
        // Main listings query
        $query = new WP_Query($search_qry);
        
        ob_start();
        
        // Output property cards directly (no wrapper div - container already has the classes)
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                
                // Load Houzez theme template part
                $template_path = locate_template(array('template-parts/listing/item-' . $card_style . '.php'));
                if ($template_path) {
                    get_template_part('template-parts/listing/item', $card_style);
                } else {
                    // Fallback to v1 if card style template doesn't exist
                    get_template_part('template-parts/listing/item', 'v1');
                }
            }
        } else {
            echo '<div class="search-no-results-found flex-grow-1 text-center mx-4">' . esc_html__("We didn't find any results", 'houzez-offplan-search') . '</div>';
        }
        
        wp_reset_postdata();
        
        $html = ob_get_clean();
        
        // Pagination
        ob_start();
        if (function_exists('houzez_ajax_pagination')) {
            houzez_ajax_pagination($query->max_num_pages);
        } else {
            // Basic pagination fallback
            $pagination = paginate_links(array(
                'base' => str_replace(999999999, '%#%', esc_url(get_pagenum_link(999999999))),
                'total' => $query->max_num_pages,
                'current' => max(1, $paged),
                'format' => '?paged=%#%',
                'show_all' => false,
                'type' => 'list',
                'end_size' => 2,
                'mid_size' => 1,
                'prev_next' => true,
                'prev_text' => '&laquo;',
                'next_text' => '&raquo;',
            ));
            echo $pagination;
        }
        $pagination_html = ob_get_clean();
        
        $response = array(
            'success' => true,
            'html' => $html,
            'pagination' => $pagination_html,
            'total' => $query->found_posts,
            'max_pages' => $query->max_num_pages,
            'current_page' => $paged,
        );
        
        wp_send_json($response);
    }
}

