<?php
/**
 * Filter handler class
 *
 * @package Houzez_Offplan_Search
 */

if (!defined('ABSPATH')) {
    exit;
}

class Houzez_Offplan_Filters {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Hook into taxonomy search filter for developer and projects taxonomy
        add_filter('houzez_taxonomy_search_filter', array($this, 'search_developer'), 10, 1);
        add_filter('houzez_taxonomy_search_filter', array($this, 'search_projects'), 10, 1);
        
        // Hook into property filter to force off-plan status
        add_filter('houzez20_property_filter', array($this, 'filter_offplan_status'), 10, 1);
    }
    
    /**
     * Filter search results by developer taxonomy (property_developers)
     * 
     * @param array $query_arg Current taxonomy query arguments
     * @return array Modified taxonomy query arguments
     */
    public function search_developer($query_arg) {
        if (!isset($_GET['developer']) || empty($_GET['developer'])) {
            return $query_arg;
        }
        
        $developer = sanitize_text_field($_GET['developer']);
        
        // Skip if "any" or empty
        if ($developer == 'all' || $developer == '-1' || $developer == '') {
            return $query_arg;
        }
        
        $query_arg[] = array(
            'taxonomy' => 'property_developers',
            'field' => 'slug',
            'terms' => $developer,
        );
        
        return $query_arg;
    }
    
    /**
     * Filter search results by projects taxonomy (property_developers)
     * 
     * @param array $query_arg Current taxonomy query arguments
     * @return array Modified taxonomy query arguments
     */
    public function search_projects($query_arg) {
        if (!isset($_GET['projects']) || empty($_GET['projects'])) {
            return $query_arg;
        }
        
        $projects = $_GET['projects'];
        
        // Handle array or single value
        if (is_array($projects)) {
            $projects = array_map('sanitize_text_field', $projects);
            $projects = array_filter($projects);
            if (empty($projects)) {
                return $query_arg;
            }
        } else {
            $projects = sanitize_text_field($projects);
            // Skip if "any" or empty
            if ($projects == 'all' || $projects == '-1' || $projects == '') {
                return $query_arg;
            }
            $projects = array($projects);
        }
        
        $query_arg[] = array(
            'taxonomy' => 'property_developers',
            'field' => 'slug',
            'terms' => $projects,
        );
        
        return $query_arg;
    }
    
    /**
     * Filter properties to only show off-plan status
     * 
     * @param array $args Query arguments
     * @return array Modified query arguments
     */
    public function filter_offplan_status($args) {
        global $post;
        
        // Ensure args is an array
        if (!is_array($args)) {
            return array();
        }
        
        // Check if this is an AJAX request for off-plan search
        $is_offplan_ajax = isset($_GET['action']) && $_GET['action'] === 'houzez_offplan_listings';
        
        // Check if shortcode is used on current page
        $is_offplan_page = false;
        if ($post) {
            $is_offplan_page = has_shortcode($post->post_content, 'houzez_offplan_search');
        }
        
        // Only apply if shortcode is used or AJAX request
        if (!$is_offplan_page && !$is_offplan_ajax) {
            return $args;
        }
        
        // Add tax query for off-plan status
        if (!isset($args['tax_query'])) {
            $args['tax_query'] = array();
        }
        
        // Check if tax_query is already an array of arrays
        $has_offplan = false;
        foreach ($args['tax_query'] as $tax_query) {
            if (isset($tax_query['taxonomy']) && $tax_query['taxonomy'] === 'property_status') {
                if (isset($tax_query['terms']) && 
                    (is_array($tax_query['terms']) && in_array('off-plan', $tax_query['terms'])) ||
                    $tax_query['terms'] === 'off-plan') {
                    $has_offplan = true;
                    break;
                }
            }
        }
        
        if (!$has_offplan) {
            $args['tax_query'][] = array(
                'taxonomy' => 'property_status',
                'field' => 'slug',
                'terms' => 'off-plan',
            );
            
            // Set relation if multiple tax queries
            if (count($args['tax_query']) > 1) {
                $args['tax_query']['relation'] = 'AND';
            }
        }
        
        return $args;
    }
    
    /**
     * Check if a taxonomy term has any off-plan properties
     * 
     * @param int|WP_Term $term Term ID or term object
     * @param string $taxonomy Taxonomy name
     * @return bool True if term has off-plan properties, false otherwise
     */
    public static function term_has_offplan_properties($term, $taxonomy) {
        // Get term ID if term object is passed
        $term_id = is_object($term) ? $term->term_id : $term;
        
        if (empty($term_id)) {
            return false;
        }
        
        // Query to check if term has any off-plan properties
        $args = array(
            'post_type' => 'property',
            'post_status' => 'publish',
            'posts_per_page' => 1,
            'fields' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $term_id,
                ),
                array(
                    'taxonomy' => 'property_status',
                    'field' => 'slug',
                    'terms' => 'off-plan',
                ),
            ),
        );
        
        $query = new WP_Query($args);
        $has_properties = $query->found_posts > 0;
        
        wp_reset_postdata();
        
        return $has_properties;
    }
    
    /**
     * Get count of off-plan properties for a taxonomy term
     * 
     * @param int|WP_Term $term Term ID or term object
     * @param string $taxonomy Taxonomy name
     * @return int Number of off-plan properties
     */
    public static function get_term_offplan_count($term, $taxonomy) {
        // Get term ID if term object is passed
        $term_id = is_object($term) ? $term->term_id : $term;
        
        if (empty($term_id)) {
            return 0;
        }
        
        // Query to get count of off-plan properties
        $args = array(
            'post_type' => 'property',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $term_id,
                ),
                array(
                    'taxonomy' => 'property_status',
                    'field' => 'slug',
                    'terms' => 'off-plan',
                ),
            ),
        );
        
        $query = new WP_Query($args);
        $count = $query->found_posts;
        
        wp_reset_postdata();
        
        return $count;
    }
}

