<?php
/**
 * Shortcode class
 *
 * @package Houzez_Offplan_Search
 */

if (!defined('ABSPATH')) {
    exit;
}

class Houzez_Offplan_Shortcode {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Register shortcode directly
        add_shortcode('houzez_offplan_search', array($this, 'render_shortcode'));
    }
    
    /**
     * Render shortcode
     * 
     * @param array $atts Shortcode attributes
     * @param string $content Shortcode content
     * @return string Shortcode output
     */
    public function render_shortcode($atts = array(), $content = null) {
        // Check if Houzez theme functions exist
        if (!function_exists('houzez_option')) {
            return '<div class="houzez-offplan-error"><p>' . esc_html__('Houzez theme is required for this shortcode to work.', 'houzez-offplan-search') . '</p></div>';
        }
        
        // Ensure we have an array
        if (!is_array($atts)) {
            $atts = array();
        }
        
        // Enqueue scripts when shortcode is rendered
        $this->enqueue_assets();
        
        // Parse shortcode attributes
        $atts = shortcode_atts(array(
            'card_style' => 'v7',
            'posts_per_page' => '',
            'show_filters' => 'yes',
            'show_sort' => 'yes',
            'columns' => '2',
        ), $atts, 'houzez_offplan_search');
        
        // Validate card style
        $valid_card_styles = array('v1', 'v2', 'v3', 'v4', 'v5', 'v6', 'v7');
        $card_style = in_array($atts['card_style'], $valid_card_styles) ? $atts['card_style'] : 'v7';
        
        // Validate and set columns
        $columns = absint($atts['columns']);
        $valid_columns = array(2, 3, 4);
        if (!in_array($columns, $valid_columns, true)) {
            $columns = 2; // Default to 2 columns
        }
        
        // Get posts per page
        $posts_per_page = !empty($atts['posts_per_page']) ? absint($atts['posts_per_page']) : 
                         (function_exists('houzez_option') ? absint(houzez_option('search_num_posts')) : 9);
        if (!$posts_per_page) {
            $posts_per_page = 9;
        }
        
        // Get current page
        $paged = max(1, get_query_var('paged') ?: get_query_var('page'));
        
        // Set up query arguments for off-plan properties
        $listing_args = array(
            'post_type' => 'property',
            'post_status' => 'publish',
            'posts_per_page' => $posts_per_page,
            'paged' => $paged,
        );
        
        // Add off-plan status filter directly (don't use houzez20_property_filter as it expects page template context)
        $listing_args['tax_query'] = array(
            array(
                'taxonomy' => 'property_status',
                'field' => 'slug',
                'terms' => 'off-plan',
            )
        );
        
        // Apply sorting
        if (function_exists('houzez_prop_sort')) {
            $listing_args = houzez_prop_sort($listing_args);
        }
        
        // Create query
        $listings_query = new WP_Query($listing_args);
        $total_listing_found = $listings_query->found_posts;
        
        // Container class
        $container_class = 'container';
        
        // Layout CSS class for grid based on columns
        $listing_view_class = 'listing-view grid-view row';
        if ($columns == 4) {
            $listing_view_class .= ' row-cols-1 row-cols-xl-4 row-cols-lg-3 row-cols-md-2 gy-4 gx-4';
        } elseif ($columns == 3) {
            $listing_view_class .= ' row-cols-lg-3 row-cols-md-2 row-cols-sm-1 gy-4 gx-4';
        } else {
            // Default 2 columns
            $listing_view_class .= ' row-cols-1 row-cols-md-2 gy-4 gx-4';
        }
        
        $max_num_pages = $listings_query->max_num_pages;
        
        $class_margin = '';
        if ($max_num_pages <= 1) {
            $class_margin = 'mb-4';
        }
        
        // Start output buffering
        ob_start();
        
        // Error handling
        try {
        ?>
        
        <div class="houzez-offplan-shortcode-wrapper">
            <section class="listing-wrap listing-offplan-search <?php echo esc_attr($class_margin); ?>" role="region">
                <?php do_action('houzez_before_listing_wrap'); ?>
                <div class="<?php echo esc_attr($container_class); ?>">
                    <?php if ($atts['show_filters'] === 'yes') : ?>
                        <!-- Filter Bar -->
                        <?php 
                        $filter_template = HOUZEZ_OFFPLAN_PLUGIN_DIR . 'templates/offplan-filters.php';
                        if (file_exists($filter_template)) {
                            $args = array('card_style' => $card_style);
                            extract($args);
                            include $filter_template;
                        }
                        ?>
                    <?php endif; ?>

                    <!-- Title and Sort Row -->
                    <div class="offplan-title-sort-row d-flex justify-content-between align-items-center mb-4">
                        <h1 class="page-title mb-0"><?php esc_html_e('Off-Plan', 'houzez-offplan-search'); ?></h1>
                        
                        <?php if ($atts['show_sort'] === 'yes') : ?>
                            <!-- Sort Options -->
                            <?php 
                            $sortby = isset($_GET['sortby']) ? sanitize_text_field($_GET['sortby']) : '';
                            ?>
                            <div class="sort-by">
                                <div class="d-flex align-items-center">
                                    <div class="sort-by-title" id="offplan-sort-label">
                                        <?php esc_html_e('Sort:', 'houzez-offplan-search'); ?>
                                    </div>
                                    <select id="offplan_sort_properties" class="selectpicker form-control bs-select-hidden offplan-sort" 
                                            title="<?php esc_html_e('Default Order', 'houzez-offplan-search'); ?>" 
                                            data-live-search="false" 
                                            data-dropdown-align-right="auto" 
                                            aria-labelledby="offplan-sort-label" 
                                            role="listbox">
                                        <option role="option" value=""><?php esc_html_e('Default Order', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'a_price'); ?> value="a_price"><?php esc_html_e('Price - Low to High', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'd_price'); ?> value="d_price"><?php esc_html_e('Price - High to Low', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'featured_first'); ?> value="featured_first"><?php esc_html_e('Featured Listings First', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'a_date'); ?> value="a_date"><?php esc_html_e('Date - Old to New', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'd_date'); ?> value="d_date"><?php esc_html_e('Date - New to Old', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'a_title'); ?> value="a_title"><?php esc_html_e('Title - ASC', 'houzez-offplan-search'); ?></option>
                                        <option role="option" <?php selected($sortby, 'd_title'); ?> value="d_title"><?php esc_html_e('Title - DESC', 'houzez-offplan-search'); ?></option>
                                    </select>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div><!-- offplan-title-sort-row -->

                    <div class="row">
                        <div class="col-lg-12 col-md-12">
                            <?php do_action('houzez_before_listing_content'); ?>

                            <!-- Results Container (AJAX Target) -->
                            <div id="offplan-results-container" class="<?php echo esc_attr($listing_view_class); ?>" 
                                 role="list" 
                                 data-view="grid" 
                                 data-card-style="<?php echo esc_attr($card_style); ?>"
                                 data-page-id="<?php echo esc_attr(get_the_ID()); ?>"
                                 data-posts-per-page="<?php echo esc_attr($posts_per_page); ?>"
                                 data-columns="<?php echo esc_attr($columns); ?>"
                                 data-layout-css="<?php echo esc_attr($listing_view_class); ?>">
                                <?php do_action('houzez_before_listing_items'); ?>
                                <?php
                                if ($listings_query->have_posts()) :
                                    while ($listings_query->have_posts()) : $listings_query->the_post();
                                        // Load Houzez theme template part
                                        $template_path = locate_template(array('template-parts/listing/item-' . $card_style . '.php'));
                                        if ($template_path) {
                                            get_template_part('template-parts/listing/item', $card_style);
                                        } else {
                                            // Fallback to v1 if card style template doesn't exist
                                            get_template_part('template-parts/listing/item', 'v1');
                                        }
                                    endwhile;
                                    wp_reset_postdata();
                                else:
                                    get_template_part('template-parts/listing/item-none');
                                endif;
                                ?>
                            </div><!-- listing-view -->
                            
                            <!-- Pagination -->
                            <div class="offplan-pagination-wrapper">
                                <?php 
                                if (function_exists('houzez_pagination')) {
                                    houzez_pagination($listings_query->max_num_pages, $total_listing_found, $posts_per_page);
                                } else {
                                    // Basic pagination fallback
                                    $pagination = paginate_links(array(
                                        'base' => str_replace(999999999, '%#%', esc_url(get_pagenum_link(999999999))),
                                        'total' => $listings_query->max_num_pages,
                                        'current' => max(1, $paged),
                                        'format' => '?paged=%#%',
                                        'show_all' => false,
                                        'type' => 'list',
                                        'end_size' => 2,
                                        'mid_size' => 1,
                                        'prev_next' => true,
                                        'prev_text' => '&laquo;',
                                        'next_text' => '&raquo;',
                                    ));
                                    echo $pagination;
                                }
                                ?>
                            </div><!-- offplan-pagination-wrapper -->
                            <?php do_action('houzez_after_listing_items'); ?>
                        </div><!-- col-lg-12 -->
                    </div><!-- row -->
                </div><!-- container -->
                <?php do_action('houzez_after_listing_wrap'); ?>
            </section><!-- listing-wrap -->
        </div><!-- houzez-offplan-shortcode-wrapper -->
        
        <?php
        } catch (Exception $e) {
            // Return error message if something goes wrong
            return '<div class="houzez-offplan-error"><p>' . esc_html__('Error rendering off-plan search:', 'houzez-offplan-search') . ' ' . esc_html($e->getMessage()) . '</p></div>';
        }
        
        return ob_get_clean();
    }
    
    /**
     * Enqueue assets
     */
    private function enqueue_assets() {
        static $enqueued = false;
        
        // Only enqueue once per page
        if ($enqueued) {
            return;
        }
        
        $enqueued = true;
        
        // Depend on theme's custom script so grid gallery (houzez_grid_image_gallery / houzez.Sliders.gridImageGallery) is available
        $script_deps = array('jquery');
        if (wp_script_is('houzez-custom', 'registered')) {
            $script_deps[] = 'houzez-custom';
        }
        wp_enqueue_script(
            'houzez-offplan-search',
            HOUZEZ_OFFPLAN_PLUGIN_URL . 'assets/js/offplan-search.js',
            $script_deps,
            HOUZEZ_OFFPLAN_VERSION,
            true
        );
        
        // Localize script
        global $post;
        wp_localize_script(
            'houzez-offplan-search',
            'houzezOffplan',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('houzez_offplan_nonce'),
                'pageId' => $post ? $post->ID : 0,
            )
        );
        
        // Enqueue CSS if file exists (use plugin version so bumping HOUZEZ_OFFPLAN_VERSION on deploy forces new ?ver= and bypasses cache)
        $css_file = HOUZEZ_OFFPLAN_PLUGIN_DIR . 'assets/css/offplan-search.css';
        if (file_exists($css_file)) {
            wp_enqueue_style(
                'houzez-offplan-search',
                HOUZEZ_OFFPLAN_PLUGIN_URL . 'assets/css/offplan-search.css',
                array(),
                HOUZEZ_OFFPLAN_VERSION
            );
        }
    }
}

