<?php
/**
 * Status-based search forms template
 *
 * One form per property_status term. Each form: Community, Property Type,
 * Min. Price, Max. Price, Bedrooms. Submits via GET to Houzez search.
 *
 * @package Houzez_Offplan_Search
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!isset($status_filter)) {
    $status_filter = '';
}

/**
 * Get areas filtered by status with property counts
 * Includes child terms in the count for parent areas
 *
 * @param string $status_slug Property status slug
 * @return array Array of area terms with counts
 */
if (!function_exists('houzez_status_forms_get_areas_by_status')) {
function houzez_status_forms_get_areas_by_status($status_slug) {
    $areas = get_terms(array(
        'taxonomy' => 'property_area',
        'hide_empty' => false,
        'orderby' => 'name',
        'order' => 'ASC',
    ));

    if (empty($areas) || is_wp_error($areas)) {
        return array();
    }

    $filtered_areas = array();
    foreach ($areas as $area) {
        $args = array(
            'post_type' => 'property',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => 'property_area',
                    'field' => 'term_id',
                    'terms' => $area->term_id,
                    'include_children' => true,
                ),
                array(
                    'taxonomy' => 'property_status',
                    'field' => 'slug',
                    'terms' => $status_slug,
                ),
            ),
        );

        $query = new WP_Query($args);
        $count = $query->found_posts;
        wp_reset_postdata();

        if ($count > 0) {
            $area->property_count = $count;
            $filtered_areas[] = $area;
        }
    }

    return $filtered_areas;
}
}

/**
 * Get property types filtered by status
 *
 * @param string $status_slug Property status slug
 * @return array Array of type terms
 */
if (!function_exists('houzez_status_forms_get_types_by_status')) {
function houzez_status_forms_get_types_by_status($status_slug) {
    $types = get_terms(array(
        'taxonomy' => 'property_type',
        'hide_empty' => false,
        'orderby' => 'name',
        'order' => 'ASC',
    ));

    if (empty($types) || is_wp_error($types)) {
        return array();
    }

    $filtered_types = array();
    foreach ($types as $type) {
        $args = array(
            'post_type' => 'property',
            'post_status' => 'publish',
            'posts_per_page' => 1,
            'fields' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => 'property_type',
                    'field' => 'term_id',
                    'terms' => $type->term_id,
                ),
                array(
                    'taxonomy' => 'property_status',
                    'field' => 'slug',
                    'terms' => $status_slug,
                ),
            ),
        );

        $query = new WP_Query($args);
        $has_properties = $query->found_posts > 0;
        wp_reset_postdata();

        if ($has_properties) {
            $filtered_types[] = $type;
        }
    }

    return $filtered_types;
}
}

$search_url = function_exists('houzez_get_search_template_link') ? houzez_get_search_template_link() : home_url('/');
$rent_slug  = '';
if (function_exists('houzez_option') && function_exists('houzez_get_term_slug')) {
    $rent_slug = houzez_get_term_slug(houzez_option('search_rent_status'), 'property_status');
}

$srh_any    = function_exists('houzez_option') ? houzez_option('srh_any', 'Any') : __('Any', 'houzez-offplan-search');
$srh_areas  = __('Community', 'houzez-offplan-search');
$srh_type   = __('Property Type', 'houzez-offplan-search');
$srh_min    = function_exists('houzez_option') ? houzez_option('srh_min_price', 'Min. Price') : __('Min. Price', 'houzez-offplan-search');
$srh_max    = function_exists('houzez_option') ? houzez_option('srh_max_price', 'Max. Price') : __('Max. Price', 'houzez-offplan-search');
$srh_beds   = function_exists('houzez_option') ? houzez_option('srh_bedrooms', 'Bedrooms') : __('Bedrooms', 'houzez-offplan-search');
$srh_search = function_exists('houzez_option') ? houzez_option('srh_search', 'Search') : __('Search', 'houzez-offplan-search');

$statuses = get_terms(array(
    'taxonomy'   => 'property_status',
    'hide_empty' => true,
    'orderby'    => 'name',
    'order'      => 'ASC',
));

if (empty($statuses) || is_wp_error($statuses)) {
    echo '<p class="houzez-status-forms-search__empty">' . esc_html__('No property statuses found.', 'houzez-offplan-search') . '</p>';
    return;
}

if (!empty($status_filter)) {
    $statuses = array_filter($statuses, function ($term) use ($status_filter) {
        return $term->slug === $status_filter;
    });
    $statuses = array_values($statuses);
    if (empty($statuses)) {
        echo '<p class="houzez-status-forms-search__empty">' . esc_html(sprintf(__('No property status found for slug "%s".', 'houzez-offplan-search'), $status_filter)) . '</p>';
        return;
    }
}
?>

<div class="houzez-status-forms-search">
    <?php foreach ($statuses as $term) : ?>
        <?php
        $is_rent = !empty($rent_slug) && $term->slug === $rent_slug;
        $form_id = 'houzez-status-form-' . sanitize_html_class($term->slug);
        ?>
        <div class="houzez-status-forms-search__form-wrap">
            <form class="houzez-status-forms-search__form" id="<?php echo esc_attr($form_id); ?>" method="get" action="<?php echo esc_url($search_url); ?>">
                <input type="hidden" name="status[]" value="<?php echo esc_attr($term->slug); ?>" />
                <div class="houzez-status-forms-search__row row g-3 align-items-end">
                    <div class="houzez-status-forms-search__field houzez-status-forms-search__field--community col-md-4 col-lg-3">
                        <label for="<?php echo esc_attr($form_id); ?>-areas" class="houzez-status-forms-search__label"><?php echo esc_html($srh_areas); ?></label>
                        <select name="areas[]" id="<?php echo esc_attr($form_id); ?>-areas" class="houzez-status-forms-search__select selectpicker form-control bs-select-hidden" data-size="5" title="<?php echo esc_attr($srh_any); ?>" data-live-search="true" data-live-search-normalize="true">
                            <option value=""><?php echo esc_html($srh_any); ?></option>
                            <?php
                            $filtered_areas = houzez_status_forms_get_areas_by_status($term->slug);
                            if (!empty($filtered_areas)) {
                                foreach ($filtered_areas as $area) {
                                    $count = isset($area->property_count) ? $area->property_count : 0;
                                    echo '<option value="' . esc_attr($area->slug) . '">' . esc_html($area->name) . ' (' . esc_html($count) . ')</option>';
                                }
                            }
                            ?>
                        </select>
                    </div>
                    <div class="houzez-status-forms-search__field col-md-4 col-lg-2">
                        <label for="<?php echo esc_attr($form_id); ?>-type" class="houzez-status-forms-search__label"><?php echo esc_html($srh_type); ?></label>
                        <select name="type[]" id="<?php echo esc_attr($form_id); ?>-type" class="houzez-status-forms-search__select selectpicker form-control bs-select-hidden" data-size="5" title="<?php echo esc_attr($srh_any); ?>" data-live-search="true" data-live-search-normalize="true">
                            <option value=""><?php echo esc_html($srh_any); ?></option>
                            <?php
                            $filtered_types = houzez_status_forms_get_types_by_status($term->slug);
                            if (!empty($filtered_types)) {
                                foreach ($filtered_types as $type) {
                                    echo '<option value="' . esc_attr($type->slug) . '">' . esc_html($type->name) . '</option>';
                                }
                            }
                            ?>
                        </select>
                    </div>
                    <div class="houzez-status-forms-search__field col-md-4 col-lg-2">
                        <label for="<?php echo esc_attr($form_id); ?>-min-price" class="houzez-status-forms-search__label"><?php echo esc_html($srh_min); ?></label>
                        <select name="min-price" id="<?php echo esc_attr($form_id); ?>-min-price" class="houzez-status-forms-search__select selectpicker form-control bs-select-hidden" data-size="5" title="<?php echo esc_attr($srh_min); ?>" data-live-search="false">
                            <?php
                            if ($is_rent && function_exists('houzez_adv_searches_min_price_rent_only')) {
                                houzez_adv_searches_min_price_rent_only();
                            } elseif (function_exists('houzez_adv_searches_min_price')) {
                                houzez_adv_searches_min_price();
                            }
                            ?>
                        </select>
                    </div>
                    <div class="houzez-status-forms-search__field col-md-4 col-lg-2">
                        <label for="<?php echo esc_attr($form_id); ?>-max-price" class="houzez-status-forms-search__label"><?php echo esc_html($srh_max); ?></label>
                        <select name="max-price" id="<?php echo esc_attr($form_id); ?>-max-price" class="houzez-status-forms-search__select selectpicker form-control bs-select-hidden" data-size="5" title="<?php echo esc_attr($srh_max); ?>" data-live-search="false">
                            <?php
                            if ($is_rent && function_exists('houzez_adv_searches_max_price_rent_only')) {
                                houzez_adv_searches_max_price_rent_only();
                            } elseif (function_exists('houzez_adv_searches_max_price')) {
                                houzez_adv_searches_max_price();
                            }
                            ?>
                        </select>
                    </div>
                    <div class="houzez-status-forms-search__field houzez-status-forms-search__field--bedrooms col-md-4 col-lg-1">
                        <label for="<?php echo esc_attr($form_id); ?>-bedrooms" class="houzez-status-forms-search__label"><?php echo esc_html($srh_beds); ?></label>
                        <select name="bedrooms" id="<?php echo esc_attr($form_id); ?>-bedrooms" class="houzez-status-forms-search__select selectpicker form-control bs-select-hidden" data-size="5" title="<?php echo esc_attr($srh_any); ?>" data-live-search="false">
                            <option value=""><?php echo esc_html($srh_any); ?></option>
                            <?php
                            if (function_exists('houzez_number_list')) {
                                houzez_number_list('bedrooms');
                            }
                            ?>
                        </select>
                    </div>
                    <div class="houzez-status-forms-search__submit-wrap col-md-12 col-lg-2">
                        <button type="submit" class="houzez-status-forms-search__btn btn"><?php echo esc_html($srh_search); ?></button>
                    </div>
                </div>
            </form>
        </div>
    <?php endforeach; ?>
</div>

<script type="text/javascript">
jQuery(document).ready(function($) {
    if (typeof $.fn.selectpicker !== 'undefined') {
        $('.houzez-status-forms-search .selectpicker').selectpicker('refresh');
    }
});
</script>
