/* global houzezSAS, wp */
(function ($) {
    'use strict';

    const debounce = (fn, delay) => {
        let t;
        return function (...args) {
            clearTimeout(t);
            t = setTimeout(() => fn.apply(this, args), delay);
        };
    };

    const getQueryVars = () => {
        const params = new URLSearchParams(window.location.search);
        return {
            status: params.get('status') || '',
            area: params.get('area') || '',
        };
    };

    const updateURL = (status, area) => {
        const params = new URLSearchParams(window.location.search);
        if (status) {
            params.set('status', status);
        } else {
            params.delete('status');
        }
        if (area) {
            params.set('area', area);
        } else {
            params.delete('area');
        }
        const newUrl = `${window.location.pathname}?${params.toString()}`;
        window.history.replaceState({}, '', newUrl);
    };

    const parsePageFromLink = (href) => {
        try {
            const url = new URL(href, window.location.origin);
            const paged = url.searchParams.get('paged');
            return paged ? parseInt(paged, 10) : 1;
        } catch (e) {
            return 1;
        }
    };

    const renderCount = (container, total) => {
        if (!container) return;
        const label = total === 1 ? houzezSAS.strings.result : houzezSAS.strings.results;
        container.text(`${total} ${label}`);
    };

    const redirectToSearch = (status, area) => {
        if (!houzezSAS.searchUrl) {
            console.error('Search URL not defined');
            return;
        }
        
        const params = new URLSearchParams();
        if (status && status !== '') {
            params.set('status[]', status);
        }
        if (area && area !== '') {
            params.set('areas[]', area);
        }
        
        const searchUrl = houzezSAS.searchUrl + (params.toString() ? '?' + params.toString() : '');
        window.location.href = searchUrl;
    };

    const initWidget = (root) => {
        const $root = $(root);
        const $tabs = $root.find('.houzez-sas__tab');
        const $areaInput = $root.find('.houzez-sas__area-input');
        const $suggestions = $root.find('.houzez-sas__suggestions');
        const $tagsContainer = $root.find('.houzez-sas__tags');
        const $searchBtn = $root.find('.houzez-sas__search-btn');

        let selectedStatus = $root.data('selected-status') || getQueryVars().status || '';
        let allAreas = []; // Store all areas for client-side filtering
        let selectedArea = null; // Store selected area as object {slug, name}

        const setActiveTab = () => {
            $tabs.removeClass('is-active');
            $tabs.each(function () {
                if ($(this).data('status') === selectedStatus) {
                    $(this).addClass('is-active');
                }
            });
        };

        const hideSuggestions = () => {
            $suggestions.attr('hidden', true);
        };

        const renderSuggestions = (areas) => {
            if (!areas || areas.length === 0) {
                $suggestions.html('').attr('hidden', true);
                return;
            }
            const list = areas
                .map(
                    (item) =>
                        `<div class="houzez-sas__suggestion" data-slug="${item.slug}" data-name="${item.name}">${item.name} <span class="houzez-sas__badge">${item.count}</span></div>`
                )
                .join('');
            $suggestions.html(list).attr('hidden', false);
        };

        const filterAreas = (searchTerm) => {
            if (!searchTerm || searchTerm.length < 1) {
                return allAreas;
            }
            const term = searchTerm.toLowerCase();
            return allAreas.filter(area => {
                const name = area.name.toLowerCase();
                const slug = area.slug.toLowerCase();
                return name.indexOf(term) !== -1 || slug.indexOf(term) !== -1;
            });
        };

        const renderAreaTag = () => {
            $tagsContainer.empty();
            const $inputWrap = $areaInput.closest('.houzez-sas__input-wrap');
            if (selectedArea) {
                const tagHtml = `
                    <div class="houzez-sas__tag">
                        <span>${selectedArea.name}</span>
                        <button type="button" class="houzez-sas__tag-remove" aria-label="Remove area">×</button>
                    </div>
                `;
                $tagsContainer.html(tagHtml);
                $inputWrap.addClass('has-tag');
            } else {
                $inputWrap.removeClass('has-tag');
            }
        };

        const removeAreaTag = () => {
            selectedArea = null;
            $areaInput.val('');
            renderAreaTag();
        };

        const fetchSuggestions = (term, showAll = false) => {
            // If no term and showAll is false, don't fetch
            if (!term && !showAll) {
                hideSuggestions();
                return;
            }
            
            // If term is less than minChars and not showAll, don't fetch
            if (term && term.length < houzezSAS.minChars && !showAll) {
                hideSuggestions();
                return;
            }
            
            $.ajax({
                url: houzezSAS.ajaxurl,
                type: 'GET',
                data: {
                    action: 'houzez_area_suggestions',
                    nonce: houzezSAS.nonce,
                    q: term || '', // Send empty string if showAll
                    show_all: showAll ? 1 : 0, // Flag to show all areas
                    status: selectedStatus || '', // Pass selected status
                },
                success: function(response) {
                    if (!response || !response.success || !response.data || !response.data.length) {
                        hideSuggestions();
                        return;
                    }
                    const items = response.data;
                    // Update cache
                    allAreas = items;
                    renderSuggestions(items);
                },
                error: function(xhr, status, error) {
                    console.error('Suggestions AJAX Error:', status, error);
                    hideSuggestions();
                }
            });
        };

        const debouncedFetchSuggestions = debounce((term) => {
            fetchSuggestions(term, false);
        }, 300);

        const loadAllAreas = (showImmediately = false) => {
            // Always reload when status changes to get accurate counts
            // Clear cache to force reload with current status
            allAreas = [];
            
            $.ajax({
                url: houzezSAS.ajaxurl,
                type: 'GET',
                data: {
                    action: 'houzez_area_suggestions',
                    nonce: houzezSAS.nonce,
                    q: '',
                    show_all: 1,
                    status: selectedStatus || '',
                },
                success: function(response) {
                    if (response && response.success && response.data && response.data.length) {
                        allAreas = response.data;
                        if (showImmediately) {
                            renderSuggestions(allAreas);
                        }
                    } else {
                        if (showImmediately) {
                            hideSuggestions();
                        }
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Suggestions AJAX Error:', status, error);
                    if (showImmediately) {
                        hideSuggestions();
                    }
                }
            });
        };

        // Tab click - just update status, don't redirect
        $tabs.on('click', function (e) {
            e.preventDefault();
            selectedStatus = $(this).data('status') || '';
            setActiveTab();
            // Reload areas with new status to update counts
            if ($areaInput.is(':focus') || $suggestions.is(':visible') || !$suggestions.attr('hidden')) {
                const term = $areaInput.val();
                if (term && term.length >= houzezSAS.minChars) {
                    fetchSuggestions(term, false);
                } else {
                    loadAllAreas(true);
                }
            }
        });

        // Area input focus - show all areas immediately
        $areaInput.on('focus', function () {
            const term = $(this).val();
            if (term && term.length > 0) {
                // Fetch from server with current status to get accurate counts
                fetchSuggestions(term, false);
            } else {
                // Show all areas immediately (from cache or load)
                if (allAreas.length > 0) {
                    // Check if cached areas match current status, if not reload
                    loadAllAreas(true);
                } else {
                    // Still loading, load and show
                    loadAllAreas(true);
                }
            }
        });

        // Show suggestions when clicking on input wrap (to catch clicks on tags area)
        $root.find('.houzez-sas__input-wrap').on('click', function(e) {
            // Don't trigger if clicking on tag remove button
            if ($(e.target).closest('.houzez-sas__tag-remove').length) {
                return;
            }
            // Focus the input to show suggestions
            $areaInput.focus();
        });

        // Area input typing - fetch from server to get accurate counts based on status
        $areaInput.on('input', function () {
            const term = $(this).val();
            if (term.length === 0) {
                // If cleared, reload all areas with current status
                loadAllAreas(true);
            } else if (term.length >= houzezSAS.minChars) {
                // Fetch from server with current status for accurate counts
                debouncedFetchSuggestions(term);
            } else {
                hideSuggestions();
            }
        });

        // Enter key on input - trigger search
        $areaInput.on('keydown', function (e) {
            if (e.key === 'Enter' || e.keyCode === 13) {
                e.preventDefault();
                $searchBtn.trigger('click');
            }
        });

        // Suggestion click - add area tag, don't redirect
        $suggestions.on('click', '.houzez-sas__suggestion', function (e) {
            e.preventDefault();
            const areaSlug = $(this).data('slug');
            const areaName = $(this).data('name');
            selectedArea = { slug: areaSlug, name: areaName };
            $areaInput.val('');
            hideSuggestions();
            renderAreaTag();
        });

        // Remove area tag
        $tagsContainer.on('click', '.houzez-sas__tag-remove', function (e) {
            e.preventDefault();
            e.stopPropagation();
            removeAreaTag();
        });

        // Search button click - redirect to search page
        $searchBtn.on('click', function (e) {
            e.preventDefault();
            const areaSlug = selectedArea ? selectedArea.slug : '';
            redirectToSearch(selectedStatus, areaSlug);
        });

        // Click outside to hide suggestions
        $(document).on('click', function(e) {
            // Don't hide if clicking inside the widget
            if ($root.is(e.target) || $root.has(e.target).length > 0) {
                return;
            }
            // Hide suggestions when clicking outside
            hideSuggestions();
        });

        // Initialize active tab state
        // If no status selected and tabs exist, select first tab by default
        if (!selectedStatus && $tabs.length > 0) {
            selectedStatus = $tabs.first().data('status') || '';
            setActiveTab();
        } else {
            setActiveTab();
        }
        renderAreaTag();
        
        // Preload all areas in background (don't show until input is clicked)
        loadAllAreas(false);
    };

    $(document).ready(function () {
        // Wait for both jQuery and houzezSAS to be available
        if (typeof jQuery === 'undefined') {
            console.error('jQuery is required for Houzez Banner Search');
            return;
        }
        
        if (typeof houzezSAS === 'undefined') {
            console.error('houzezSAS object not defined. Make sure the plugin scripts are loaded.');
            return;
        }
        
        $('.houzez-sas').each(function () {
            initWidget(this);
        });
    });
})(jQuery);
