<?php
/**
 * AJAX handlers for Status & Area search.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Houzez_Status_Area_Ajax {

    public function __construct() {
        // Register AJAX actions for both logged in and logged out users
        add_action('wp_ajax_houzez_area_suggestions', [$this, 'area_suggestions']);
        add_action('wp_ajax_nopriv_houzez_area_suggestions', [$this, 'area_suggestions']);

        add_action('wp_ajax_houzez_status_area_filter', [$this, 'status_area_filter']);
        add_action('wp_ajax_nopriv_houzez_status_area_filter', [$this, 'status_area_filter']);
        
        // Debug: Log that actions are registered (remove in production)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Houzez Banner Search: AJAX actions registered');
        }
    }

    /**
     * Return area suggestions (only from property_area taxonomy, hide empty terms).
     */
    public function area_suggestions() {
        // Verify nonce but don't die on failure - return error instead
        $nonce = isset($_GET['nonce']) ? sanitize_text_field(wp_unslash($_GET['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'houzez_sas_nonce')) {
            wp_send_json_error(['message' => 'Invalid security token']);
            wp_die();
            return;
        }

        $search = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $show_all = isset($_GET['show_all']) && (int)$_GET['show_all'] === 1;
        $status_slug = isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '';
        
        // If show_all is true, show popular areas even without search term
        if ($show_all && empty($search)) {
            $args = [
                'taxonomy'   => 'property_area', // Only property_area taxonomy
                'hide_empty' => true, // Only show areas that have at least one property
                // No limit - show all areas
                'orderby'    => 'count',
                'order'      => 'DESC', // Show areas with most properties first
            ];
        } elseif (strlen($search) < 2) {
            // If search term is too short and not show_all, return empty
            wp_send_json_success([]);
            wp_die();
            return;
        } else {
            // Only query property_area taxonomy with search
            $args = [
                'taxonomy'   => 'property_area', // Only property_area taxonomy
                'hide_empty' => true, // Only show areas that have at least one property
                // No limit - show all matching areas
                'orderby'    => 'count',
                'order'      => 'DESC', // Show areas with most properties first
                'search'     => $search, // WordPress 4.5+ search parameter (searches name and slug)
            ];

            // Fallback for older WordPress versions
            if (!isset($args['search'])) {
                $args['name__like'] = $search;
            }
        }

        $terms = get_terms($args);
        
        if (is_wp_error($terms)) {
            wp_send_json_success([]);
            wp_die();
            return;
        }

        // Helper function to count properties in area filtered by status
        $count_properties_in_area = function($area_term_id, $status_slug) {
            $query_args = [
                'post_type'      => 'property',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'fields'         => 'ids',
                'tax_query'      => [
                    [
                        'taxonomy' => 'property_area',
                        'field'    => 'term_id',
                        'terms'    => $area_term_id,
                    ],
                ],
            ];
            
            // Add status filter if provided
            if (!empty($status_slug)) {
                $query_args['tax_query'][] = [
                    'taxonomy' => 'property_status',
                    'field'    => 'slug',
                    'terms'    => $status_slug,
                ];
                $query_args['tax_query']['relation'] = 'AND';
            }
            
            // Apply Houzez filters
            if (function_exists('houzez_prop_sort')) {
                $query_args = apply_filters('houzez_sold_status_filter', $query_args);
            }
            
            $query = new WP_Query($query_args);
            return (int) $query->found_posts;
        };
        
        // Ensure we only return property_area terms and filter out empty ones
        $results = [];
        foreach ($terms as $term) {
            // Double-check it's property_area taxonomy
            if ($term->taxonomy !== 'property_area') {
                continue;
            }
            
            // Calculate count based on selected status
            $area_count = $count_properties_in_area($term->term_id, $status_slug);
            
            // Only include areas with at least one property for the selected status
            if ($area_count < 1) {
                continue;
            }
            
            // If show_all is true, include all areas without search filtering
            if ($show_all && empty($search)) {
                $results[] = [
                    'id'    => (int) $term->term_id,
                    'name'  => $term->name,
                    'slug'  => $term->slug,
                    'count' => $area_count,
                ];
                continue;
            }
            
            // Additional search filter (case-insensitive) - only if searching
            if (!empty($search)) {
                $search_lower = mb_strtolower($search, 'UTF-8');
                $name_lower = mb_strtolower($term->name, 'UTF-8');
                $slug_lower = mb_strtolower($term->slug, 'UTF-8');
                
                // Check if search term matches name or slug
                if (strpos($name_lower, $search_lower) === false && strpos($slug_lower, $search_lower) === false) {
                    continue;
                }
            }
            
            $results[] = [
                'id'    => (int) $term->term_id,
                'name'  => $term->name,
                'slug'  => $term->slug,
                'count' => $area_count,
            ];
        }

        // Sort by count (descending) then by name
        usort($results, function($a, $b) {
            if ($a['count'] === $b['count']) {
                return strcmp($a['name'], $b['name']);
            }
            return $b['count'] - $a['count'];
        });

        // No limit - return all results
        wp_send_json_success($results);
        
        // Exit to prevent any output after JSON response
        wp_die();
    }

    /**
     * Build filtered results HTML (listings + pagination).
     */
    public function status_area_filter() {
        try {
            // Check if this is an AJAX request
            if (!defined('DOING_AJAX') || !DOING_AJAX) {
                wp_send_json_error(['message' => 'Not an AJAX request']);
                wp_die();
                return;
            }
            
            // Verify nonce but don't die on failure - return error instead
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (empty($nonce) || !wp_verify_nonce($nonce, 'houzez_sas_nonce')) {
                wp_send_json_error([
                    'message' => 'Invalid security token. Please refresh the page.',
                    'debug' => [
                        'nonce_received' => !empty($nonce),
                        'nonce_length' => strlen($nonce),
                    ]
                ]);
                wp_die();
                return;
            }

            $status_slug = isset($_POST['status']) ? sanitize_text_field(wp_unslash($_POST['status'])) : '';
            $area_slug   = isset($_POST['area']) ? sanitize_text_field(wp_unslash($_POST['area'])) : '';
            $paged       = isset($_POST['paged']) ? max(1, absint($_POST['paged'])) : 1;

        $search_num_posts = function_exists('houzez_option') ? houzez_option('search_num_posts') : 9;
        $per_page         = $search_num_posts ? $search_num_posts : 9;

        $search_qry = [
            'post_type'      => 'property',
            'posts_per_page' => $per_page,
            'paged'          => $paged,
            'post_status'    => 'publish',
        ];

        $tax_query = [];

        if ($status_slug && $status_slug !== 'all') {
            $tax_query[] = [
                'taxonomy' => 'property_status',
                'field'    => 'slug',
                'terms'    => $status_slug,
            ];
        }

        if ($area_slug && $area_slug !== 'all') {
            $tax_query[] = [
                'taxonomy' => 'property_area',
                'field'    => 'slug',
                'terms'    => $area_slug,
            ];
        }

        if (!empty($tax_query)) {
            $tax_query['relation']      = 'AND';
            $search_qry['tax_query']    = $tax_query;
        }

        // Apply Houzez filters
        if (function_exists('houzez_prop_sort')) {
            $search_qry = apply_filters('houzez_sold_status_filter', $search_qry);
            $search_qry = houzez_prop_sort($search_qry);
        }

        $query = new WP_Query($search_qry);

        // Determine layout similar to normal-page-search-results
        $default_view_option = function_exists('houzez_option') ? houzez_option('search_result_posts_layout', 'list-view-v1') : 'list-view-v1';
        $view_settings       = function_exists('houzez_get_listing_view_settings') ? houzez_get_listing_view_settings($default_view_option) : [];

        $current_item_template = isset($view_settings['current_item_template']) ? $view_settings['current_item_template'] : 'v1';
        $current_view          = isset($view_settings['current_view']) ? $view_settings['current_view'] : 'list-view';
        $item_version          = isset($view_settings['item_version']) ? $view_settings['item_version'] : 'v1';
        $layout                = function_exists('houzez_option') ? houzez_option('search_result_layout', 'right-sidebar') : 'right-sidebar';
        $grid_columns          = function_exists('houzez_option') ? houzez_option('search_grid_columns', '3') : '3';

        $listing_view_class = function_exists('houzez_get_listing_view_class')
            ? houzez_get_listing_view_class($current_view, $item_version, $layout, $grid_columns)
            : 'listing-view';

        ob_start();
        echo '<div class="' . esc_attr($listing_view_class) . '" role="list" data-view="' . esc_attr($current_view) . '">';

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $template_exists = locate_template(['template-parts/listing/item-' . $current_item_template . '.php']);
                if ($template_exists) {
                    get_template_part('template-parts/listing/item', $current_item_template);
                } else {
                    get_template_part('template-parts/listing/item', 'v1');
                }
            }
        } else {
            echo '<div class="search-no-results-found-wrap"><div class="search-no-results-found">';
            esc_html_e('No results found', 'houzez-status-area-search');
            echo '</div></div>';
        }

        echo '</div>';
        $html = ob_get_clean();

        wp_reset_postdata();

        // Pagination HTML
        ob_start();
        if (function_exists('houzez_ajax_pagination')) {
            houzez_ajax_pagination($query->max_num_pages);
        } else {
            echo paginate_links([
                'base'      => str_replace(999999999, '%#%', esc_url(get_pagenum_link(999999999))),
                'total'     => $query->max_num_pages,
                'current'   => max(1, $paged),
                'format'    => '?paged=%#%',
                'show_all'  => false,
                'type'      => 'list',
                'end_size'  => 2,
                'mid_size'  => 1,
                'prev_next' => true,
                'prev_text' => '&laquo;',
                'next_text' => '&raquo;',
            ]);
        }
        $pagination = ob_get_clean();

            // Always send a response
            wp_send_json_success([
                'html'       => $html,
                'pagination' => $pagination,
                'total'      => (int) $query->found_posts,
                'max_pages'  => (int) $query->max_num_pages,
                'per_page'   => (int) $per_page,
            ]);
            
        } catch (Exception $e) {
            // Catch any fatal errors and return proper JSON error
            wp_send_json_error([
                'message' => 'An error occurred while processing your request.',
                'error' => defined('WP_DEBUG') && WP_DEBUG ? $e->getMessage() : ''
            ]);
        } catch (Error $e) {
            // Catch PHP 7+ fatal errors
            wp_send_json_error([
                'message' => 'A fatal error occurred while processing your request.',
                'error' => defined('WP_DEBUG') && WP_DEBUG ? $e->getMessage() : ''
            ]);
        }
        
        // Exit to prevent any output after JSON response
        wp_die();
    }

    private function verify_nonce() {
        $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'houzez_sas_nonce')) {
            wp_send_json_error(['message' => 'Invalid security token']);
            exit;
        }
    }
}
