<?php
/**
 * Plugin Name: Media Bulk Download
 * Plugin URI: https://www.devmizan.com/media-bulk-download
 * Description: Professional WordPress plugin that adds bulk download functionality to the Media Library. Download selected media files or entire media library as ZIP archives with one click. Features smart button visibility, security checks, and automatic cleanup.
 * Version: 1.0.0
 * Author: Dev Mizan
 * Author URI: https://www.devmizan.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: media-bulk-download
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * 
 * @package MediaBulkDownload
 * @author Dev Mizan
 * @version 1.0.0
 * @link https://www.devmizan.com
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MBD_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MBD_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('MBD_VERSION', '1.0.0');

/**
 * Main Media Bulk Download class
 */
class MediaBulkDownload {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'init'));
    }
    
    /**
     * Initialize the plugin
     */
    public function init() {
        // Add bulk actions to media library (for list view)
        add_filter('bulk_actions-upload', array($this, 'add_bulk_actions'));
        add_filter('handle_bulk_actions-upload', array($this, 'handle_bulk_download'), 10, 3);
        
        // Add admin notices
        add_action('admin_notices', array($this, 'admin_notices'));
        
        // Add JavaScript to handle grid view bulk actions
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Add CSS to ensure button is visible
        add_action('admin_head', array($this, 'add_admin_css'));
        
        // Add AJAX handler for download action
        add_action('wp_ajax_mbd_download_selected', array($this, 'ajax_download_selected'));
        add_action('wp_ajax_mbd_download_all', array($this, 'ajax_download_all'));
        
        // Register font MIME types to ensure they're recognized
        add_filter('upload_mimes', array($this, 'register_font_mime_types'));
        
        // Debug: Log that hooks are registered (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Media Bulk Download: Hooks registered successfully'); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        }
    }
    
    /**
     * Add bulk actions to the media library
     */
    public function add_bulk_actions($bulk_actions) {
        // Add our custom bulk action
        $bulk_actions['download_selected'] = __('Download Selected', 'media-bulk-download');
        
        // Debug: Log that this function is being called (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Media Bulk Download: add_bulk_actions called with actions: ' . print_r($bulk_actions, true)); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
        }
        
        return $bulk_actions;
    }
    
    /**
     * Handle the bulk download action
     */
    public function handle_bulk_download($redirect_to, $doaction, $post_ids) {
        // Check if this is our action
        if ($doaction !== 'download_selected') {
            return $redirect_to;
        }
        
        // Security checks
        if (!$this->verify_security()) {
            wp_die(esc_html__('Security check failed. Please try again.', 'media-bulk-download'));
        }
        
        // Check user capabilities
        if (!current_user_can('upload_files') && !current_user_can('delete_posts')) {
            wp_die(esc_html__('You do not have permission to download media files.', 'media-bulk-download'));
        }
        
        // Validate post IDs
        if (empty($post_ids) || !is_array($post_ids)) {
            wp_die(esc_html__('No media files selected.', 'media-bulk-download'));
        }
        
        // Process the download
        $this->process_download($post_ids);
        
        // This should not be reached as process_download will exit
        return $redirect_to;
    }
    
    /**
     * Verify security nonce
     */
    private function verify_security() {
        // Check nonce for bulk actions
        if (!isset($_REQUEST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])), 'bulk-media')) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Process the actual download
     */
    private function process_download($post_ids) {
        // Filter valid attachment IDs - include ALL file types
        $valid_ids = array();
        foreach ($post_ids as $id) {
            // Check if it's a valid attachment and has a file
            if ($this->is_valid_attachment($id)) {
                $valid_ids[] = intval($id);
            }
        }
        
        if (empty($valid_ids)) {
            wp_die(esc_html__('No valid media files found in selection.', 'media-bulk-download'));
        }
        
        // Create temporary ZIP file
        $zip_file = $this->create_zip_file($valid_ids);
        
        if (!$zip_file || !file_exists($zip_file)) {
            wp_die(esc_html__('Failed to create ZIP file.', 'media-bulk-download'));
        }
        
        // Force download
        $this->force_download($zip_file);
        
        // Clean up
        $this->cleanup_zip_file($zip_file);
        
        exit;
    }
    
    /**
     * Register font MIME types to ensure they're recognized by WordPress
     */
    public function register_font_mime_types($mimes) {
        // Add font MIME types if not already present
        $font_mimes = array(
            'woff'  => 'font/woff',
            'woff2' => 'font/woff2',
            'ttf'   => 'font/ttf',
            'otf'   => 'font/otf',
            'eot'   => 'application/vnd.ms-fontobject',
            'svg'   => 'image/svg+xml'
        );
        
        foreach ($font_mimes as $ext => $mime) {
            if (!isset($mimes[$ext])) {
                $mimes[$ext] = $mime;
            }
        }
        
        return $mimes;
    }
    
    /**
     * Check if attachment is valid and has a downloadable file
     */
    private function is_valid_attachment($attachment_id) {
        // Check if it's a valid attachment post
        if (get_post_type($attachment_id) !== 'attachment') {
            return false;
        }
        
        // Check if the file exists using our improved method
        $file_path = $this->get_attachment_file_path($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return false;
        }
        
        // Include ALL file types - no restrictions
        // Additional check: ensure it's a file we can process
        $file_extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        $allowed_extensions = array(
            // Images
            'jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg',
            // Videos
            'mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv',
            // Audio
            'mp3', 'wav', 'ogg', 'm4a', 'aac', 'flac',
            // Documents
            'pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'csv', 'rtf',
            // Fonts
            'woff', 'woff2', 'ttf', 'otf', 'eot',
            // Archives
            'zip', 'rar', '7z', 'tar', 'gz',
            // Other
            'xml', 'json', 'css', 'js', 'html', 'htm'
        );
        
        // If it's a known file type, include it
        if (in_array($file_extension, $allowed_extensions)) {
            return true;
        }
        
        // For unknown extensions, still include them (no restrictions)
        return true;
    }
    
    /**
     * Check if attachment is a document (including fonts)
     */
    private function is_document($attachment_id) {
        $mime_type = get_post_mime_type($attachment_id);
        $document_types = array(
            // Standard documents
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-powerpoint',
            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            'text/plain',
            'text/csv',
            // Font files
            'font/woff',
            'font/woff2',
            'application/font-woff',
            'application/font-woff2',
            'application/x-font-woff',
            'application/x-font-woff2',
            'font/ttf',
            'application/font-sfnt',
            'application/x-font-ttf',
            'font/otf',
            'application/font-sfnt',
            'application/x-font-otf',
            'application/vnd.ms-fontobject', // EOT
            'image/svg+xml', // SVG
            'application/xml',
            'text/xml'
        );
        
        return in_array($mime_type, $document_types);
    }
    
    /**
     * Get attachment file path with fallback methods
     */
    private function get_attachment_file_path($attachment_id) {
        // First try the standard WordPress method
        $file_path = get_attached_file($attachment_id);
        
        if ($file_path && file_exists($file_path)) {
            return $file_path;
        }
        
        // Fallback: Try to get file path from attachment metadata
        $attachment_meta = wp_get_attachment_metadata($attachment_id);
        if ($attachment_meta && isset($attachment_meta['file'])) {
            $upload_dir = wp_upload_dir();
            $file_path = $upload_dir['basedir'] . '/' . $attachment_meta['file'];
            
            if (file_exists($file_path)) {
                return $file_path;
            }
        }
        
        // Fallback: Try to get file path from attachment URL
        $attachment_url = wp_get_attachment_url($attachment_id);
        if ($attachment_url) {
            $upload_dir = wp_upload_dir();
            $file_path = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $attachment_url);
            
            if (file_exists($file_path)) {
                return $file_path;
            }
        }
        
        // Last fallback: Try to get file path from post meta
        $file_path = get_post_meta($attachment_id, '_wp_attached_file', true);
        if ($file_path) {
            $upload_dir = wp_upload_dir();
            $full_path = $upload_dir['basedir'] . '/' . $file_path;
            
            if (file_exists($full_path)) {
                return $full_path;
            }
        }
        
        return false;
    }
    
    /**
     * Create ZIP file with selected media files
     */
    private function create_zip_file($attachment_ids) {
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive')) {
            wp_die(esc_html__('ZIP functionality is not available on this server.', 'media-bulk-download'));
        }
        
        // Create temporary file
        $temp_dir = wp_upload_dir();
        $zip_filename = 'media-bulk-download-' . time() . '-' . wp_generate_password(8, false) . '.zip';
        $zip_path = $temp_dir['basedir'] . '/' . $zip_filename;
        
        $zip = new ZipArchive();
        $result = $zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE);
        
        if ($result !== TRUE) {
            return false;
        }
        
        $added_files = 0;
        
        foreach ($attachment_ids as $attachment_id) {
            $file_path = $this->get_attachment_file_path($attachment_id);
            
            if (!$file_path || !file_exists($file_path)) {
                // Debug: Log files that couldn't be found
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Media Bulk Download: File not found for attachment ID ' . $attachment_id . ' - Path: ' . $file_path); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                }
                continue;
            }
            
            // Debug: Log files being processed
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $mime_type = get_post_mime_type($attachment_id);
                error_log('Media Bulk Download: Processing file - ID: ' . $attachment_id . ', Path: ' . $file_path . ', MIME: ' . $mime_type); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            }
            
            // Get original filename
            $filename = basename($file_path);
            
            // Ensure unique filename in ZIP
            $counter = 1;
            $original_filename = $filename;
            while ($zip->locateName($filename) !== false) {
                $pathinfo = pathinfo($original_filename);
                $filename = $pathinfo['filename'] . '_' . $counter . '.' . $pathinfo['extension'];
                $counter++;
            }
            
            // Add file to ZIP
            if ($zip->addFile($file_path, $filename)) {
                $added_files++;
            }
        }
        
        $zip->close();
        
        if ($added_files === 0) {
            // Remove empty ZIP file
            if (file_exists($zip_path)) {
                wp_delete_file($zip_path);
            }
            return false;
        }
        
        return $zip_path;
    }
    
    /**
     * Force download of the ZIP file
     */
    private function force_download($zip_path) {
        $filename = basename($zip_path);
        
        // Clear any previous output
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        // Set headers for file download
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . filesize($zip_path));
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
        
        // Output file using WordPress filesystem
        if (function_exists('wp_filesystem')) {
            global $wp_filesystem;
            if (empty($wp_filesystem)) {
                require_once ABSPATH . '/wp-admin/includes/file.php';
                WP_Filesystem();
            }
            if ($wp_filesystem && $wp_filesystem->exists($zip_path)) {
                // For binary files like ZIP, we need to output directly without escaping
                $file_contents = $wp_filesystem->get_contents($zip_path);
                if ($file_contents !== false) {
                    echo $file_contents; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                } else {
                    // Fallback to readfile if WP_Filesystem fails
                    readfile($zip_path); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile
                }
            } else {
                // Fallback to readfile if WP_Filesystem fails
                readfile($zip_path); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile
            }
        } else {
            // Fallback to readfile if wp_filesystem function doesn't exist
            readfile($zip_path); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile
        }
    }
    
    /**
     * Clean up temporary ZIP file
     */
    private function cleanup_zip_file($zip_path) {
        if (file_exists($zip_path)) {
            wp_delete_file($zip_path);
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on media library page
        if ($hook !== 'upload.php') {
            return;
        }
        
        // Add inline JavaScript to add download button to media toolbar
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                console.log("Media Bulk Download: Script loaded");
                
                // Function to add download button
                function addDownloadButton() {
                    if ($(".download-selected-button").length === 0) {
                        console.log("Media Bulk Download: Adding download button");
                        var downloadButton = $("<button type=\"button\" class=\"button media-button button-primary button-large download-selected-button\" style=\"display: none; margin-left: 10px;\" disabled=\"disabled\">Download Selected</button>");
                        var downloadAllButton = $("<button type=\"button\" class=\"button media-button button-secondary button-large download-all-button\" style=\"display: none; margin-left: 10px;\">Download All Media</button>");
                        
                        // Try different insertion methods
                        if ($(".delete-selected-button").length > 0) {
                            $(".delete-selected-button").after(downloadButton);
                            $(".delete-selected-button").after(downloadAllButton);
                            console.log("Media Bulk Download: Buttons added after delete button");
                        } else if ($(".select-mode-toggle-button").length > 0) {
                            $(".select-mode-toggle-button").before(downloadButton);
                            $(".select-mode-toggle-button").before(downloadAllButton);
                            console.log("Media Bulk Download: Buttons added before cancel button");
                        } else {
                            $(".media-toolbar-secondary").append(downloadButton);
                            $(".media-toolbar-secondary").append(downloadAllButton);
                            console.log("Media Bulk Download: Buttons added to toolbar");
                        }
                        
                        // Handle download button click
                        downloadButton.on("click", function() {
                            console.log("Media Bulk Download: Download button clicked");
                            var selectedIds = [];
                            
                            // Debug: Log all possible selectors
                            console.log("Media Bulk Download: Debugging selection detection:");
                            console.log("- input[name^=media[]]:checked count:", $("input[name^=\"media[]\"]:checked").length);
                            console.log("- .attachment.selected count:", $(".attachment.selected").length);
                            console.log("- .attachment input[type=checkbox]:checked count:", $(".attachment input[type=checkbox]:checked").length);
                            console.log("- .attachment input[type=checkbox] count:", $(".attachment input[type=checkbox]").length);
                            
                            // Method 1: Try standard WordPress media checkboxes
                            $("input[name^=\"media[]\"]:checked").each(function() {
                                selectedIds.push($(this).val());
                                console.log("Media Bulk Download: Found media[] checkbox:", $(this).val());
                            });
                            
                            // Method 2: Try attachment checkboxes
                            if (selectedIds.length === 0) {
                                $(".attachment input[type=checkbox]:checked").each(function() {
                                    selectedIds.push($(this).val());
                                    console.log("Media Bulk Download: Found attachment checkbox:", $(this).val());
                                });
                            }
                            
                            // Method 3: Try selected attachments
                            if (selectedIds.length === 0) {
                                $(".attachment.selected").each(function() {
                                    var checkbox = $(this).find("input[type=checkbox]");
                                    if (checkbox.length > 0) {
                                        selectedIds.push(checkbox.val());
                                        console.log("Media Bulk Download: Found selected attachment:", checkbox.val());
                                    }
                                });
                            }
                            
                            // Method 4: Try WordPress media library selection
                            if (selectedIds.length === 0 && window.wp && window.wp.media && window.wp.media.frame) {
                                try {
                                    var selection = window.wp.media.frame.state().get("selection");
                                    if (selection && selection.models) {
                                        selection.models.forEach(function(model) {
                                            selectedIds.push(model.id);
                                            console.log("Media Bulk Download: Found wp.media selection:", model.id);
                                        });
                                    }
                                } catch (e) {
                                    console.log("Media Bulk Download: Error accessing wp.media:", e);
                                }
                            }
                            
                            // Method 5: Try to get from data attributes
                            if (selectedIds.length === 0) {
                                $(".attachment.selected").each(function() {
                                    var attachmentId = $(this).data("id") || $(this).attr("data-id");
                                    if (attachmentId) {
                                        selectedIds.push(attachmentId);
                                        console.log("Media Bulk Download: Found data-id:", attachmentId);
                                    }
                                });
                            }
                            
                            console.log("Media Bulk Download: Final selected IDs:", selectedIds);
                            
                            // If still no IDs found, let us inspect the DOM structure
                            if (selectedIds.length === 0) {
                                console.log("Media Bulk Download: Inspecting DOM structure for debugging:");
                                console.log("All attachments:", $(".attachment").length);
                                console.log("All checkboxes:", $("input[type=checkbox]").length);
                                console.log("All media inputs:", $("input[name^=media]").length);
                                
                                // Log the HTML of selected attachments
                                $(".attachment.selected").each(function(index) {
                                    console.log("Selected attachment " + index + ":", $(this).html());
                                });
                                
                                // Log the HTML of checked checkboxes
                                $("input[type=checkbox]:checked").each(function(index) {
                                    console.log("Checked checkbox " + index + ":", $(this).attr("name"), $(this).val());
                                });
                                
                                alert("Please select at least one media file.");
                                console.log("Media Bulk Download: No media files detected as selected");
                                return;
                            }
                            
                            // Create form and submit
                            var form = $("<form method=\"post\" action=\"' . admin_url('admin-ajax.php') . '\">");
                            form.append("<input type=\"hidden\" name=\"action\" value=\"mbd_download_selected\">");
                            form.append("<input type=\"hidden\" name=\"_wpnonce\" value=\"' . wp_create_nonce('mbd_download_nonce') . '\">");
                            
                            $.each(selectedIds, function(index, id) {
                                form.append("<input type=\"hidden\" name=\"media_ids[]\" value=\"" + id + "\">");
                            });
                            
                            $("body").append(form);
                            form.submit();
                            form.remove();
                        });
                        
                        // Handle download all button click
                        downloadAllButton.on("click", function() {
                            console.log("Media Bulk Download: Download All button clicked");
                            
                            // Get all media IDs from the current page
                            var allIds = [];
                            $(".attachment").each(function() {
                                var checkbox = $(this).find("input[type=checkbox]");
                                if (checkbox.length > 0 && checkbox.val()) {
                                    allIds.push(checkbox.val());
                                }
                            });
                            
                            // Also try to get from data attributes
                            if (allIds.length === 0) {
                                $(".attachment").each(function() {
                                    var attachmentId = $(this).data("id") || $(this).attr("data-id");
                                    if (attachmentId) {
                                        allIds.push(attachmentId);
                                    }
                                });
                            }
                            
                            console.log("Media Bulk Download: All media IDs found:", allIds.length);
                            
                            if (allIds.length === 0) {
                                alert("No media files found to download.");
                                return;
                            }
                            
                            // Create form and submit
                            var form = $("<form method=\"post\" action=\"' . admin_url('admin-ajax.php') . '\">");
                            form.append("<input type=\"hidden\" name=\"action\" value=\"mbd_download_all\">");
                            form.append("<input type=\"hidden\" name=\"_wpnonce\" value=\"' . wp_create_nonce('mbd_download_nonce') . '\">");
                            
                            $.each(allIds, function(index, id) {
                                form.append("<input type=\"hidden\" name=\"media_ids[]\" value=\"" + id + "\">");
                            });
                            
                            $("body").append(form);
                            form.submit();
                            form.remove();
                        });
                        
                        // Initially hide the buttons (they will show when appropriate)
                        setTimeout(function() {
                            if ($(".download-selected-button").length > 0) {
                                $(".download-selected-button").hide();
                                console.log("Media Bulk Download: Download selected button initially hidden");
                            }
                            if ($(".download-all-button").length > 0) {
                                $(".download-all-button").hide();
                                console.log("Media Bulk Download: Download all button initially hidden");
                            }
                        }, 100);
                    }
                }
                
                // Function to update button visibility and state
                function updateButtonVisibility() {
                    // Check if bulk selection mode is active
                    var isBulkMode = false;
                    
                    // Method 1: Check if the toggle button shows "Cancel" (bulk mode active)
                    var toggleButtonText = $(".select-mode-toggle-button").text().trim();
                    console.log("Media Bulk Download: Toggle button text:", toggleButtonText);
                    
                    if (toggleButtonText === "Cancel") {
                        isBulkMode = true;
                        console.log("Media Bulk Download: Bulk mode detected via Cancel button");
                    }
                    
                    // Method 2: Check if delete button is visible and enabled (not disabled)
                    var deleteButtonVisible = $(".delete-selected-button").is(":visible");
                    var deleteButtonDisabled = $(".delete-selected-button").prop("disabled");
                    console.log("Media Bulk Download: Delete button visible:", deleteButtonVisible, "disabled:", deleteButtonDisabled);
                    
                    if (deleteButtonVisible && !deleteButtonDisabled) {
                        isBulkMode = true;
                        console.log("Media Bulk Download: Bulk mode detected via delete button state");
                    }
                    
                    // Method 3: Check if we can see attachment checkboxes (bulk mode indicator)
                    var visibleCheckboxes = $(".attachment input[type=checkbox]:visible").length;
                    console.log("Media Bulk Download: Visible checkboxes:", visibleCheckboxes);
                    
                    if (visibleCheckboxes > 0) {
                        isBulkMode = true;
                        console.log("Media Bulk Download: Bulk mode detected via visible checkboxes");
                    }
                    
                    var checkedCount = 0;
                    
                    // Only count selections if in bulk mode
                    if (isBulkMode) {
                        // Count different types of selections
                        checkedCount += $("input[name^=\"media[]\"]:checked").length;
                        checkedCount += $(".attachment.selected").length;
                        checkedCount += $(".attachment input[type=\"checkbox\"]:checked").length;
                    }
                    
                    console.log("Media Bulk Download: Final bulk mode status:", isBulkMode);
                    console.log("Media Bulk Download: Checked count:", checkedCount);
                    
                    // Show buttons based on mode
                    if (isBulkMode) {
                        // Show download selected button in bulk mode
                        $(".download-selected-button").show().removeClass("hidden");
                        console.log("Media Bulk Download: Showing download selected button (bulk mode active)");
                        
                        // Enable/disable based on selection
                        if (checkedCount > 0) {
                            $(".download-selected-button").prop("disabled", false);
                            console.log("Media Bulk Download: Download selected button enabled (items selected)");
                        } else {
                            $(".download-selected-button").prop("disabled", true);
                            console.log("Media Bulk Download: Download selected button disabled (no items selected)");
                        }
                        
                        // Hide download all button in bulk mode
                        $(".download-all-button").hide().addClass("hidden");
                        console.log("Media Bulk Download: Hiding download all button (bulk mode active)");
                    } else {
                        // Hide download selected button when not in bulk mode
                        $(".download-selected-button").hide().addClass("hidden");
                        console.log("Media Bulk Download: Hiding download selected button (not in bulk mode)");
                        
                        // Show download all button when not in bulk mode
                        $(".download-all-button").show().removeClass("hidden");
                        console.log("Media Bulk Download: Showing download all button (not in bulk mode)");
                    }
                }
                
                // Hook into WordPress media library events
                $(document).on("change", "input[name^=\"media[]\"], .attachment input[type=\"checkbox\"]", function() {
                    console.log("Media Bulk Download: Checkbox changed");
                    setTimeout(function() {
                        addDownloadButton();
                        updateButtonVisibility();
                    }, 100);
                });
                
                // Also listen for WordPress media library specific events
                $(document).on("attachment-selected", function() {
                    console.log("Media Bulk Download: Attachment selected");
                    setTimeout(function() {
                        addDownloadButton();
                        updateButtonVisibility();
                    }, 100);
                });
                
                $(document).on("attachment-deselected", function() {
                    console.log("Media Bulk Download: Attachment deselected");
                    setTimeout(function() {
                        updateButtonVisibility();
                    }, 100);
                });
                
                // Listen for clicks on media items
                $(document).on("click", ".attachment", function() {
                    console.log("Media Bulk Download: Attachment clicked");
                    setTimeout(function() {
                        addDownloadButton();
                        updateButtonVisibility();
                    }, 100);
                });
                
                // Listen for bulk selection changes
                $(document).on("click", ".select-mode-toggle-button", function() {
                    console.log("Media Bulk Download: Bulk selection mode toggled");
                    setTimeout(function() {
                        addDownloadButton();
                        updateButtonVisibility();
                    }, 100);
                });
                
                // Also listen for clicks on the bulk select button specifically
                $(document).on("click", "button[aria-label*=Select], .select-mode-toggle-button", function() {
                    console.log("Media Bulk Download: Bulk select button clicked");
                    setTimeout(function() {
                        addDownloadButton();
                        updateButtonVisibility();
                    }, 200);
                });
                
                // Try to add button immediately and then periodically
                addDownloadButton();
                
                // Initial state - ensure buttons are in correct state
                setTimeout(function() {
                    $(".download-selected-button").hide().addClass("hidden").prop("disabled", true);
                    $(".download-all-button").hide().addClass("hidden");
                    console.log("Media Bulk Download: Buttons initially set to correct state");
                    
                    // Force initial check
                    updateButtonVisibility();
                }, 100);
                
                // Also check periodically to ensure button state is correct
                var visibilityCheckInterval = setInterval(function() {
                    updateButtonVisibility();
                }, 1000);
                
                // Stop checking after 30 seconds
                setTimeout(function() {
                    clearInterval(visibilityCheckInterval);
                }, 30000);
                
                // Check periodically for the delete button and add our button
                var checkInterval = setInterval(function() {
                    if ($(".delete-selected-button").length > 0 && $(".download-selected-button").length === 0) {
                        console.log("Media Bulk Download: Found delete button, adding download button");
                        addDownloadButton();
                    }
                    
                    // Also check button visibility based on bulk mode
                    if ($(".download-selected-button").length > 0) {
                        updateButtonVisibility();
                    }
                }, 500);
                
                // Stop checking after 10 seconds
                setTimeout(function() {
                    clearInterval(checkInterval);
                }, 10000);
                
                // Also try to add button after a longer delay
                setTimeout(function() {
                    console.log("Media Bulk Download: Final attempt to add button");
                    addDownloadButton();
                    
                    // Check visibility based on bulk mode
                    updateButtonVisibility();
                }, 2000);
            });
        ');
    }
    
    /**
     * Add CSS to ensure button is visible
     */
    public function add_admin_css() {
        $screen = get_current_screen();
        if ($screen && $screen->id === 'upload') {
            echo '<style>
                .download-selected-button, .download-all-button {
                    display: inline-block !important;
                    visibility: visible !important;
                    opacity: 1 !important;
                }
                .media-toolbar-secondary .download-selected-button,
                .media-toolbar-secondary .download-all-button {
                    margin-left: 10px;
                }
                .download-selected-button:disabled {
                    opacity: 0.6 !important;
                    cursor: not-allowed !important;
                }
                /* Force hide when not in bulk mode */
                .download-selected-button.hidden,
                .download-all-button.hidden {
                    display: none !important;
                    visibility: hidden !important;
                    opacity: 0 !important;
                }
            </style>';
        }
    }
    
    /**
     * AJAX handler for download selected action
     */
    public function ajax_download_selected() {
        // Security checks
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'mbd_download_nonce')) {
            wp_die(esc_html__('Security check failed.', 'media-bulk-download'));
        }
        
        // Check user capabilities
        if (!current_user_can('upload_files') && !current_user_can('delete_posts')) {
            wp_die(esc_html__('You do not have permission to download media files.', 'media-bulk-download'));
        }
        
        // Get media IDs
        $post_ids = isset($_POST['media_ids']) ? array_map('intval', $_POST['media_ids']) : array();
        
        if (empty($post_ids)) {
            wp_die(esc_html__('No media files selected.', 'media-bulk-download'));
        }
        
        // Process the download
        $this->process_download($post_ids);
        
        exit;
    }
    
    /**
     * AJAX handler for download all action
     */
    public function ajax_download_all() {
        // Security checks
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'mbd_download_nonce')) {
            wp_die(esc_html__('Security check failed.', 'media-bulk-download'));
        }
        
        // Check user capabilities
        if (!current_user_can('upload_files') && !current_user_can('delete_posts')) {
            wp_die(esc_html__('You do not have permission to download media files.', 'media-bulk-download'));
        }
        
        // Get all media IDs from the database
        $args = array(
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );
        
        $all_attachments = get_posts($args);
        
        if (empty($all_attachments)) {
            wp_die(esc_html__('No media files found to download.', 'media-bulk-download'));
        }
        
        // Process the download
        $this->process_download($all_attachments);
        
        exit;
    }
    
    /**
     * Get information about font files in the media library
     */
    private function get_font_files_info() {
        $font_extensions = array('woff', 'woff2', 'ttf', 'otf', 'eot', 'svg');
        $font_files = array();
        
        // Get all attachments
        $args = array(
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );
        
        $attachments = get_posts($args);
        
        foreach ($attachments as $attachment_id) {
            $file_path = $this->get_attachment_file_path($attachment_id);
            if ($file_path) {
                $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
                if (in_array($extension, $font_extensions)) {
                    $filename = basename($file_path);
                    $mime_type = get_post_mime_type($attachment_id);
                    $font_files[] = $filename . ' (' . $extension . ' - ' . $mime_type . ')';
                }
            }
        }
        
        return $font_files;
    }
    
    /**
     * Display admin notices
     */
    public function admin_notices() {
        // Check if we have a notice to display
        if (isset($_GET['bulk_downloaded']) && $_GET['bulk_downloaded'] === '1') {
            // Verify nonce for security
            if (isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'bulk_downloaded_notice')) {
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p>' . esc_html__('Media files downloaded successfully.', 'media-bulk-download') . '</p>';
                echo '</div>';
            }
        }
        
        // Debug notice to show plugin is active (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $screen = get_current_screen();
            if ($screen && $screen->id === 'upload') {
                echo '<div class="notice notice-info is-dismissible">';
                echo '<p><strong>Media Bulk Download:</strong> Plugin is active. Screen ID: ' . esc_html($screen->id) . '</p>';
                echo '<p>Select media files to see "Download Selected" button next to "Delete permanently".</p>';
                echo '<p>Current user can upload files: ' . (current_user_can('upload_files') ? 'Yes' : 'No') . '</p>';
                
                // Show font file detection info
                $font_files = $this->get_font_files_info();
                if (!empty($font_files)) {
                    echo '<p><strong>Font files detected:</strong> ' . esc_html(implode(', ', $font_files)) . '</p>';
                } else {
                    echo '<p><strong>No font files detected in media library.</strong></p>';
                }
                
                echo '</div>';
            }
        }
    }
}

// Initialize the plugin
function mbd_init_plugin() {
    new MediaBulkDownload();
}
add_action('plugins_loaded', 'mbd_init_plugin');

// Activation hook
register_activation_hook(__FILE__, 'mbd_activate');
function mbd_activate() {
    // Check WordPress version
    if (version_compare(get_bloginfo('version'), '5.0', '<')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(esc_html__('This plugin requires WordPress 5.0 or higher.', 'media-bulk-download'));
    }
    
    // Check if ZipArchive is available
    if (!class_exists('ZipArchive')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(esc_html__('This plugin requires the ZipArchive PHP extension.', 'media-bulk-download'));
    }
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'mbd_deactivate');
function mbd_deactivate() {
    // Clean up any temporary files
    $upload_dir = wp_upload_dir();
    $pattern = $upload_dir['basedir'] . '/media-bulk-download-*.zip';
    $files = glob($pattern);
    
    if ($files) {
        foreach ($files as $file) {
            if (file_exists($file)) {
                wp_delete_file($file);
            }
        }
    }
}