/**
 * Pixxi Forms Frontend Handler
 * Handles form validation, submission, and user feedback
 */

(function($) {
    'use strict';

    // Initialize on document ready
    $(document).ready(function() {
        initializeForms();
    });

    function initializeForms() {
        $('.pixxi-form').each(function() {
            var $form = $(this);
            
            // Prevent multiple initializations
            if ($form.data('initialized')) {
                return;
            }
            $form.data('initialized', true);

            // Initialize phone country selector(s)
            initializePhoneCountrySelector($form);

            // Handle form submission
            $form.on('submit', function(e) {
                e.preventDefault();
                handleFormSubmission($form);
            });

            // Real-time validation
            $form.find('input, select, textarea').on('blur', function() {
                validateField($(this));
            });

            // Remove error on input
            $form.find('input, select, textarea').on('input change', function() {
                var $field = $(this);
                $field.removeClass('error');
                // Phone field errors are rendered under the whole form-field wrapper
                if ($field.closest('.pixxi-phone-field').length) {
                    $field.closest('.pixxi-form-field').find('.pixxi-field-error').remove();
                } else {
                    $field.siblings('.pixxi-field-error').remove();
                }
            });
        });
    }

    function initializePhoneCountrySelector($form) {
        var flagsBaseUrl = ($form.data('flags-url') || (window.pixxiFormsHandler && window.pixxiFormsHandler.flags_base_url) || '');
        if (typeof flagsBaseUrl !== 'string') {
            flagsBaseUrl = '';
        }

        $form.find('.pixxi-phone-field').each(function () {
            var $wrap = $(this);

            if ($wrap.data('phoneInitialized')) {
                return;
            }
            $wrap.data('phoneInitialized', true);

            var $trigger = $wrap.find('.pixxi-phone-country-trigger');
            var $localInput = $wrap.find('.pixxi-phone-input');
            var $hiddenPhone = $wrap.find('.pixxi-phone-e164');
            var $hiddenDial = $wrap.find('.pixxi-phone-dialcode');

            if ($wrap.hasClass('pixxi-phone-field-free')) {
                function syncE164Free() {
                    var dialCode = ($hiddenDial.val() || '').toString().trim();
                    var local = ($localInput.val() || '').toString().trim();
                    $hiddenPhone.val(dialCode && local ? dialCode + ' ' + local : '');
                }
                $localInput.on('input', syncE164Free);
                $form.on('submit.pixxiPhoneSync', function () { syncE164Free(); });
                $wrap.find('.pixxi-phone-country-placeholder').on('click', function (e) {
                    e.preventDefault();
                    var upgradeUrl = (window.pixxiFormsHandler && window.pixxiFormsHandler.upgrade_url) || '#';
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({
                            icon: 'info',
                            title: 'Country code selector',
                            html: 'Choose country and dial code with flags in <strong>Pixxi Forms Pro</strong>. <a href="' + upgradeUrl + '" target="_blank" rel="noopener">Upgrade to Pro</a>',
                            confirmButtonColor: '#000000'
                        });
                    } else {
                        window.open(upgradeUrl, '_blank');
                    }
                });
                return;
            }

            if ($trigger.length === 0) {
                return;
            }

            var countries = window.PixxiCountryData || [];
            var defaultIso2 = ($wrap.data('default-country') || 'ae').toString().toLowerCase();

            var $flag = $wrap.find('.pixxi-phone-flag');
            var $dial = $wrap.find('.pixxi-phone-dial');
            var $dropdown = $wrap.find('.pixxi-phone-dropdown');
            var $search = $wrap.find('.pixxi-phone-search');
            var $list = $wrap.find('.pixxi-phone-country-list');
            var $hiddenCountry = $wrap.find('.pixxi-phone-country');

            function normalize(str) {
                return (str || '').toString().toLowerCase().trim();
            }

            function setCountry(country) {
                if (!country || !country.iso2) {
                    return;
                }
                var iso2 = normalize(country.iso2);
                var dialCode = country.dialCode || '';

                $hiddenCountry.val(iso2);
                $hiddenDial.val(dialCode);
                $dial.text(dialCode);

                if (flagsBaseUrl) {
                    $flag.attr('src', flagsBaseUrl + iso2 + '.svg');
                } else {
                    $flag.removeAttr('src');
                }

                syncE164();
            }

            function syncE164() {
                var dialCode = ($hiddenDial.val() || '').toString().trim();
                var local = ($localInput.val() || '').toString().trim();
                var combined = '';

                if (dialCode && local) {
                    combined = dialCode + ' ' + local;
                }
                $hiddenPhone.val(combined);
            }

            function renderList(filterValue) {
                var q = normalize(filterValue);
                var items = countries;

                if (q) {
                    items = countries.filter(function (c) {
                        var name = normalize(c.name);
                        var dial = normalize(c.dialCode);
                        var iso2 = normalize(c.iso2);
                        return name.indexOf(q) !== -1 || dial.indexOf(q) !== -1 || iso2.indexOf(q) !== -1;
                    });
                }

                var html = '';
                items.forEach(function (c) {
                    var iso2 = normalize(c.iso2);
                    var flagSrc = flagsBaseUrl ? (flagsBaseUrl + iso2 + '.svg') : '';
                    html +=
                        '<button type="button" class="pixxi-phone-country-item" role="option" data-iso2="' + iso2 + '" data-dial="' + (c.dialCode || '') + '" data-name="' + (c.name || '') + '">' +
                            (flagSrc ? ('<img class="pixxi-phone-country-flag" src="' + flagSrc + '" alt="" />') : '') +
                            '<span class="pixxi-phone-country-name">' + (c.name || '') + '</span>' +
                            '<span class="pixxi-phone-country-dial">' + (c.dialCode || '') + '</span>' +
                        '</button>';
                });

                $list.html(html);
            }

            function openDropdown() {
                $dropdown.prop('hidden', false);
                $wrap.addClass('pixxi-phone-open');
                renderList($search.val());
                setTimeout(function () {
                    $search.trigger('focus');
                }, 0);
            }

            function closeDropdown() {
                $dropdown.prop('hidden', true);
                $wrap.removeClass('pixxi-phone-open');
            }

            function toggleDropdown() {
                if ($dropdown.prop('hidden')) {
                    openDropdown();
                } else {
                    closeDropdown();
                }
            }

            // Pick default country
            var defaultCountry = countries.find(function (c) { return normalize(c.iso2) === defaultIso2; }) ||
                countries.find(function (c) { return normalize(c.iso2) === 'ae'; }) ||
                countries[0];
            setCountry(defaultCountry);

            // Interactions
            $trigger.on('click', function () {
                toggleDropdown();
            });

            $search.on('input', function () {
                renderList($search.val());
            });

            $list.on('click', '.pixxi-phone-country-item', function () {
                var $btn = $(this);
                var iso2 = normalize($btn.data('iso2'));
                var dial = ($btn.data('dial') || '').toString();

                var selected = countries.find(function (c) { return normalize(c.iso2) === iso2; }) || { iso2: iso2, name: $btn.data('name'), dialCode: dial };
                setCountry(selected);
                closeDropdown();
            });

            $localInput.on('input', function () {
                syncE164();
            });

            // Close when clicking outside
            $(document).on('mousedown.pixxiPhoneCountry', function (e) {
                if ($wrap.has(e.target).length === 0) {
                    closeDropdown();
                }
            });

            // Ensure hidden phone is synced right before submit
            $form.on('submit.pixxiPhoneSync', function () {
                syncE164();
            });
        });
    }

    function handleFormSubmission($form) {
        // Clear previous messages
        $form.find('.pixxi-form-error, .pixxi-form-success').remove();

        // Validate all fields
        var isValid = true;
        $form.find('input[required], select[required], textarea[required]').each(function() {
            if (!validateField($(this))) {
                isValid = false;
            }
        });

        if (!isValid) {
            Swal.fire({
                icon: 'error',
                title: 'Validation Error',
                text: 'Please fill in all required fields correctly.',
                confirmButtonColor: '#000000'
            });
            return;
        }

        // Get form data
        var formData = {};
        
        // Add nonce
        formData.nonce = pixxiFormsHandler.nonce;
        formData.action = 'pixxi_submit_form';

        // Get all form fields
        $form.find('input, select, textarea').each(function() {
            var $field = $(this);
            if ($field.data('pixxi-skip-submit')) {
                return;
            }
            var name = $field.attr('name');
            var value = $field.val();

            if (name && value) {
                // Handle extraData fields
                if (name.startsWith('extraData[')) {
                    if (!formData.extraData) {
                        formData.extraData = {};
                    }
                    var fieldName = name.match(/extraData\[(.+?)\]/)[1];
                    formData.extraData[fieldName] = value;
                } else {
                    formData[name] = value;
                }
            }
        });

        var $phoneE164 = $form.find('.pixxi-phone-e164');
        if ($phoneE164.length) {
            formData.phone = $phoneE164.val() || '';
        }

        var $recaptcha = $form.find('[name="g-recaptcha-response"]');
        if ($recaptcha.length) {
            formData['g-recaptcha-response'] = $recaptcha.val() || '';
        }

        // Show loading state
        var $submitBtn = $form.find('button[type="submit"]');
        $submitBtn.prop('disabled', true).addClass('loading');

        // Submit via AJAX
        $.ajax({
            url: pixxiFormsHandler.ajax_url,
            method: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    var data = response.data || {};
                    if (data.redirect_url) {
                        $form[0].reset();
                        window.location = data.redirect_url;
                        return;
                    }
                    var message = data.success_message || data.message || 'Your form has been submitted successfully.';
                    Swal.fire({
                        icon: 'success',
                        title: 'Success!',
                        text: message,
                        confirmButtonColor: '#000000'
                    });
                    $form[0].reset();
                } else {
                    // Show error message
                    Swal.fire({
                        icon: 'error',
                        title: 'Submission Failed',
                        text: response.data.message || 'There was an error submitting your form. Please try again.',
                        confirmButtonColor: '#000000'
                    });
                }
            },
            error: function(xhr, status, error) {
                // Show error message
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'There was a network error. Please check your connection and try again.',
                    confirmButtonColor: '#000000'
                });
                console.error('Form submission error:', error);
            },
            complete: function() {
                // Remove loading state
                $submitBtn.prop('disabled', false).removeClass('loading');
            }
        });
    }

    function validateField($field) {
        var value = $field.val();
        var type = $field.attr('type');
        var isRequired = $field.prop('required');
        var isValid = true;
        var errorMessage = '';

        // Remove previous error
        $field.removeClass('error');
        if ($field.closest('.pixxi-phone-field').length) {
            $field.closest('.pixxi-form-field').find('.pixxi-field-error').remove();
        } else {
            $field.siblings('.pixxi-field-error').remove();
        }

        // Required validation
        if (isRequired && !value) {
            isValid = false;
            errorMessage = 'This field is required';
        }

        // Email validation
        if (isValid && type === 'email' && value) {
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(value)) {
                isValid = false;
                errorMessage = 'Please enter a valid email address';
            }
        }

        // Phone validation
        if (isValid && type === 'tel' && value) {
            var phoneRegex = /^[\d\s\-\+\(\)]+$/;
            if (!phoneRegex.test(value)) {
                isValid = false;
                errorMessage = 'Please enter a valid phone number';
            }
        }

        // URL validation
        if (isValid && type === 'url' && value) {
            var urlRegex = /^https?:\/\/.+/;
            if (!urlRegex.test(value)) {
                isValid = false;
                errorMessage = 'Please enter a valid URL';
            }
        }

        // Number validation
        if (isValid && type === 'number' && value) {
            if (isNaN(value)) {
                isValid = false;
                errorMessage = 'Please enter a valid number';
            }
        }

        // Show error if invalid
        if (!isValid) {
            $field.addClass('error');
            // Phone field: render error below the whole widget, not inside the flex row
            if ($field.closest('.pixxi-phone-field').length) {
                $field.closest('.pixxi-form-field').append('<div class="pixxi-field-error">' + errorMessage + '</div>');
            } else {
                $field.after('<div class="pixxi-field-error">' + errorMessage + '</div>');
            }
        }

        return isValid;
    }

    // Rate limiting to prevent spam
    var lastSubmitTime = 0;
    var minSubmitInterval = 3000; // 3 seconds

    $(document).on('submit', '.pixxi-form', function(e) {
        var currentTime = new Date().getTime();
        if (currentTime - lastSubmitTime < minSubmitInterval) {
            e.preventDefault();
            Swal.fire({
                icon: 'warning',
                title: 'Please Wait',
                text: 'Please wait a few seconds before submitting again.',
                confirmButtonColor: '#000000'
            });
            return false;
        }
        lastSubmitTime = currentTime;
    });

})(jQuery);

