<?php
/**
 * Pixxi Forms API Client
 * Handles communication with Pixxi CRM API
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_Forms_API_Client {

    private $api_url;
    private $api_token;
    private $form_id = '';

    public function __construct() {
        $this->api_url = get_option('pixxi_forms_api_url', 'https://dataapi.pixxicrm.ae/pixxiapi/webhook/v1/form');
        $this->api_token = get_option('pixxi_forms_api_token', '');
    }

    /**
     * Send form data to Pixxi CRM
     *
     * @param array  $data Form data
     * @param string $form_type 'listing' or 'general'
     * @param string $form_id Optional form ID for per-form credentials (Pro).
     * @return array Response array with success status and message
     */
    public function send_form_data($data, $form_type = 'general', $form_id = '') {
        $this->form_id = is_string($form_id) ? $form_id : '';
        $creds = apply_filters('pixxi_forms_api_credentials', array(
            'token' => $this->api_token,
            'url'   => $this->api_url,
        ), $this->form_id);
        $token = isset($creds['token']) ? $creds['token'] : $this->api_token;
        $url = isset($creds['url']) ? $creds['url'] : $this->api_url;

        // Validate API token
        if (empty($token)) {
            return array(
                'success' => false,
                'message' => 'API token is not configured. Please configure it in settings.',
                'response' => null
            );
        }

        // Prepare the request body
        $body = $this->prepare_request_body($data, $form_type);

        // Make API request (with optional per-form credentials)
        $response = $this->make_request($body, $token, $url);

        return $response;
    }

    /**
     * Prepare request body based on form type
     *
     * @param array $data Form data
     * @param string $form_type Form type
     * @return array Prepared body
     */
    private function prepare_request_body($data, $form_type) {
        $body = array(
            'formId' => isset($data['formId']) ? sanitize_text_field($data['formId']) : ''
        );

        // Add propertyId if provided (required for listing forms)
        if (isset($data['propertyId']) && !empty($data['propertyId'])) {
            $body['propertyId'] = sanitize_text_field($data['propertyId']);
        }

        // Add standard fields
        if (isset($data['name']) && !empty($data['name'])) {
            $body['name'] = sanitize_text_field($data['name']);
        }

        if (isset($data['email']) && !empty($data['email'])) {
            $body['email'] = sanitize_email($data['email']);
        }

        if (isset($data['phone']) && !empty($data['phone'])) {
            $body['phone'] = sanitize_text_field($data['phone']);
        }

        // Add extra data
        if (isset($data['extraData']) && is_array($data['extraData']) && !empty($data['extraData'])) {
            $body['extraData'] = array();
            foreach ($data['extraData'] as $key => $value) {
                $body['extraData'][sanitize_text_field($key)] = sanitize_text_field($value);
            }
        }

        return $body;
    }

    /**
     * Make HTTP request to Pixxi CRM API
     *
     * @param array  $body Request body
     * @param string $token Optional token (default from instance).
     * @param string $url Optional URL (default from instance).
     * @return array Response array
     */
    private function make_request($body, $token = null, $url = null) {
        if ($token === null) {
            $token = $this->api_token;
        }
        if ($url === null) {
            $url = $this->api_url;
        }
        $args = array(
            'method'  => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-PIXXI-TOKEN' => $token,
            ),
            'body'    => wp_json_encode($body),
        );

        $response = wp_remote_post($url, $args);

        // Check for errors
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => 'API request failed: ' . $response->get_error_message(),
                'response' => null
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);

        // Check response code
        if ($response_code >= 200 && $response_code < 300) {
            return array(
                'success' => true,
                'message' => 'Form submitted successfully to Pixxi CRM',
                'response' => $response_data,
                'code' => $response_code
            );
        } else {
            $error_message = 'API request failed with code ' . $response_code;
            if ($response_data && isset($response_data['message'])) {
                $error_message .= ': ' . $response_data['message'];
            }
            
            return array(
                'success' => false,
                'message' => $error_message,
                'response' => $response_data,
                'code' => $response_code
            );
        }
    }

    /**
     * Test API connection
     *
     * @return array Response array
     */
    public function test_connection() {
        if (empty($this->api_token)) {
            return array(
                'success' => false,
                'message' => 'API token is not configured.'
            );
        }

        // Send a test request with minimal data
        $test_data = array(
            'formId' => 'test-connection',
            'name' => 'Test Connection',
            'email' => 'test@example.com'
        );

        $body = $this->prepare_request_body($test_data, 'general');

        $args = array(
            'method' => 'POST',
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-PIXXI-TOKEN' => $this->api_token
            ),
            'body' => json_encode($body)
        );

        $response = wp_remote_post($this->api_url, $args);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => 'Connection failed: ' . $response->get_error_message()
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        // Connection is successful if we get any response from the API
        if ($response_code >= 200 && $response_code < 500) {
            return array(
                'success' => true,
                'message' => 'API connection successful (Response code: ' . $response_code . ')',
                'code' => $response_code,
                'response' => $response_body
            );
        } else {
            return array(
                'success' => false,
                'message' => 'Connection failed with code ' . $response_code,
                'code' => $response_code
            );
        }
    }
}

