<?php
/**
 * Pixxi Forms Handler
 * Handles form submissions and AJAX requests
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_Forms_Handler {
    
    public function __construct() {
        // Register AJAX handlers
        add_action('wp_ajax_pixxi_submit_form', array($this, 'handle_form_submission'));
        add_action('wp_ajax_nopriv_pixxi_submit_form', array($this, 'handle_form_submission'));
        
        // Admin AJAX handlers
        add_action('wp_ajax_pixxi_test_connection', array($this, 'test_api_connection'));
        add_action('wp_ajax_pixxi_save_form', array($this, 'save_form'));
        add_action('wp_ajax_pixxi_delete_form', array($this, 'delete_form'));
        add_action('wp_ajax_pixxi_get_form', array($this, 'get_form'));
    }

    /**
     * Handle form submission from frontend
     */
    public function handle_form_submission() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pixxi_forms_submit_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        // reCAPTCHA verification (Pro only; filter returns false if verification fails)
        $recaptcha_response = isset($_POST['g-recaptcha-response']) ? sanitize_text_field($_POST['g-recaptcha-response']) : '';
        if (apply_filters('pixxi_forms_verify_recaptcha', true, $recaptcha_response) === false) {
            wp_send_json_error(array('message' => 'reCAPTCHA verification failed. Please try again.'));
            return;
        }

        // Get form data
        $form_id = isset($_POST['formId']) ? sanitize_text_field($_POST['formId']) : '';
        $form_type = isset($_POST['formType']) ? sanitize_text_field($_POST['formType']) : 'general';
        
        if (empty($form_id)) {
            wp_send_json_error(array('message' => 'Form ID is required'));
            return;
        }

        // Required standard fields
        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        $phone = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';

        if (empty($name)) {
            wp_send_json_error(array('message' => 'Name is required'));
            return;
        }
        if (empty($email) || !is_email($email)) {
            wp_send_json_error(array('message' => 'A valid email is required'));
            return;
        }
        if (empty($phone)) {
            wp_send_json_error(array('message' => 'Phone is required'));
            return;
        }

        // Prepare data for API
        $form_data = array(
            'formId' => $form_id
        );

        // Add propertyId if present
        if (isset($_POST['propertyId']) && !empty($_POST['propertyId'])) {
            $form_data['propertyId'] = sanitize_text_field($_POST['propertyId']);
        }

        // Add standard fields
        $form_data['name'] = $name;
        $form_data['email'] = $email;
        $form_data['phone'] = $phone;

        // Add extra data fields
        if (isset($_POST['extraData']) && is_array($_POST['extraData'])) {
            $form_data['extraData'] = array();
            foreach ($_POST['extraData'] as $key => $value) {
                $form_data['extraData'][sanitize_text_field($key)] = sanitize_text_field($value);
            }
        }

        // Send to API (pass form_id for Pro multiple API profiles)
        $api_client = new Pixxi_Forms_API_Client();
        $response = $api_client->send_form_data($form_data, $form_type, $form_id);

        // Log the submission
        $logger = new Pixxi_Forms_Logger();
        $log_data = array(
            'form_id' => $form_id,
            'form_name' => $this->get_form_name($form_id),
            'form_data' => json_encode($form_data),
            'api_response' => json_encode($response['response']),
            'status' => $response['success'] ? 'success' : 'failed',
            'error_message' => !$response['success'] ? $response['message'] : null,
            'ip_address' => $this->get_client_ip(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : ''
        );
        $logger->log_submission($log_data);

        // Fire action for Pro (email, webhooks, etc.)
        if ($response['success']) {
            do_action('pixxi_forms_after_submission', $form_id, $form_data, $response);
        }

        // Return response
        if ($response['success']) {
            $payload = array('message' => $response['message']);
            $form_config = Pixxi_Forms_Settings::get_form_by_form_id($form_id);
            if ($form_config && !empty($form_config['settings']) && is_array($form_config['settings'])) {
                $settings = $form_config['settings'];
                $success_type = isset($settings['success_type']) ? $settings['success_type'] : 'message';
                if ($success_type === 'redirect') {
                    $redirect_url = '';
                    if (!empty($settings['redirect_url']) && is_string($settings['redirect_url'])) {
                        $redirect_url = esc_url_raw($settings['redirect_url']);
                    }
                    if (empty($redirect_url) && !empty($settings['redirect_page_id'])) {
                        $page_id = absint($settings['redirect_page_id']);
                        if ($page_id) {
                            $page_url = get_permalink($page_id);
                            if ($page_url) {
                                $redirect_url = $page_url;
                            }
                        }
                    }
                    if (!empty($redirect_url)) {
                        $payload['redirect_url'] = $redirect_url;
                    }
                } else {
                    if (!empty($settings['success_message']) && is_string($settings['success_message'])) {
                        $payload['success_message'] = $settings['success_message'];
                    }
                }
            }
            wp_send_json_success($payload);
        } else {
            wp_send_json_error(array('message' => $response['message']));
        }
    }

    /**
     * Test API connection (Admin only)
     */
    public function test_api_connection() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pixxi_forms_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        $api_client = new Pixxi_Forms_API_Client();
        $response = $api_client->test_connection();

        if ($response['success']) {
            wp_send_json_success($response);
        } else {
            wp_send_json_error($response);
        }
    }

    /**
     * Save form configuration (Admin only)
     */
    public function save_form() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pixxi_forms_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $form_name = isset($_POST['form_name']) ? sanitize_text_field($_POST['form_name']) : '';
        $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';
        $form_type = isset($_POST['form_type']) ? sanitize_text_field($_POST['form_type']) : 'general';

        $raw_mappings = isset($_POST['field_mappings']) && is_array($_POST['field_mappings']) ? $_POST['field_mappings'] : array();
        $raw_settings = isset($_POST['settings']) && is_array($_POST['settings']) ? $_POST['settings'] : array();

        if (!empty($raw_mappings)) {
            $field_mappings_sanitized = array();
            foreach ($raw_mappings as $row) {
                $field_mappings_sanitized[] = array(
                    'field_name'  => isset($row['field_name']) ? sanitize_text_field(wp_unslash($row['field_name'])) : '',
                    'field_label' => isset($row['field_label']) ? sanitize_text_field(wp_unslash($row['field_label'])) : '',
                    'field_type'  => isset($row['field_type']) ? sanitize_key(wp_unslash($row['field_type'])) : 'text',
                    'required'    => !empty($row['required']),
                );
            }
            $field_mappings = wp_json_encode($field_mappings_sanitized);
        } else {
            $field_mappings = null;
        }
        if (!empty($raw_settings)) {
            $settings_sanitized = array();
            $textarea_keys = array('success_message');
            foreach ($raw_settings as $k => $v) {
                $key = sanitize_key($k);
                if (!is_scalar($v)) {
                    $settings_sanitized[$key] = '';
                } elseif (in_array($key, $textarea_keys, true)) {
                    $settings_sanitized[$key] = sanitize_textarea_field(wp_unslash($v));
                } else {
                    $settings_sanitized[$key] = sanitize_text_field(wp_unslash($v));
                }
            }
            $settings = wp_json_encode($settings_sanitized);
        } else {
            $settings = null;
        }

        if (empty($form_name) || empty($form_id)) {
            wp_send_json_error(array('message' => 'Form name and Form ID are required'));
            return;
        }

        // On create, check if form_id is already used
        if ($id <= 0) {
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$table} WHERE form_id = %s LIMIT 1",
                $form_id
            ));
            if ($exists) {
                wp_send_json_error(array('message' => 'A form with this Pixxi CRM Form ID already exists. Use a unique Form ID or edit the existing form.'));
                return;
            }
        } else {
            // On update, check duplicate form_id only for other rows
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$table} WHERE form_id = %s AND id != %d LIMIT 1",
                $form_id,
                $id
            ));
            if ($exists) {
                wp_send_json_error(array('message' => 'A form with this Pixxi CRM Form ID already exists. Use a unique Form ID.'));
                return;
            }
        }

        $data = array(
            'form_name' => $form_name,
            'form_id' => $form_id,
            'form_type' => $form_type,
        );
        if ($field_mappings !== null) {
            $data['field_mappings'] = $field_mappings;
        }
        if ($settings !== null) {
            $data['settings'] = $settings;
        }

        if ($id > 0) {
            // Update existing form (preserve field_mappings/settings if not sent)
            $result = $wpdb->update($table, $data, array('id' => $id));
            if ($result !== false) {
                wp_send_json_success(array('message' => 'Form updated successfully', 'id' => $id));
            } else {
                $msg = $wpdb->last_error ? $wpdb->last_error : 'Failed to update form';
                wp_send_json_error(array('message' => $msg));
            }
        } else {
            // Insert new form (need field_mappings and settings for insert)
            $data['field_mappings'] = $field_mappings !== null ? $field_mappings : wp_json_encode(array());
            $data['settings'] = $settings !== null ? $settings : wp_json_encode(array());
            $result = $wpdb->insert($table, $data);
            if ($result) {
                wp_send_json_success(array('message' => 'Form created successfully', 'id' => $wpdb->insert_id));
            } else {
                $msg = $wpdb->last_error ? $wpdb->last_error : 'Failed to create form';
                wp_send_json_error(array('message' => $msg));
            }
        }
    }

    /**
     * Delete form (Admin only)
     */
    public function delete_form() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pixxi_forms_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;

        if ($id <= 0) {
            wp_send_json_error(array('message' => 'Invalid form ID'));
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';

        $result = $wpdb->delete($table, array('id' => $id));

        if ($result) {
            wp_send_json_success(array('message' => 'Form deleted successfully'));
        } else {
            wp_send_json_error(array('message' => 'Failed to delete form'));
        }
    }

    /**
     * Get form data (Admin only)
     */
    public function get_form() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pixxi_forms_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;

        if ($id <= 0) {
            wp_send_json_error(array('message' => 'Invalid form ID'));
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';

        $form = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id), ARRAY_A);

        if ($form) {
            $form['field_mappings'] = json_decode($form['field_mappings'], true);
            $form['settings'] = json_decode($form['settings'], true);
            wp_send_json_success($form);
        } else {
            wp_send_json_error(array('message' => 'Form not found'));
        }
    }

    /**
     * Get form name by form_id
     */
    private function get_form_name($form_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';
        $form = $wpdb->get_row($wpdb->prepare("SELECT form_name FROM $table WHERE form_id = %s", $form_id));
        return $form ? $form->form_name : 'Unknown Form';
    }

    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip = '';
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
        }
        return sanitize_text_field($ip);
    }
}

