<?php
/**
 * Pixxi Forms Logger
 * Handles logging and retrieval of form submissions
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_Forms_Logger {
    
    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'pixxi_submissions';
    }

    /**
     * Log form submission
     *
     * @param array $data Submission data
     * @return int|false Insert ID or false on failure
     */
    public function log_submission($data) {
        global $wpdb;

        $defaults = array(
            'form_id' => '',
            'form_name' => '',
            'form_data' => '',
            'api_response' => '',
            'status' => 'pending',
            'error_message' => null,
            'ip_address' => '',
            'user_agent' => ''
        );

        $data = wp_parse_args($data, $defaults);

        $result = $wpdb->insert($this->table_name, $data);

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Get submissions with pagination and filters
     *
     * @param array $args Query arguments
     * @return array Array of submissions
     */
    public function get_submissions($args = array()) {
        global $wpdb;

        $defaults = array(
            'per_page' => 20,
            'page' => 1,
            'form_id' => '',
            'status' => '',
            'date_from' => '',
            'date_to' => '',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );

        $args = wp_parse_args($args, $defaults);

        // Build WHERE clause
        $where = array('1=1');
        $where_values = array();

        if (!empty($args['form_id'])) {
            $where[] = 'form_id = %s';
            $where_values[] = $args['form_id'];
        }

        if (!empty($args['status'])) {
            $where[] = 'status = %s';
            $where_values[] = $args['status'];
        }

        if (!empty($args['date_from'])) {
            $where[] = 'created_at >= %s';
            $where_values[] = $args['date_from'] . ' 00:00:00';
        }

        if (!empty($args['date_to'])) {
            $where[] = 'created_at <= %s';
            $where_values[] = $args['date_to'] . ' 23:59:59';
        }

        $where_clause = implode(' AND ', $where);

        // Build ORDER BY clause
        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']);
        if (!$orderby) {
            $orderby = 'created_at DESC';
        }

        // Calculate offset
        $offset = ($args['page'] - 1) * $args['per_page'];

        // Build query
        $query = "SELECT * FROM {$this->table_name} WHERE {$where_clause} ORDER BY {$orderby} LIMIT %d OFFSET %d";
        $where_values[] = $args['per_page'];
        $where_values[] = $offset;

        if (!empty($where_values)) {
            $query = $wpdb->prepare($query, $where_values);
        }

        $results = $wpdb->get_results($query, ARRAY_A);

        return $results;
    }

    /**
     * Get total count of submissions
     *
     * @param array $args Filter arguments
     * @return int Total count
     */
    public function get_submissions_count($args = array()) {
        global $wpdb;

        $defaults = array(
            'form_id' => '',
            'status' => '',
            'date_from' => '',
            'date_to' => ''
        );

        $args = wp_parse_args($args, $defaults);

        // Build WHERE clause
        $where = array('1=1');
        $where_values = array();

        if (!empty($args['form_id'])) {
            $where[] = 'form_id = %s';
            $where_values[] = $args['form_id'];
        }

        if (!empty($args['status'])) {
            $where[] = 'status = %s';
            $where_values[] = $args['status'];
        }

        if (!empty($args['date_from'])) {
            $where[] = 'created_at >= %s';
            $where_values[] = $args['date_from'] . ' 00:00:00';
        }

        if (!empty($args['date_to'])) {
            $where[] = 'created_at <= %s';
            $where_values[] = $args['date_to'] . ' 23:59:59';
        }

        $where_clause = implode(' AND ', $where);

        $query = "SELECT COUNT(*) FROM {$this->table_name} WHERE {$where_clause}";

        if (!empty($where_values)) {
            $query = $wpdb->prepare($query, $where_values);
        }

        $count = $wpdb->get_var($query);

        return intval($count);
    }

    /**
     * Get single submission by ID
     *
     * @param int $id Submission ID
     * @return array|null Submission data or null
     */
    public function get_submission($id) {
        global $wpdb;

        $query = $wpdb->prepare("SELECT * FROM {$this->table_name} WHERE id = %d", $id);
        $result = $wpdb->get_row($query, ARRAY_A);

        return $result;
    }

    /**
     * Delete submission
     *
     * @param int $id Submission ID
     * @return bool Success status
     */
    public function delete_submission($id) {
        global $wpdb;

        $result = $wpdb->delete($this->table_name, array('id' => $id), array('%d'));

        return $result !== false;
    }

    /**
     * Get submission statistics
     *
     * @return array Statistics
     */
    public function get_statistics() {
        global $wpdb;

        $total = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        $success = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status = 'success'");
        $failed = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status = 'failed'");
        $today = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE DATE(created_at) = CURDATE()");

        return array(
            'total' => intval($total),
            'success' => intval($success),
            'failed' => intval($failed),
            'today' => intval($today)
        );
    }

    /**
     * Export submissions to CSV
     *
     * @param array $args Filter arguments
     * @return string CSV content
     */
    public function export_to_csv($args = array()) {
        $submissions = $this->get_submissions(array_merge($args, array('per_page' => 999999, 'page' => 1)));

        $csv = "ID,Form ID,Form Name,Status,Created At,IP Address\n";

        foreach ($submissions as $submission) {
            $csv .= sprintf(
                '"%s","%s","%s","%s","%s","%s"' . "\n",
                $submission['id'],
                $submission['form_id'],
                $submission['form_name'],
                $submission['status'],
                $submission['created_at'],
                $submission['ip_address']
            );
        }

        return $csv;
    }
}

