<?php
/**
 * Pixxi Forms Settings
 * Handles plugin settings management
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_Forms_Settings {
    
    /**
     * Get API token
     */
    public static function get_api_token() {
        return get_option('pixxi_forms_api_token', '');
    }

    /**
     * Set API token
     */
    public static function set_api_token($token) {
        return update_option('pixxi_forms_api_token', sanitize_text_field($token));
    }

    /**
     * Get API URL
     */
    public static function get_api_url() {
        return get_option('pixxi_forms_api_url', 'https://dataapi.pixxicrm.ae/pixxiapi/webhook/v1/form');
    }

    /**
     * Set API URL
     */
    public static function set_api_url($url) {
        return update_option('pixxi_forms_api_url', esc_url_raw($url));
    }

    /**
     * Default phone country option name
     */
    const DEFAULT_PHONE_COUNTRY_OPTION = 'pixxi_forms_default_phone_country';

    /**
     * Option name for enabling country code selector on phone field (Pro).
     */
    const PHONE_SHOW_COUNTRY_CODE_OPTION = 'pixxi_forms_phone_show_country_code';

    /**
     * Whether to show the country code selector on the phone field (Pro only; when false, plain phone input is shown).
     *
     * @return bool
     */
    public static function get_phone_show_country_code() {
        return (bool) get_option(self::PHONE_SHOW_COUNTRY_CODE_OPTION, true);
    }

    /**
     * Get list of countries for phone selector (Pro provides via filter when active).
     *
     * @return array List of [ 'iso2' => [ 'name' => string, 'dialCode' => string ], ... ]
     */
    public static function get_phone_countries() {
        return apply_filters('pixxi_forms_phone_countries', array());
    }

    /**
     * Get default phone country ISO2 code for the phone selector.
     *
     * @return string ISO2 code (e.g. 'ae')
     */
    public static function get_default_phone_country() {
        $saved = get_option(self::DEFAULT_PHONE_COUNTRY_OPTION, 'ae');
        $countries = self::get_phone_countries();
        return isset($countries[ strtolower($saved) ]) ? strtolower($saved) : 'ae';
    }

    /**
     * Get dial code for a country ISO2 (or default country if not found).
     *
     * @param string $iso2 Optional. ISO2 code. Uses default country if empty.
     * @return string Dial code including + (e.g. '+971')
     */
    public static function get_phone_dial_code($iso2 = '') {
        if (empty($iso2)) {
            $iso2 = self::get_default_phone_country();
        }
        $countries = self::get_phone_countries();
        $iso2 = strtolower($iso2);
        return isset($countries[ $iso2 ]) ? $countries[ $iso2 ]['dialCode'] : '+971';
    }

    /**
     * Get all forms
     */
    public static function get_all_forms() {
        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';
        $results = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC", ARRAY_A);
        
        // Decode JSON fields
        foreach ($results as &$form) {
            $form['field_mappings'] = json_decode($form['field_mappings'], true);
            $form['settings'] = json_decode($form['settings'], true);
        }
        
        return $results;
    }

    /**
     * Get form by ID
     */
    public static function get_form_by_id($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';
        $form = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id), ARRAY_A);
        
        if ($form) {
            $form['field_mappings'] = json_decode($form['field_mappings'], true);
            $form['settings'] = json_decode($form['settings'], true);
        }
        
        return $form;
    }

    /**
     * Get form by form_id
     */
    public static function get_form_by_form_id($form_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'pixxi_forms';
        $form = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE form_id = %s", $form_id), ARRAY_A);
        
        if ($form) {
            $form['field_mappings'] = json_decode($form['field_mappings'], true);
            $form['settings'] = json_decode($form['settings'], true);
        }
        
        return $form;
    }

}

