<?php
/**
 * Pixxi Forms Shortcodes
 * Handles shortcode registration and rendering
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_Forms_Shortcodes {
    
    public function __construct() {
        add_shortcode('pixxi_listing_form', array($this, 'render_listing_form'));
        add_shortcode('pixxi_general_form', array($this, 'render_general_form'));
    }

    /**
     * Render listing form shortcode
     * Usage: [pixxi_listing_form form_id="xxx" property_id="xxx" show_labels="yes"]
     * If property_id is not provided, it will be auto-detected from the current post
     */
    public function render_listing_form($atts) {
        $atts = shortcode_atts(array(
            'form_id' => '',
            'property_id' => '',
            'class' => '',
            'show_labels' => 'no'
        ), $atts);

        if (empty($atts['form_id'])) {
            return '<p class="pixxi-form-error">Error: Form ID is required</p>';
        }

        // Auto-detect property_id if not provided
        if (empty($atts['property_id'])) {
            $atts['property_id'] = $this->auto_detect_property_id();
        }

        if (empty($atts['property_id'])) {
            return '<p class="pixxi-form-error">Error: Invalid Property ID. The property ID (fave_property_id) is missing or empty. Please ensure you are on a valid property page with a property ID set.</p>';
        }

        return $this->render_form($atts['form_id'], 'listing', $atts['property_id'], $atts['class'], $atts['show_labels']);
    }

    /**
     * Render general form shortcode
     * Usage: [pixxi_general_form form_id="xxx" show_labels="yes"]
     * Property ID is optional - if not provided, will auto-detect if on property page
     */
    public function render_general_form($atts) {
        $atts = shortcode_atts(array(
            'form_id' => '',
            'property_id' => '',
            'class' => '',
            'show_labels' => 'no'
        ), $atts);

        if (empty($atts['form_id'])) {
            return '<p class="pixxi-form-error">Error: Form ID is required</p>';
        }

        // Auto-detect property_id if not provided (optional for general forms)
        if (empty($atts['property_id'])) {
            $atts['property_id'] = $this->auto_detect_property_id();
        }

        return $this->render_form($atts['form_id'], 'general', $atts['property_id'], $atts['class'], $atts['show_labels']);
    }

    /**
     * Render form HTML
     */
    private function render_form($form_id, $form_type, $property_id = '', $custom_class = '', $show_labels = 'yes') {
        // Get form configuration
        $form = Pixxi_Forms_Settings::get_form_by_form_id($form_id);
        
        $form_name = $form ? $form['form_name'] : 'Contact Form';
        $field_mappings = $form && isset($form['field_mappings']) ? $form['field_mappings'] : array();
        
        // Determine if labels should be shown
        $show_labels = in_array(strtolower($show_labels), array('yes', 'true', '1'));
        
        // Start output buffering
        ob_start();
        
        $unique_id = 'pixxi-form-' . uniqid();
        $form_class = 'pixxi-form pixxi-form-' . $form_type;
        if (!$show_labels) {
            $form_class .= ' pixxi-form-no-labels';
        }
        ?>
        
        <?php
        // Get agent information if property_id is available
        $agent_data = null;
        if (!empty($property_id)) {
            $agent_data = $this->get_agent_from_property($property_id);
        }
        ?>
        
        <?php $flags_base_url = apply_filters('pixxi_forms_flags_base_url', ''); ?>
        <div class="pixxi-form-wrapper <?php echo esc_attr($custom_class); ?>">
            <form id="<?php echo esc_attr($unique_id); ?>" class="<?php echo esc_attr($form_class); ?>" data-form-type="<?php echo esc_attr($form_type); ?>" data-flags-url="<?php echo esc_url($flags_base_url); ?>">
                
                <input type="hidden" name="formId" value="<?php echo esc_attr($form_id); ?>">
                <input type="hidden" name="formType" value="<?php echo esc_attr($form_type); ?>">
                
                <?php if (!empty($property_id)): ?>
                <input type="hidden" name="propertyId" value="<?php echo esc_attr($property_id); ?>">
                <?php endif; ?>

                <?php if ($agent_data): ?>
                    <!-- Agent Info Display -->
                    <div class="pixxi-agent-info">
                        <?php
                        // Fallback image when agent has no featured image (or if it fails to load)
                        $fallback_agent_image = 'https://twentyfirstcentury.demzdigital.website/wp-content/uploads/New-Project-1.jpg';
                        $agent_image_src = !empty($agent_data['image']) ? (string) $agent_data['image'] : $fallback_agent_image;
                        ?>
                        <div class="pixxi-agent-image">
                            <img
                                src="<?php echo esc_url($agent_image_src); ?>"
                                alt="<?php echo esc_attr($agent_data['name']); ?>"
                                onerror="this.onerror=null;this.src='<?php echo esc_url($fallback_agent_image); ?>';"
                            >
                        </div>
                        <div class="pixxi-agent-details">
                            <div class="pixxi-agent-name"><?php echo esc_html($agent_data['name']); ?></div>
                            <?php
                            $agent_url = (!empty($agent_data['id']) && is_numeric($agent_data['id']))
                                ? get_permalink((int) $agent_data['id'])
                                : '';
                            ?>
                            <?php if (!empty($agent_url)): ?>
                                <a class="pixxi-view-listing-btn" href="<?php echo esc_url($agent_url); ?>">
                                    View Listings
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="pixxi-form-fields">
                    
                    <!-- Name Field -->
                    <div class="pixxi-form-field">
                        <?php if ($show_labels): ?>
                            <label for="<?php echo esc_attr($unique_id); ?>-name">Name</label>
                        <?php endif; ?>
                        <input 
                            type="text" 
                            id="<?php echo esc_attr($unique_id); ?>-name" 
                            name="name" 
                            class="pixxi-input"
                            placeholder="Name *"
                            required
                        >
                    </div>

                    <!-- Email Field -->
                    <div class="pixxi-form-field">
                        <?php if ($show_labels): ?>
                            <label for="<?php echo esc_attr($unique_id); ?>-email">Email <span class="required">*</span></label>
                        <?php endif; ?>
                        <input 
                            type="email" 
                            id="<?php echo esc_attr($unique_id); ?>-email" 
                            name="email" 
                            class="pixxi-input"
                            placeholder="Email *"
                            required
                        >
                    </div>

                    <!-- Phone Field -->
                    <?php
                    $phone_default_iso2 = Pixxi_Forms_Settings::get_default_phone_country();
                    $phone_default_dial = Pixxi_Forms_Settings::get_phone_dial_code($phone_default_iso2);
                    $phone_show_country = function_exists('pixxi_forms_is_pro_active') && pixxi_forms_is_pro_active() && Pixxi_Forms_Settings::get_phone_show_country_code();
                    if ($phone_show_country) {
                        $phone_default_flag_src = $flags_base_url . $phone_default_iso2 . '.svg';
                    }
                    ?>
                    <div class="pixxi-form-field">
                        <?php if ($show_labels): ?>
                            <label for="<?php echo esc_attr($unique_id); ?>-phone">Phone</label>
                        <?php endif; ?>
                        <?php if ($phone_show_country): ?>
                        <div class="pixxi-phone-field" data-default-country="<?php echo esc_attr($phone_default_iso2); ?>">
                            <button class="pixxi-phone-country-trigger" type="button" aria-label="Select country">
                                <img class="pixxi-phone-flag" alt="" src="<?php echo esc_url($phone_default_flag_src); ?>" />
                                <span class="pixxi-phone-dial"><?php echo esc_html($phone_default_dial); ?></span>
                                <span class="pixxi-phone-chevron" aria-hidden="true"></span>
                            </button>

                            <div class="pixxi-phone-dropdown" hidden>
                                <div class="pixxi-phone-search-wrap">
                                    <input type="text" class="pixxi-phone-search" placeholder="Search country..." autocomplete="off" />
                                </div>
                                <div class="pixxi-phone-country-list" role="listbox" aria-label="Countries"></div>
                            </div>

                            <input
                                type="tel"
                                id="<?php echo esc_attr($unique_id); ?>-phone"
                                name="phone_local"
                                class="pixxi-input pixxi-phone-input"
                                placeholder="Mobile Number"
                                required
                                data-pixxi-skip-submit="1"
                                inputmode="tel"
                                autocomplete="tel-national"
                            >
                            <input type="hidden" name="phone" class="pixxi-phone-e164" value="" />
                            <input type="hidden" name="phoneCountry" class="pixxi-phone-country" value="<?php echo esc_attr($phone_default_iso2); ?>" data-pixxi-skip-submit="1" />
                            <input type="hidden" name="phoneDialCode" class="pixxi-phone-dialcode" value="<?php echo esc_attr($phone_default_dial); ?>" data-pixxi-skip-submit="1" />
                        </div>
                        <?php else : ?>
                        <div class="pixxi-phone-field pixxi-phone-field-free">
                            <input
                                type="tel"
                                id="<?php echo esc_attr($unique_id); ?>-phone"
                                name="phone_local"
                                class="pixxi-input pixxi-phone-input"
                                placeholder="Mobile Number"
                                required
                                data-pixxi-skip-submit="1"
                                inputmode="tel"
                                autocomplete="tel-national"
                            >
                            <input type="hidden" name="phone" class="pixxi-phone-e164" value="" />
                            <input type="hidden" name="phoneCountry" class="pixxi-phone-country" value="<?php echo esc_attr($phone_default_iso2); ?>" data-pixxi-skip-submit="1" />
                            <input type="hidden" name="phoneDialCode" class="pixxi-phone-dialcode" value="<?php echo esc_attr($phone_default_dial); ?>" data-pixxi-skip-submit="1" />
                        </div>
                        <?php endif; ?>
                    </div>

                    <?php
                    // Render custom fields from field mappings
                    if (!empty($field_mappings) && is_array($field_mappings)) {
                        foreach ($field_mappings as $field) {
                            if (!isset($field['field_name']) || in_array($field['field_name'], array('name', 'email', 'phone', 'propertyId', 'formId'))) {
                                continue;
                            }
                            
                            $field_name = sanitize_key($field['field_name']);
                            $field_label = isset($field['field_label']) && is_string($field['field_label']) ? $field['field_label'] : ucfirst($field_name);
                            $field_type = isset($field['field_type']) ? sanitize_key($field['field_type']) : 'text';
                            $field_required = isset($field['required']) && $field['required'];
                            $placeholder = $show_labels ? '' : $field_label . ($field_required ? ' *' : '');
                            ?>
                            <div class="pixxi-form-field">
                                <?php if ($show_labels): ?>
                                    <label for="<?php echo esc_attr($unique_id . '-' . $field_name); ?>">
                                        <?php echo esc_html($field_label); ?>
                                        <?php if ($field_required): ?><span class="required">*</span><?php endif; ?>
                                    </label>
                                <?php endif; ?>
                                
                                <?php if ($field_type === 'textarea'): ?>
                                    <textarea 
                                        id="<?php echo esc_attr($unique_id . '-' . $field_name); ?>"
                                        name="extraData[<?php echo esc_attr($field_name); ?>]"
                                        class="pixxi-input pixxi-textarea"
                                        rows="4"
                                        placeholder="<?php echo esc_attr($placeholder); ?>"
                                        <?php echo $field_required ? ' required' : ''; ?>
                                    ></textarea>
                                <?php else: ?>
                                    <input 
                                        type="<?php echo esc_attr($field_type); ?>"
                                        id="<?php echo esc_attr($unique_id . '-' . $field_name); ?>"
                                        name="extraData[<?php echo esc_attr($field_name); ?>]"
                                        class="pixxi-input"
                                        placeholder="<?php echo esc_attr($placeholder); ?>"
                                        <?php echo $field_required ? ' required' : ''; ?>
                                    >
                                <?php endif; ?>
                            </div>
                            <?php
                        }
                    }
                    ?>

                </div>

                <?php
                $recaptcha_site_key = function_exists('pixxi_forms_pro_get_recaptcha_site_key') ? pixxi_forms_pro_get_recaptcha_site_key() : '';
                $recaptcha_enabled = function_exists('pixxi_forms_is_pro_active') && pixxi_forms_is_pro_active() && !empty($recaptcha_site_key);
                if ($recaptcha_enabled) :
                ?>
                <div class="pixxi-form-field pixxi-recaptcha-field">
                    <div class="g-recaptcha" data-sitekey="<?php echo esc_attr($recaptcha_site_key); ?>"></div>
                    <noscript>Please enable JavaScript for reCAPTCHA.</noscript>
                </div>
                <?php endif; ?>

                <div class="pixxi-form-submit">
                    <?php
                    $agent_mobile = ($agent_data && !empty($agent_data['mobile'])) ? (string) $agent_data['mobile'] : '';
                    $agent_whatsapp = ($agent_data && !empty($agent_data['whatsapp'])) ? (string) $agent_data['whatsapp'] : '';

                    // Build safe link values
                    $tel_value = $agent_mobile ? preg_replace('/[^0-9+]/', '', $agent_mobile) : '';
                    $wa_digits = $agent_whatsapp ? preg_replace('/\D+/', '', $agent_whatsapp) : '';
                    ?>

                    <?php if (!empty($tel_value) || !empty($wa_digits)): ?>
                        <div class="pixxi-agent-contact-actions" aria-label="Agent contact actions">
                            <?php if (!empty($tel_value)): ?>
                                <a class="pixxi-btn pixxi-btn-contact" href="<?php echo esc_url('tel:' . $tel_value); ?>">
                                    Call
                                </a>
                            <?php endif; ?>
                            <?php if (!empty($wa_digits)): ?>
                                <a class="pixxi-btn pixxi-btn-contact" href="<?php echo esc_url('https://wa.me/' . $wa_digits); ?>" target="_blank" rel="noopener">
                                    WhatsApp
                                </a>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                    <button type="submit" class="pixxi-btn pixxi-btn-primary">
                        <span class="pixxi-btn-text">Send Message</span>
                        <span class="pixxi-btn-loader">
                            <svg class="pixxi-spinner" viewBox="0 0 50 50">
                                <circle class="path" cx="25" cy="25" r="20" fill="none" stroke-width="5"></circle>
                            </svg>
                        </span>
                    </button>
                </div>

            </form>
        </div>

        <?php
        return ob_get_clean();
    }

    /**
     * Auto-detect property ID from current post/page
     * Only uses fave_property_id meta field (Houzez theme)
     * Returns empty string if not found (will show error)
     * 
     * @return string Property ID or empty string if not found
     */
    private function auto_detect_property_id() {
        // Get current post ID first
        $post_id = get_the_ID();
        
        if (!$post_id) {
            return '';
        }

        // Only check for fave_property_id (Houzez theme)
        $fave_property_id = get_post_meta($post_id, 'fave_property_id', true);
        if (!empty($fave_property_id)) {
            // Handle array format: array(0 => '1032701261797')
            if (is_array($fave_property_id)) {
                // Get first element of array
                $fave_property_id = reset($fave_property_id);
            }
            if (!empty($fave_property_id)) {
                return (string) $fave_property_id;
            }
        }

        // Return empty if fave_property_id is not found
        return '';
    }

    /**
     * Get agent information from property
     * 
     * @param string $property_id The property ID (fave_property_id)
     * @return array|null Agent data (name, image) or null if not found
     */
    private function get_agent_from_property($property_id) {
        // Find property post by fave_property_id
        $property_posts = get_posts(array(
            'post_type' => array('property', 'properties', 'houzez_property', 'estate_property'),
            'meta_key' => 'fave_property_id',
            'meta_value' => $property_id,
            'posts_per_page' => 1,
            'fields' => 'ids'
        ));

        if (empty($property_posts)) {
            return null;
        }

        $property_post_id = $property_posts[0];

        // Get agent ID from property meta
        $agent_id = get_post_meta($property_post_id, 'fave_agents', true);
        
        // Handle array format
        if (is_array($agent_id)) {
            $agent_id = reset($agent_id);
        }

        if (empty($agent_id) || !is_numeric($agent_id)) {
            return null;
        }

        // Get agent post
        $agent_post = get_post($agent_id);
        if (!$agent_post || $agent_post->post_type !== 'houzez_agent') {
            return null;
        }

        // Get agent featured image
        $agent_image = '';
        if (has_post_thumbnail($agent_id)) {
            $image_url = get_the_post_thumbnail_url($agent_id, 'thumbnail');
            if ($image_url) {
                $agent_image = $image_url;
            }
        }

        // Agent meta contact details
        $agent_mobile = get_post_meta($agent_id, 'fave_agent_mobile', true);
        if (is_array($agent_mobile)) {
            $agent_mobile = reset($agent_mobile);
        }
        $agent_whatsapp = get_post_meta($agent_id, 'fave_agent_whatsapp', true);
        if (is_array($agent_whatsapp)) {
            $agent_whatsapp = reset($agent_whatsapp);
        }

        return array(
            'id' => (int) $agent_id,
            'name' => $agent_post->post_title,
            'image' => $agent_image,
            'mobile' => is_string($agent_mobile) ? $agent_mobile : '',
            'whatsapp' => is_string($agent_whatsapp) ? $agent_whatsapp : ''
        );
    }
}

