<?php
/**
 * Plugin Name: Pixxi Forms
 * Plugin URI:  https://devmizan.com/plugins/pixxi-forms/
 * Description: Send form submissions to Pixxi CRM with support for Listing and General form types.
 * Version: 1.0.2
 * Author: Dev Mizan
 * Author URI:  https://devmizan.com
 * License: GPL-3.0+
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Plugin Constants
define('PIXXI_FORMS_VERSION', '1.0.2');
define('PIXXI_FORMS_PATH', plugin_dir_path(__FILE__));
define('PIXXI_FORMS_URL', plugin_dir_url(__FILE__));
define('PIXXI_FORMS_TABLE', 'wp_pixxi_forms');
define('PIXXI_FORMS_SUBMISSIONS_TABLE', 'wp_pixxi_submissions');

// Include required files
require_once PIXXI_FORMS_PATH . 'includes/class-api-client.php';
require_once PIXXI_FORMS_PATH . 'includes/class-form-handler.php';
require_once PIXXI_FORMS_PATH . 'includes/class-settings.php';
require_once PIXXI_FORMS_PATH . 'includes/class-logger.php';
require_once PIXXI_FORMS_PATH . 'includes/class-shortcodes.php';

/**
 * Check if Pixxi Forms Pro is active.
 *
 * @return bool
 */
function pixxi_forms_is_pro_active() {
    return defined('PIXXI_FORMS_PRO_VERSION') || class_exists('Pixxi_Forms_Pro');
}

/**
 * Add "Get Pro" link on Plugins page when Pro is not active (highlighted).
 */
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'pixxi_forms_plugin_action_links', 10, 1);
function pixxi_forms_plugin_action_links($links) {
    if (pixxi_forms_is_pro_active()) {
        return $links;
    }
    $get_pro = sprintf(
        '<a href="%1$s" target="_blank" rel="noopener noreferrer" style="font-weight: 600; color: #d63638 !important;">%2$s</a>',
        esc_url('https://devmizan.com/plugins/pixxi-forms/'),
        __('Get Pro', 'pixxi-forms')
    );
    return array_merge(array('pixxi_get_pro' => $get_pro), $links);
}

/**
 * Plugin Activation Hook
 */
register_activation_hook(__FILE__, 'pixxi_forms_activate');
function pixxi_forms_activate() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    // Create forms configuration table
    $forms_table = $wpdb->prefix . 'pixxi_forms';
    $forms_sql = "CREATE TABLE IF NOT EXISTS $forms_table (
        id int(11) NOT NULL AUTO_INCREMENT,
        form_name varchar(255) NOT NULL,
        form_id varchar(255) NOT NULL,
        form_type varchar(50) NOT NULL,
        field_mappings longtext,
        settings longtext,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY form_id (form_id)
    ) $charset_collate;";

    // Create submissions log table
    $submissions_table = $wpdb->prefix . 'pixxi_submissions';
    $submissions_sql = "CREATE TABLE IF NOT EXISTS $submissions_table (
        id int(11) NOT NULL AUTO_INCREMENT,
        form_id varchar(255) NOT NULL,
        form_name varchar(255),
        form_data longtext NOT NULL,
        api_response longtext,
        status varchar(50) NOT NULL,
        error_message text,
        ip_address varchar(100),
        user_agent text,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY form_id (form_id),
        KEY status (status),
        KEY created_at (created_at)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($forms_sql);
    dbDelta($submissions_sql);

    // Set default options
    if (!get_option('pixxi_forms_api_token')) {
        add_option('pixxi_forms_api_token', '');
    }
    if (!get_option('pixxi_forms_api_url')) {
        add_option('pixxi_forms_api_url', 'https://dataapi.pixxicrm.ae/pixxiapi/webhook/v1/form');
    }
}

/**
 * Plugin Deactivation Hook
 */
register_deactivation_hook(__FILE__, 'pixxi_forms_deactivate');
function pixxi_forms_deactivate() {
    // Cleanup if needed
}

/**
 * Enqueue Admin Scripts and Styles
 */
add_action('admin_enqueue_scripts', 'pixxi_forms_admin_scripts');
function pixxi_forms_admin_scripts($hook) {
    // Only load on our plugin pages
    $current_page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
    $valid_pages = array('pixxi-forms-settings', 'pixxi-forms-manage', 'pixxi-forms-logs', 'pixxi-forms-field-mapping', 'pixxi-forms-pro-features');
    
    if (!in_array($current_page, $valid_pages)) {
        return;
    }

    $version = PIXXI_FORMS_VERSION;
    
    // Admin CSS
    wp_enqueue_style('pixxi-forms-admin', PIXXI_FORMS_URL . 'admin/css/admin-styles.css', array(), $version);
    
    // SweetAlert2
    wp_enqueue_style('sweetalert2', 'https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css', array(), '11');
    wp_enqueue_script('sweetalert2', 'https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.js', array(), '11', false);
    
    // Admin JS
    wp_enqueue_script('pixxi-forms-admin', PIXXI_FORMS_URL . 'admin/js/admin-scripts.js', array('jquery', 'sweetalert2'), $version, true);
    
    // Localize script
    wp_localize_script('pixxi-forms-admin', 'pixxiForms', array(
        'ajax_url' => esc_url(admin_url('admin-ajax.php')),
        'nonce' => wp_create_nonce('pixxi_forms_nonce')
    ));
}

/**
 * Enqueue Frontend Scripts and Styles
 */
add_action('wp_enqueue_scripts', 'pixxi_forms_frontend_scripts');
function pixxi_forms_frontend_scripts() {
    $version = PIXXI_FORMS_VERSION;
    
    // Frontend CSS
    wp_enqueue_style('pixxi-forms-frontend', PIXXI_FORMS_URL . 'assets/css/frontend-forms.css', array(), $version);
    
    // SweetAlert2
    wp_enqueue_style('sweetalert2', 'https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css', array(), '11');
    wp_enqueue_script('sweetalert2', 'https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.js', array(), '11', true);
    
    // Phone country dataset stub (Pro replaces with full data when active)
    wp_enqueue_script('pixxi-country-data', PIXXI_FORMS_URL . 'assets/js/pixxi-country-data.js', array(), $version, true);

    // reCAPTCHA script (only when Pro is active and has site key)
    if (apply_filters('pixxi_forms_need_recaptcha_script', false)) {
        wp_enqueue_script('google-recaptcha', 'https://www.google.com/recaptcha/api.js', array(), null, true);
    }

    // Frontend JS
    wp_enqueue_script('pixxi-forms-handler', PIXXI_FORMS_URL . 'assets/js/form-handler.js', array('jquery', 'sweetalert2', 'pixxi-country-data'), $version, true);

    // License checkout return: show SweetAlert when URL has pixxi_license=success|cancel|fail
    wp_enqueue_style('pixxi-forms-license-return-swal', PIXXI_FORMS_URL . 'assets/css/license-return-swal.css', array('sweetalert2'), $version);
    wp_enqueue_script('pixxi-forms-license-return', PIXXI_FORMS_URL . 'assets/js/license-return.js', array('sweetalert2'), $version, true);

    // Localize script (flags URL from Pro when active)
    wp_localize_script('pixxi-forms-handler', 'pixxiFormsHandler', array(
        'ajax_url' => esc_url(admin_url('admin-ajax.php')),
        'nonce' => wp_create_nonce('pixxi_forms_submit_nonce'),
        'flags_base_url' => esc_url(apply_filters('pixxi_forms_flags_base_url', '')),
        'upgrade_url' => esc_url('https://devmizan.com/plugins/pixxi-forms/')
    ));
}

/**
 * Add Admin Menu
 */
add_action('admin_menu', 'pixxi_forms_admin_menu');
function pixxi_forms_admin_menu() {
    add_menu_page(
        'Pixxi Forms',
        'Pixxi Forms',
        'manage_options',
        'pixxi-forms-settings',
        'pixxi_forms_settings_page',
        'dashicons-feedback',
        30
    );

    add_submenu_page(
        'pixxi-forms-settings',
        'Settings',
        'Settings',
        'manage_options',
        'pixxi-forms-settings',
        'pixxi_forms_settings_page'
    );

    add_submenu_page(
        'pixxi-forms-settings',
        'Manage Forms',
        'Manage Forms',
        'manage_options',
        'pixxi-forms-manage',
        'pixxi_forms_manage_page'
    );

    add_submenu_page(
        'pixxi-forms-settings',
        'Field Mapping',
        'Field Mapping',
        'manage_options',
        'pixxi-forms-field-mapping',
        'pixxi_forms_field_mapping_page'
    );

    add_submenu_page(
        'pixxi-forms-settings',
        'Submission Logs',
        'Submission Logs',
        'manage_options',
        'pixxi-forms-logs',
        'pixxi_forms_logs_page'
    );

    if (!pixxi_forms_is_pro_active()) {
        add_submenu_page(
            'pixxi-forms-settings',
            'Upgrade to Pro',
            'Upgrade to Pro',
            'manage_options',
            'pixxi-forms-pro-features',
            'pixxi_forms_pro_features_page'
        );
    }
}

/**
 * Admin Page Callbacks
 */
function pixxi_forms_settings_page() {
    require_once PIXXI_FORMS_PATH . 'admin/pages/settings.php';
}

function pixxi_forms_manage_page() {
    require_once PIXXI_FORMS_PATH . 'admin/pages/forms.php';
}

function pixxi_forms_field_mapping_page() {
    require_once PIXXI_FORMS_PATH . 'admin/pages/field-mapping.php';
}

function pixxi_forms_logs_page() {
    if (pixxi_forms_is_pro_active() && defined('PIXXI_FORMS_PRO_PATH')) {
        require_once PIXXI_FORMS_PRO_PATH . 'admin/pages/logs.php';
    } else {
        require_once PIXXI_FORMS_PATH . 'admin/pages/logs-upgrade-notice.php';
    }
}

function pixxi_forms_pro_features_page() {
    require_once PIXXI_FORMS_PATH . 'admin/pages/pro-features.php';
}

/**
 * Initialize Classes
 */
add_action('init', 'pixxi_forms_init');
function pixxi_forms_init() {
    // Initialize shortcodes
    new Pixxi_Forms_Shortcodes();
    
    // Initialize form handler
    new Pixxi_Forms_Handler();
}

