<?php
/**
 * Feature Gates for Pixxi Property Importer
 * Checks plan limits and feature access via API
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_License_Feature_Gates {
    
    private static $api_client = null;
    
    /**
     * Get API client instance
     */
    private static function get_api_client() {
        if (self::$api_client === null) {
            require_once PIXXI_PATH . 'includes/class-license-api-client.php';
            self::$api_client = new Pixxi_License_API_Client();
        }
        return self::$api_client;
    }
    
    /**
     * Check if license is active
     */
    public static function is_license_active() {
        $client = self::get_api_client();
        $license_data = $client->validate_license();
        return $license_data['valid'] ?? false;
    }
    
    /**
     * Check if feature is allowed
     */
    public static function check_feature($feature_name) {
        $client = self::get_api_client();
        $result = $client->check_feature($feature_name);
        return $result['allowed'] ?? false;
    }
    
    /**
     * Get plan limit
     */
    public static function get_limit($limit_name) {
        $client = self::get_api_client();
        return $client->get_limit($limit_name);
    }
    
    /**
     * Get current plan
     */
    public static function get_current_plan() {
        $client = self::get_api_client();
        $license_data = $client->validate_license();
        
        if ($license_data['valid']) {
            return $license_data['plan_id'] ?? null;
        }
        
        return null;
    }
    
    /**
     * Check if auto delete is allowed
     */
    public static function can_auto_delete() {
        return self::check_feature('auto_delete');
    }
    
    /**
     * Check if user has "easy" plan or higher (required for auto-delete features)
     */
    public static function has_easy_plan() {
        $client = self::get_api_client();
        $license_data = $client->validate_license();
        
        if (!$license_data['valid']) {
            return false;
        }
        
        $plan_id = $license_data['plan_id'] ?? '';
        
        // "easy" plan unique identifier check
        // Plans that have access to auto-delete features
        $allowed_plans = array('easy', 'pro', 'enterprise', 'unlimited', 'premium');
        
        return in_array(strtolower($plan_id), $allowed_plans);
    }
    
    /**
     * Check if auto delete properties feature is allowed
     */
    public static function can_auto_delete_properties() {
        return self::has_easy_plan();
    }
    
    /**
     * Check if orphaned taxonomy cleanup feature is allowed
     */
    public static function can_cleanup_orphaned_taxonomy() {
        return self::has_easy_plan();
    }
    
    /**
     * Get max properties limit
     */
    public static function get_max_properties() {
        return self::get_limit('max_properties');
    }
    
    /**
     * Get sync interval minutes
     */
    public static function get_sync_interval_minutes() {
        $interval = self::get_limit('sync_interval_minutes');
        return $interval > 0 ? $interval : 60; // Default to 60 minutes
    }
    
    /**
     * Report usage
     */
    public static function report_usage($feature, $count = 1) {
        $client = self::get_api_client();
        return $client->report_usage($feature, $count);
    }
}

