<?php
/**
 * API Client for Pixxi Property Importer
 * Communicates with License Manager API
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixxi_License_API_Client {
    
    private $api_url;
    private $license_key;
    private $api_key;
    private $domain;
    
    public function __construct($api_url = null, $license_key = null, $api_key = null) {
        // Hardcode API URL to devmizan.com
        $this->api_url = $api_url ?: 'https://devmizan.com';
        $this->license_key = $license_key ?: get_option('pixxi_license_key', '');
        $this->api_key = $api_key ?: get_option('pixxi_license_api_key', '');
        $this->domain = $this->get_current_domain();
    }
    
    /**
     * Validate license using license key or domain-based lookup
     */
    public function validate_license() {
        // Prioritize license key validation if available (manual activation)
        if (!empty($this->license_key)) {
            $cache_key_legacy = 'pixxi_license_validation_' . md5($this->license_key . $this->domain);
            $cached_legacy = get_transient($cache_key_legacy);
            
            if ($cached_legacy !== false) {
                return $cached_legacy;
            }
            
            if (!empty($this->api_url) && !empty($this->domain)) {
                $response = $this->make_request('POST', '/validate', array(
                    'license_key' => $this->license_key,
                    'domain' => $this->domain,
                    'plugin_version' => '1.0.0'
                ));
                
                if (!is_wp_error($response) && isset($response['valid'])) {
                    set_transient($cache_key_legacy, $response, 300); // Cache for 5 minutes
                    return $response;
                }
            }
        }
        
        // Fallback to domain-based lookup (for existing installations)
        $cache_key = 'pixxi_license_validation_' . md5($this->domain);
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        if (empty($this->api_url) || empty($this->domain)) {
            return array('valid' => false, 'error' => 'Domain not available');
        }
        
        $response = $this->make_request('GET', '/domain/' . urlencode($this->domain));
        
        if (!is_wp_error($response) && isset($response['valid'])) {
            set_transient($cache_key, $response, 300); // Cache for 5 minutes
            return $response;
        }
        
        return array('valid' => false, 'error' => 'No active license found for this domain');
    }
    
    /**
     * Get pricing plans
     */
    public function get_pricing_plans() {
        $cache_key = 'pixxi_license_plans';
        $cache_version_key = 'pixxi_license_plans_version';
        $cached = get_transient($cache_key);
        $cached_version = get_option($cache_version_key);
        
        // API URL is now hardcoded to devmizan.com
        if (empty($this->api_url)) {
            error_log('Pixxi License: API URL is empty');
            return array('plans' => array());
        }
        
        $response = $this->make_request('GET', '/plans');
        
        if (!is_wp_error($response) && isset($response['plans'])) {
            // Check if plans were updated (cache invalidation)
            $server_last_updated = isset($response['last_updated']) ? $response['last_updated'] : '';
            if ($cached !== false && $cached_version !== false && $cached_version !== $server_last_updated) {
                delete_transient($cache_key);
                delete_option($cache_version_key);
            }
            
            // Cache the response with version
            set_transient($cache_key, $response, 3600); // Cache for 1 hour
            update_option($cache_version_key, $server_last_updated);
            
            return $response;
        }
        
        // If we have cached data and API failed, return cached data
        if ($cached !== false) {
            return $cached;
        }
        
        // Log error for debugging
        if (is_wp_error($response)) {
            error_log('Pixxi License: Failed to fetch plans - ' . $response->get_error_message());
            error_log('Pixxi License: API URL was: ' . $this->api_url);
        } else {
            error_log('Pixxi License: Plans response missing or invalid. Response: ' . print_r($response, true));
        }
        
        return array('plans' => array());
    }
    
    /**
     * Check feature access
     */
    public function check_feature($feature) {
        if (empty($this->api_url) || empty($this->license_key)) {
            return array('allowed' => false);
        }
        
        $domain = $this->get_current_domain();
        $response = $this->make_request('POST', '/check-feature', array(
            'license_key' => $this->license_key,
            'domain' => $domain,
            'feature' => $feature
        ));
        
        if (!is_wp_error($response) && isset($response['allowed'])) {
            return $response;
        }
        
        return array('allowed' => false);
    }
    
    /**
     * Get limit value
     */
    public function get_limit($limit_name) {
        $license_data = $this->validate_license();
        
        if (!$license_data['valid']) {
            return -1;
        }
        
        $limits = $license_data['limits'] ?? array();
        return $limits[$limit_name] ?? -1;
    }
    
    /**
     * Report usage
     */
    public function report_usage($feature, $count = 1) {
        if (empty($this->api_url) || empty($this->license_key)) {
            return false;
        }
        
        $domain = $this->get_current_domain();
        $response = $this->make_request('POST', '/usage', array(
            'license_key' => $this->license_key,
            'domain' => $domain,
            'feature' => $feature,
            'count' => $count
        ));
        
        return !is_wp_error($response);
    }
    
    /**
     * Make API request
     */
    private function make_request($method, $endpoint, $data = array()) {
        $url = rtrim($this->api_url, '/') . '/wp-json/pixxi-license/v1' . $endpoint;
        
        $args = array(
            'method' => $method,
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'sslverify' => true
        );
        
        // Only add API key if provided (optional for public endpoints)
        if (!empty($this->api_key)) {
            $args['headers']['X-API-Key'] = $this->api_key;
        }
        
        if ($method === 'POST' && !empty($data)) {
            $args['body'] = json_encode($data);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            error_log('Pixxi License API Error: ' . $response->get_error_message() . ' | URL: ' . $url);
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status = wp_remote_retrieve_response_code($response);
        
        if ($status >= 200 && $status < 300) {
            $decoded = json_decode($body, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                error_log('Pixxi License: JSON decode error - ' . json_last_error_msg() . ' | Body: ' . substr($body, 0, 200));
                return new WP_Error('json_error', 'Failed to decode JSON response', array('body' => substr($body, 0, 200)));
            }
            return $decoded;
        }
        
        error_log('Pixxi License API HTTP Error: Status ' . $status . ' | URL: ' . $url . ' | Body: ' . substr($body, 0, 200));
        return new WP_Error('api_error', 'API request failed with status ' . $status, array('status' => $status, 'body' => substr($body, 0, 200)));
    }
    
    /**
     * Verify Stripe checkout session and activate license
     */
    public function verify_checkout_session($session_id) {
        if (empty($this->api_url) || empty($session_id)) {
            return array('success' => false, 'error' => 'Missing parameters');
        }
        
        $response = $this->make_request('POST', '/verify-checkout', array(
            'session_id' => $session_id,
            'domain' => $this->domain
        ));
        
        if (!is_wp_error($response) && isset($response['success'])) {
            return $response;
        }
        
        return array('success' => false, 'error' => 'Failed to verify session');
    }
    
    /**
     * Get Stripe Billing Portal URL for switch plan / cancel.
     *
     * @param string $return_url URL to redirect after leaving the portal (e.g. Subscriptions page).
     * @return string|WP_Error Portal URL or error.
     */
    public function get_subscription_portal_url($return_url = '') {
        if (empty($this->license_key) || empty($this->domain)) {
            return new WP_Error('missing_license', 'License key and domain are required');
        }
        if (empty($return_url)) {
            $return_url = is_admin() ? admin_url('admin.php?page=pixxi-subscriptions') : home_url('/');
        }
        $response = $this->make_request('POST', '/subscription/portal', array(
            'license_key' => $this->license_key,
            'domain' => $this->domain,
            'return_url' => $return_url,
        ));
        if (is_wp_error($response)) {
            return $response;
        }
        if (empty($response['success']) || empty($response['url'])) {
            return new WP_Error('portal_failed', isset($response['message']) ? $response['message'] : 'Failed to get portal URL');
        }
        return $response['url'];
    }
    
    /**
     * Cancel subscription: mark license cancelled and cancel Stripe subscription if present.
     *
     * @return true|WP_Error
     */
    public function cancel_subscription() {
        if (empty($this->license_key) || empty($this->domain)) {
            return new WP_Error('missing_license', 'License key and domain are required');
        }
        $response = $this->make_request('POST', '/subscription/cancel', array(
            'license_key' => $this->license_key,
            'domain' => $this->domain,
        ));
        if (is_wp_error($response)) {
            return $response;
        }
        if (empty($response['success'])) {
            return new WP_Error('cancel_failed', isset($response['message']) ? $response['message'] : 'Failed to cancel subscription');
        }
        return true;
    }
    
    /**
     * Get current domain
     */
    public function get_current_domain() {
        $domain = $_SERVER['HTTP_HOST'] ?? '';
        if (empty($domain)) {
            $domain = parse_url(home_url(), PHP_URL_HOST);
        }
        // Remove port if present
        if (strpos($domain, ':') !== false) {
            $domain = explode(':', $domain)[0];
        }
        return $domain;
    }
}

