<?php
class pixxi_property_class{
	function init(){
		if(isset($_GET['smart_download'])){
			file_put_contents(PIXXI_DOWN, time());
			$data['action'] = 'delete';
			$data['chunk'] = 0;
			$data['size'] = 1000;
			update_option('pixxi_api_info',json_encode($data));
			$this->get_curl_data();
		}
		if(isset($_GET['smart_update'])){
			file_put_contents(PIXXI_UPDATE, time());
			$data = get_option('pixxi_api_info');
			// Migration: Check for old option name
			if(!$data) {
				$data = get_option('lw_api_info');
				if($data) {
					update_option('pixxi_api_info', $data);
				}
			}
			$lw = json_decode($data,true);
			if($lw['action']!='stop'){
				$this->update_products($lw);
			}
			// Clean output before exit
			while (ob_get_level()) {
				ob_end_clean();
			}
			return;
		}
	}
	function get_curl_data(){
		// Start output buffering to capture any echo statements
		ob_start();
		
		// Check license before proceeding
		if (class_exists('Pixxi_License_Feature_Gates')) {
			if (!Pixxi_License_Feature_Gates::is_license_active()) {
				return array('error' => 'License is not active. Please check your license settings.');
			}
		}
		
		// Get API settings from database
		$opt = get_option('pixxi_opt_val');
		// Migration: Check for old option name
		if(!$opt) {
			$opt = get_option('lw_opt_val');
			if($opt) {
				update_option('pixxi_opt_val', $opt);
			}
		}
		$pixxi_option = json_decode($opt, true);
		
		if(!is_array($pixxi_option)) {
			$pixxi_option = array();
		}
		
		// Get API URL (full endpoint) and Token from dashboard settings
		$api_url = !empty($pixxi_option['pixxi_apiurl']) ? trim($pixxi_option['pixxi_apiurl']) : (!empty($pixxi_option['lw_apiurl']) ? trim($pixxi_option['lw_apiurl']) : '');
		$api_token = !empty($pixxi_option['pixxi_token']) ? $pixxi_option['pixxi_token'] : (!empty($pixxi_option['lw_token']) ? $pixxi_option['lw_token'] : '');
		
		// Validate required settings
		if(empty($api_url)) {
			return array('error' => 'API Endpoint URL is required. Please configure it in settings.');
		}
		
		if(empty($api_token)) {
			return array('error' => 'API Token is required. Please configure it in settings.');
		}
		
		// Ensure URL encoding is preserved - check if company name part needs encoding
		// Extract company name from URL and ensure it's properly encoded with %20 (not +)
		if(preg_match('#/v1/properties/(.+)$#', $api_url, $matches)) {
			$company_name = $matches[1];
			$base_url = str_replace('/v1/properties/' . $company_name, '', $api_url);
			
			// If company name contains spaces but no %20, encode it with rawurlencode
			if(strpos($company_name, ' ') !== false && strpos($company_name, '%20') === false && strpos($company_name, '+') === false) {
				$company_name = rawurlencode($company_name);
			}
			// If company name has + encoding, convert to %20
			elseif(strpos($company_name, '+') !== false) {
				$decoded = urldecode($company_name); // This handles + as spaces
				$company_name = rawurlencode($decoded); // Re-encode with %20
			}
			// If company name is already decoded (has spaces), encode it
			elseif(strpos($company_name, ' ') !== false) {
				$company_name = rawurlencode($company_name);
			}
			
			$api_url = $base_url . '/v1/properties/' . $company_name;
		}
		
		// Use the API URL directly (it should already include the full endpoint path)
		$url = $api_url;
		
		// Prepare request body according to API documentation
		$request_body = json_encode(array(
			'page' => 1,
			'size' => 1000,
			'status' => 'ACTIVE',
			'sort' => 'ID',
			'sortType' => 'DESC'
		));
		
		$curl = curl_init();
		curl_setopt_array($curl, array(
			CURLOPT_URL => $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 300, // 5 minutes timeout
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => 'POST',
			CURLOPT_POSTFIELDS => $request_body,
			CURLOPT_HTTPHEADER => array(
				'X-PIXXI-TOKEN: ' . $api_token,
				'Content-Type: application/json'
			),
		));
		
		$response = curl_exec($curl);
		$httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$curl_error = curl_error($curl);
		curl_close($curl);
		
		// Handle errors
		if($curl_error) {
			return array('error' => 'CURL Error: ' . $curl_error);
		}
		
		if($httpcode != 200) {
			$error_msg = 'HTTP Error ' . $httpcode;
			if($response) {
				$error_data = json_decode($response, true);
				if(isset($error_data['msg'])) {
					$error_msg .= ': ' . $error_data['msg'];
				} elseif(isset($error_data['message'])) {
					$error_msg .= ': ' . $error_data['message'];
				} else {
					$error_msg .= ': ' . substr(strip_tags($response), 0, 200);
				}
			} else {
				$error_msg .= ': No response from server';
			}
			return array('error' => $error_msg, 'http_code' => $httpcode);
		}
		
		// Validate response is valid JSON
		$response_data = json_decode($response, true);
		if(json_last_error() !== JSON_ERROR_NONE) {
			return array('error' => 'Invalid JSON response from API: ' . json_last_error_msg());
		}
		
		// Check API response status
		if(isset($response_data['statusCode']) && $response_data['statusCode'] != 200) {
			$error_msg = isset($response_data['message']) ? $response_data['message'] : 'Unknown API error';
			return array('error' => 'API Error: ' . $error_msg);
		}
		
		// Save response and process
		file_put_contents(PIXXI_LOCAL, $response);
		$download_result = $this->download_curl();
		
		// Discard any output from get_curl_data
		ob_end_clean();
		
		// Check for errors in download processing
		if(isset($download_result['error'])) {
			return $download_result;
		}
		
		$message = 'Properties downloaded successfully';
		if(isset($download_result['inserted'])) {
			$message .= ' (' . $download_result['inserted'] . ' properties inserted)';
		}
		
		return array('success' => true, 'message' => $message, 'data' => $download_result);
	}
	function download_curl(){
		// Start output buffering to capture any echo statements
		ob_start();
		
		$datafile = file_get_contents(PIXXI_LOCAL);
		$dataArray = json_decode($datafile, true);
		
		if(!$dataArray || !isset($dataArray['data'])) {
			ob_end_clean();
			return array('error' => 'Invalid response format from API');
		}
		
		// Check if list exists in response
		if(!isset($dataArray['data']['list']) || !is_array($dataArray['data']['list'])) {
			ob_end_clean();
			return array('error' => 'No properties list found in API response');
		}
		
		// Check property limit from license
		$max_properties = -1; // Unlimited by default
		if (class_exists('Pixxi_License_Feature_Gates')) {
			$max_properties = Pixxi_License_Feature_Gates::get_max_properties();
		}
		
		global $wpdb;
		$wpdb->query( "TRUNCATE TABLE ". PIXXI_TABLE  );
		
		$inserted = 0;
		$total_items = count($dataArray['data']['list']);
		
		// Limit items based on license
		if ($max_properties > 0 && $total_items > $max_properties) {
			$dataArray['data']['list'] = array_slice($dataArray['data']['list'], 0, $max_properties);
		}
		
		foreach($dataArray['data']['list'] as $item){
			if(!isset($item['id'])) {
				continue; // Skip items without ID
			}
			
			$time = (int)time(); // Ensure it's an integer
			$data = array(
				'pid'=> '',
				'sku'=> (string)$item['id'], // Ensure SKU is a string
				'data'=> json_encode($item),
				'status'=> '',
				'updated'=> $time
			);
			
			$result = $wpdb->insert(PIXXI_TABLE, $data);
			if($result) {
				$inserted++;
			}
		}
		
		// Report usage to license API
		if (class_exists('Pixxi_License_Feature_Gates')) {
			Pixxi_License_Feature_Gates::report_usage('property_sync', $inserted);
		}
		
		// Discard any output
		ob_end_clean();
		
		$result = array('success' => true, 'inserted' => $inserted, 'total' => count($dataArray['data']['list']));
		
		// Add warning if limit reached
		if ($max_properties > 0 && $total_items > $max_properties) {
			$result['warning'] = "Property limit reached. Only $max_properties properties imported out of $total_items available. Please upgrade your plan for unlimited properties.";
		}
		
		return $result;
	}
	function get_save_sku(){
		global $wpdb;
		$posts = $wpdb->get_results("select * from {$wpdb->prefix}postmeta where meta_key='_xm_api_sku'");
		$data = [];
		foreach($posts as $p){
			$data[$p->meta_value] = $p->post_id;
		}
		return $data;
	}
	function get_products_data(){
		global $wpdb;
		$products = $wpdb->get_results("select * from " . PIXXI_TABLE .'');
		$data = [];
		foreach($products as $product){
			$data[$product->sku] =  $product->data;
		}
		return $data;
	}
	function attach_product_thumbnail($p_id, $post_id, $image_paths_arr) {
		if(empty($image_paths_arr) || !is_array($image_paths_arr)) {
			return false;
		}
		
		// Clear existing gallery images to prevent duplicates
		delete_post_meta($post_id, 'fave_property_images');
		
		$_x = 0;
		$attach_id_array = '';
		
		foreach($image_paths_arr as $image_path) {
			// Skip if image path is empty or invalid
			if(empty($image_path) || !is_string($image_path)) {
				continue;
			}
			
			// Validate URL
			if(!filter_var($image_path, FILTER_VALIDATE_URL)) {
				error_log('Invalid image URL: ' . $image_path);
				continue;
			}
			
			try {
				$image_path_name = basename(parse_url($image_path, PHP_URL_PATH));
				if(empty($image_path_name)) {
					$image_path_name = 'image_' . uniqid();
				}
				
				$info = pathinfo($image_path_name);
				$extension = isset($info['extension']) ? $info['extension'] : 'jpg';
				
				// Validate extension
				$allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'webp');
				if(!in_array(strtolower($extension), $allowed_extensions)) {
					$extension = 'jpg';
				}
				
				// Generate unique filename
				$filename = $p_id . '_' . uniqid() . '_' . time() . '.' . $extension;
				$upload_dir = wp_upload_dir();
				
				// Check for upload errors
				if($upload_dir['error']) {
					error_log('Upload directory error: ' . $upload_dir['error']);
					continue;
				}
				
				// Check folder permission and define file location
				if(wp_mkdir_p($upload_dir['path'])) {
					$file = $upload_dir['path'] . '/' . $filename;
				} else {
					$file = $upload_dir['basedir'] . '/' . $filename;
				}
				
				// Download image with timeout and error handling
				$image_data = @file_get_contents($image_path, false, stream_context_create(array(
					'http' => array(
						'timeout' => 30,
						'user_agent' => 'Pixxi Property Importer by Dev Mizan'
					)
				)));
				
				if($image_data === false) {
					error_log('Failed to download image: ' . $image_path);
					continue;
				}
				
				// Save the file
				if(file_put_contents($file, $image_data)) {
					// Check image file type
					$wp_filetype = wp_check_filetype($filename, null);
					
					// Validate file is actually an image
					if(empty($wp_filetype['type']) || strpos($wp_filetype['type'], 'image/') !== 0) {
						@unlink($file); // Delete invalid file
						error_log('Invalid image file type: ' . $filename);
						continue;
					}
					
					// Set attachment data
					$attachment = array(
						'post_mime_type' => $wp_filetype['type'],
						'post_title'     => sanitize_file_name($filename),
						'post_content'   => '',
						'post_status'    => 'inherit'
					);
					
					// Create the attachment
					$attach_id = wp_insert_attachment($attachment, $file, $post_id);
					
					if(is_wp_error($attach_id)) {
						@unlink($file);
						error_log('Failed to create attachment: ' . $attach_id->get_error_message());
						continue;
					}
					
					// Include image.php
					require_once(ABSPATH . 'wp-admin/includes/image.php');
					
					// Define attachment metadata
					$attach_data = wp_generate_attachment_metadata($attach_id, $file);
					
					// Assign metadata to attachment
					wp_update_attachment_metadata($attach_id, $attach_data);
					
					if($_x == 0){
						set_post_thumbnail( $post_id, $attach_id );
					} else {
						$attach_id_array .= ','.$attach_id;
					}
					
					add_post_meta( $post_id, 'fave_property_images', $attach_id, false);
					$_x++;
				} else {
					error_log('Failed to save image file: ' . $file);
				}
			} catch (Exception $e) {
				error_log('Error attaching image: ' . $e->getMessage());
				continue;
			}
		}
		
		return $attach_id_array;
	}
	function pixxi_create_propertyterm($post_id, $pbrand, $taxonomy){
		if(empty($pbrand) || empty($taxonomy)) {
			return;
		}
		
		// Ensure value is a string and trim whitespace
		$pbrand = is_string($pbrand) ? trim($pbrand) : (string)$pbrand;
		if(empty($pbrand)) {
			return;
		}
		
		$brand_term = term_exists( $pbrand, $taxonomy);
		if ( $brand_term == 0 && $brand_term == null ) {
			$brand_term = wp_insert_term(
				$pbrand,
				$taxonomy
			);
		}
		
		// Check if wp_insert_term returned a WP_Error
		if (is_wp_error($brand_term)) {
			error_log('Error creating term "' . $pbrand . '" in taxonomy "' . $taxonomy . '": ' . $brand_term->get_error_message());
			return;
		}
		
		// Check if we have a valid term_id
		if(isset($brand_term['term_id'])) {
			$result = wp_set_object_terms( $post_id, (int)$brand_term['term_id'], $taxonomy );
			if(is_wp_error($result)) {
				error_log('Error setting term "' . $pbrand . '" (ID: ' . $brand_term['term_id'] . ') for taxonomy "' . $taxonomy . '" on post ' . $post_id . ': ' . $result->get_error_message());
			}
		} else {
			error_log('Invalid term data for "' . $pbrand . '" in taxonomy "' . $taxonomy . '"');
			return;
		}
	}
	function pixxi_create_propertyfeatures($post_id, $feat_arr, $taxonomy){
		if(empty($feat_arr) || !is_array($feat_arr)) {
			return;
		}
		
		$term_arr = [];
		foreach($feat_arr as $feat){
			if(empty($feat)) {
				continue;
			}
			
			//pterm object
			$feat_term = term_exists( $feat, $taxonomy);
			//check if pterm does not exist
			if ( $feat_term == 0 && $feat_term == null ) {
				//pterm object
				$feat_term = wp_insert_term(
					$feat,
					$taxonomy
				);
			}
			
			if(isset($feat_term['term_id'])) {
				$term_arr[] = (int)$feat_term['term_id'];
			}
		}
		
		if(!empty($term_arr)) {
			wp_set_object_terms( $post_id, $term_arr, $taxonomy );
		}
	}
	function upload_image_from_url($image_url, $post_id) {
		require_once(ABSPATH . 'wp-admin/includes/file.php');
		require_once(ABSPATH . 'wp-admin/includes/media.php');
		require_once(ABSPATH . 'wp-admin/includes/image.php');
		$tmp = download_url($image_url);
		if (is_wp_error($tmp)) {
			error_log('Error downloading image: ' . $tmp->get_error_message());
			return false;
		}
		$file_array = array(
			'name'     => basename($image_url),
			'tmp_name' => $tmp
		);
		$attachment_id = media_handle_sideload($file_array, $post_id);
		if (is_wp_error($attachment_id)) {
			@unlink($tmp);
			error_log('Error uploading image: ' . $attachment_id->get_error_message());
			return false;
		}
		return $attachment_id;
	}
	function create_houzez_agent_from_array($agent_data) {
		if (empty($agent_data['email'])) {
			//error_log('Agent email missing, cannot proceed.');
			return;
		}
		$agent_email = sanitize_email($agent_data['email']);
		// 1. Check if agent already exists by email
		$existing_agents = get_posts([
			'post_type'  => 'houzez_agent',
			'meta_key'   => 'fave_agent_email',
			'meta_value' => $agent_email,
			'posts_per_page' => 1,
			'fields' => 'ids',
		]);
		if (!empty($existing_agents)) {
			$existing_id = $existing_agents[0];
			//error_log("Agent already exists (ID: $existing_id)");
			return $existing_id;
		}
		// 2. Create new agent post
		$post_id = wp_insert_post([
			'post_title'   => sanitize_text_field($agent_data['name']),
			'post_type'    => 'houzez_agent',
			'post_status'  => 'publish',
		]);
		if (is_wp_error($post_id)) {
			//error_log("Failed to create agent: " . $post_id->get_error_message());
			return;
		}
		// 3. Save metadata
		update_post_meta($post_id, 'fave_agent_email', $agent_email);
		update_post_meta($post_id, 'fave_agent_mobile', sanitize_text_field($agent_data['phone']));
		update_post_meta($post_id, 'fave_agent_whatsapp', sanitize_text_field($agent_data['phone']));
		update_post_meta($post_id, 'fave_agent_position', 'Agent');
		update_post_meta($post_id, 'fave_agent_company', $agent_data['subCompanyConfigId']);
		// 4. Set featured image
		if (!empty($agent_data['avatar'])) {
			$attachment_id = $this->upload_image_from_url($agent_data['avatar'], $post_id);
			if ($attachment_id) {
				set_post_thumbnail($post_id, $attachment_id);
			}
		}
		//error_log("New agent created (ID: $post_id)");
		return $post_id;
	}
	function check_if_mvalue_exist($mvalue){
		global $wpdb;
		$results = $wpdb->get_results( "select post_id, meta_key from $wpdb->postmeta where meta_value = '$mvalue'", ARRAY_A );
		if(empty($results)){
			return true;
		}else{
			return false;
		}
	}
	function get_field_value($data, $field_path, $default = '') {
		// Handle dot notation for nested fields (e.g., rentParam.washRoom)
		$keys = explode('.', $field_path);
		$value = $data;
		
		foreach($keys as $key) {
			if(is_array($value) && array_key_exists($key, $value)) {
				$value = $value[$key];
			} elseif(is_object($value) && isset($value->$key)) {
				$value = $value->$key;
			} else {
				return $default;
			}
		}
		
		// Return value if it's an array (even if empty, let the caller decide)
		// For arrays, return as-is; for other values, return only if not empty
		if(is_array($value)) {
			return $value;
		}
		
		// For non-array values, return value if not empty, otherwise default
		// Use strict comparison to distinguish between null and empty string
		if($value === null) {
			return $default;
		}
		
		return !empty($value) ? $value : $default;
	}
	
	/**
	 * Extract coordinates from API data - checks multiple possible field names and formats
	 * 
	 * @param array $pd Property data from API
	 * @return array|null Returns array with 'lat' and 'lng' keys, or null if not found
	 */
	function extract_coordinates_from_api($pd) {
		// #region agent log
		$log_file = __DIR__ . '/../.cursor/debug.log';
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H1',
			'location' => 'extract_coordinates_from_api:entry',
			'message' => 'Function entry - checking for coordinates',
			'data' => array(
				'pd_is_array' => is_array($pd),
				'pd_is_empty' => empty($pd),
				'has_position' => isset($pd['position']),
				'position_value' => isset($pd['position']) ? (is_string($pd['position']) ? $pd['position'] : gettype($pd['position'])) : 'NOT_SET',
				'has_coordinates' => isset($pd['coordinates']),
				'has_latitude' => isset($pd['latitude']),
				'has_longitude' => isset($pd['longitude']),
				'has_lat' => isset($pd['lat']),
				'has_lng' => isset($pd['lng']),
				'has_location' => isset($pd['location']),
				'has_geo' => isset($pd['geo']),
				'top_level_keys' => is_array($pd) ? array_slice(array_keys($pd), 0, 20) : array()
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		if(empty($pd) || !is_array($pd)) {
			// #region agent log
			$log_entry = json_encode(array(
				'sessionId' => 'debug-session',
				'runId' => 'run1',
				'hypothesisId' => 'H1',
				'location' => 'extract_coordinates_from_api:early_return',
				'message' => 'Early return - empty or not array',
				'data' => array(),
				'timestamp' => time() * 1000
			)) . "\n";
			@file_put_contents($log_file, $log_entry, FILE_APPEND);
			// #endregion
			return null;
		}
		
		$lat = null;
		$lng = null;
		
		// Try position field (comma-separated string: "lat,lng" or "lat,lng,zoom")
		if(!empty($pd['position']) && is_string($pd['position'])) {
			$position_parts = explode(',', $pd['position']);
			// #region agent log
			$log_entry = json_encode(array(
				'sessionId' => 'debug-session',
				'runId' => 'run1',
				'hypothesisId' => 'H1',
				'location' => 'extract_coordinates_from_api:position_check',
				'message' => 'Checking position field',
				'data' => array(
					'position_string' => $pd['position'],
					'position_parts_count' => count($position_parts),
					'position_parts' => $position_parts
				),
				'timestamp' => time() * 1000
			)) . "\n";
			@file_put_contents($log_file, $log_entry, FILE_APPEND);
			// #endregion
			if(count($position_parts) >= 2) {
				$lat = trim($position_parts[0]);
				$lng = trim($position_parts[1]);
				if(is_numeric($lat) && is_numeric($lng)) {
					// #region agent log
					$log_entry = json_encode(array(
						'sessionId' => 'debug-session',
						'runId' => 'run1',
						'hypothesisId' => 'H1',
						'location' => 'extract_coordinates_from_api:found_position',
						'message' => 'Found coordinates in position field',
						'data' => array('lat' => $lat, 'lng' => $lng),
						'timestamp' => time() * 1000
					)) . "\n";
					@file_put_contents($log_file, $log_entry, FILE_APPEND);
					// #endregion
					return array('lat' => $lat, 'lng' => $lng);
				}
			}
		}
		
		// Try coordinates array/object
		if(!empty($pd['coordinates'])) {
			if(is_array($pd['coordinates'])) {
				// Array format: ['lat' => x, 'lng' => y] or [lat, lng] or ['latitude' => x, 'longitude' => y]
				if(isset($pd['coordinates']['lat']) && isset($pd['coordinates']['lng'])) {
					$lat = $pd['coordinates']['lat'];
					$lng = $pd['coordinates']['lng'];
				} elseif(isset($pd['coordinates']['latitude']) && isset($pd['coordinates']['longitude'])) {
					$lat = $pd['coordinates']['latitude'];
					$lng = $pd['coordinates']['longitude'];
				} elseif(isset($pd['coordinates'][0]) && isset($pd['coordinates'][1])) {
					// Array format: [lat, lng]
					$lat = $pd['coordinates'][0];
					$lng = $pd['coordinates'][1];
				}
			} elseif(is_object($pd['coordinates'])) {
				// Object format
				if(isset($pd['coordinates']->lat) && isset($pd['coordinates']->lng)) {
					$lat = $pd['coordinates']->lat;
					$lng = $pd['coordinates']->lng;
				} elseif(isset($pd['coordinates']->latitude) && isset($pd['coordinates']->longitude)) {
					$lat = $pd['coordinates']->latitude;
					$lng = $pd['coordinates']->longitude;
				}
			}
			if(is_numeric($lat) && is_numeric($lng)) {
				return array('lat' => $lat, 'lng' => $lng);
			}
		}
		
		// Try separate latitude and longitude fields
		if(isset($pd['latitude']) && isset($pd['longitude'])) {
			$lat = $pd['latitude'];
			$lng = $pd['longitude'];
		} elseif(isset($pd['lat']) && isset($pd['lng'])) {
			$lat = $pd['lat'];
			$lng = $pd['lng'];
		} elseif(isset($pd['geoLat']) && isset($pd['geoLng'])) {
			$lat = $pd['geoLat'];
			$lng = $pd['geoLng'];
		}
		
		if(is_numeric($lat) && is_numeric($lng)) {
			return array('lat' => $lat, 'lng' => $lng);
		}
		
		// Try nested location object/array
		if(!empty($pd['location'])) {
			$loc = $pd['location'];
			if(is_array($loc)) {
				if(isset($loc['lat']) && isset($loc['lng'])) {
					$lat = $loc['lat'];
					$lng = $loc['lng'];
				} elseif(isset($loc['latitude']) && isset($loc['longitude'])) {
					$lat = $loc['latitude'];
					$lng = $loc['longitude'];
				} elseif(isset($loc['coordinates']) && is_array($loc['coordinates'])) {
					if(isset($loc['coordinates'][0]) && isset($loc['coordinates'][1])) {
						$lat = $loc['coordinates'][0];
						$lng = $loc['coordinates'][1];
					}
				}
			} elseif(is_object($loc)) {
				if(isset($loc->lat) && isset($loc->lng)) {
					$lat = $loc->lat;
					$lng = $loc->lng;
				} elseif(isset($loc->latitude) && isset($loc->longitude)) {
					$lat = $loc->latitude;
					$lng = $loc->longitude;
				}
			}
			if(is_numeric($lat) && is_numeric($lng)) {
				return array('lat' => $lat, 'lng' => $lng);
			}
		}
		
		// Try geo object/array
		if(!empty($pd['geo'])) {
			$geo = $pd['geo'];
			if(is_array($geo)) {
				if(isset($geo['lat']) && isset($geo['lng'])) {
					$lat = $geo['lat'];
					$lng = $geo['lng'];
				} elseif(isset($geo['latitude']) && isset($geo['longitude'])) {
					$lat = $geo['latitude'];
					$lng = $geo['longitude'];
				}
			} elseif(is_object($geo)) {
				if(isset($geo->lat) && isset($geo->lng)) {
					$lat = $geo->lat;
					$lng = $geo->lng;
				} elseif(isset($geo->latitude) && isset($geo->longitude)) {
					$lat = $geo->latitude;
					$lng = $geo->longitude;
				}
			}
			if(is_numeric($lat) && is_numeric($lng)) {
				return array('lat' => $lat, 'lng' => $lng);
			}
		}
		
		// Try nested structures: newParam, rentParam, sellParam
		$nested_fields = array('newParam', 'rentParam', 'sellParam');
		foreach($nested_fields as $nested_field) {
			if(!empty($pd[$nested_field]) && (is_array($pd[$nested_field]) || is_object($pd[$nested_field]))) {
				$nested = $pd[$nested_field];
				// Check for position in nested structure
				if(is_array($nested)) {
					if(!empty($nested['position']) && is_string($nested['position'])) {
						$position_parts = explode(',', $nested['position']);
						if(count($position_parts) >= 2) {
							$lat = trim($position_parts[0]);
							$lng = trim($position_parts[1]);
							if(is_numeric($lat) && is_numeric($lng)) {
								// #region agent log
								$log_entry = json_encode(array(
									'sessionId' => 'debug-session',
									'runId' => 'run1',
									'hypothesisId' => 'H1',
									'location' => 'extract_coordinates_from_api:found_nested_position',
									'message' => 'Found coordinates in nested position field',
									'data' => array('nested_field' => $nested_field, 'lat' => $lat, 'lng' => $lng),
									'timestamp' => time() * 1000
								)) . "\n";
								@file_put_contents($log_file, $log_entry, FILE_APPEND);
								// #endregion
								return array('lat' => $lat, 'lng' => $lng);
							}
						}
					}
					// Check for lat/lng directly in nested structure
					if(isset($nested['lat']) && isset($nested['lng'])) {
						$lat = $nested['lat'];
						$lng = $nested['lng'];
						if(is_numeric($lat) && is_numeric($lng)) {
							// #region agent log
							$log_entry = json_encode(array(
								'sessionId' => 'debug-session',
								'runId' => 'run1',
								'hypothesisId' => 'H1',
								'location' => 'extract_coordinates_from_api:found_nested_lat_lng',
								'message' => 'Found coordinates in nested lat/lng',
								'data' => array('nested_field' => $nested_field, 'lat' => $lat, 'lng' => $lng),
								'timestamp' => time() * 1000
							)) . "\n";
							@file_put_contents($log_file, $log_entry, FILE_APPEND);
							// #endregion
							return array('lat' => $lat, 'lng' => $lng);
						}
					}
				} elseif(is_object($nested)) {
					if(!empty($nested->position) && is_string($nested->position)) {
						$position_parts = explode(',', $nested->position);
						if(count($position_parts) >= 2) {
							$lat = trim($position_parts[0]);
							$lng = trim($position_parts[1]);
							if(is_numeric($lat) && is_numeric($lng)) {
								return array('lat' => $lat, 'lng' => $lng);
							}
						}
					}
					if(isset($nested->lat) && isset($nested->lng)) {
						$lat = $nested->lat;
						$lng = $nested->lng;
						if(is_numeric($lat) && is_numeric($lng)) {
							return array('lat' => $lat, 'lng' => $lng);
						}
					}
				}
			}
		}
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H1',
			'location' => 'extract_coordinates_from_api:not_found',
			'message' => 'No coordinates found in any field',
			'data' => array(
				'checked_nested' => $nested_fields,
				'has_newParam' => isset($pd['newParam']),
				'has_rentParam' => isset($pd['rentParam']),
				'has_sellParam' => isset($pd['sellParam'])
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		return null;
	}
	
	/**
	 * Set map data for a property in Houzez format
	 * 
	 * @param int $post_id WordPress post ID
	 * @param array $pd Property data from API
	 * @param array $coords Coordinates array with 'lat' and 'lng' keys
	 */
	function set_houzez_map_data($post_id, $pd, $coords) {
		// #region agent log
		$log_file = __DIR__ . '/../.cursor/debug.log';
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H2',
			'location' => 'set_houzez_map_data:entry',
			'message' => 'Function entry',
			'data' => array(
				'post_id' => $post_id,
				'coords_provided' => !empty($coords),
				'coords_has_lat' => isset($coords['lat']),
				'coords_has_lng' => isset($coords['lng']),
				'coords_value' => $coords
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		if(empty($coords) || !isset($coords['lat']) || !isset($coords['lng'])) {
			// #region agent log
			$log_entry = json_encode(array(
				'sessionId' => 'debug-session',
				'runId' => 'run1',
				'hypothesisId' => 'H2',
				'location' => 'set_houzez_map_data:invalid_coords',
				'message' => 'Invalid coordinates provided',
				'data' => array('coords' => $coords),
				'timestamp' => time() * 1000
			)) . "\n";
			@file_put_contents($log_file, $log_entry, FILE_APPEND);
			// #endregion
			return false;
		}
		
		$lat = trim($coords['lat']);
		$lng = trim($coords['lng']);
		
		if(!is_numeric($lat) || !is_numeric($lng)) {
			// #region agent log
			$log_entry = json_encode(array(
				'sessionId' => 'debug-session',
				'runId' => 'run1',
				'hypothesisId' => 'H2',
				'location' => 'set_houzez_map_data:non_numeric',
				'message' => 'Coordinates not numeric',
				'data' => array('lat' => $lat, 'lng' => $lng, 'lat_type' => gettype($lat), 'lng_type' => gettype($lng)),
				'timestamp' => time() * 1000
			)) . "\n";
			@file_put_contents($log_file, $log_entry, FILE_APPEND);
			// #endregion
			return false;
		}
		
		// Set fave_property_map to 1 (Show) - Houzez uses 1 for show, 0 for hide
		$map_result = update_post_meta($post_id, 'fave_property_map', 1);
		
		// Set fave_property_location in format "lat,lng,zoom" (default zoom 12)
		$zoom = 12; // Default zoom level
		$location_value = $lat . ',' . $lng . ',' . $zoom;
		$location_result = update_post_meta($post_id, 'fave_property_location', $location_value);
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H3',
			'location' => 'set_houzez_map_data:meta_saved',
			'message' => 'Map meta saved',
			'data' => array(
				'post_id' => $post_id,
				'map_result' => $map_result,
				'location_result' => $location_result,
				'location_value' => $location_value,
				'map_value_read_back' => get_post_meta($post_id, 'fave_property_map', true),
				'location_value_read_back' => get_post_meta($post_id, 'fave_property_location', true)
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		// Build map address from API data - Format: community, area, city
		$address_parts = array();
		
		// 1. Community (first priority)
		if(!empty($pd['community'])) {
			// Handle different formats: string, object with name, array with name
			if(is_array($pd['community']) && isset($pd['community']['name'])) {
				$address_parts[] = $pd['community']['name'];
			} elseif(is_object($pd['community']) && isset($pd['community']->name)) {
				$address_parts[] = $pd['community']->name;
			} elseif(is_string($pd['community'])) {
				$address_parts[] = $pd['community'];
			}
		} elseif(!empty($pd['communityName'])) {
			$address_parts[] = $pd['communityName'];
		}
		
		// 2. Area/Region (second priority)
		if(!empty($pd['region'])) {
			$address_parts[] = $pd['region'];
		}
		
		// 3. City (third priority)
		if(!empty($pd['city'])) {
			$address_parts[] = $pd['city'];
		} elseif(!empty($pd['cityName'])) {
			$address_parts[] = $pd['cityName'];
		}
		
		// Additional address fields (if needed)
		if(!empty($pd['address'])) {
			$address_parts[] = $pd['address'];
		}
		if(!empty($pd['street'])) {
			$address_parts[] = $pd['street'];
		}
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H4',
			'location' => 'set_houzez_map_data:address_parts',
			'message' => 'Address parts from API',
			'data' => array(
				'post_id' => $post_id,
				'address_parts_count' => count($address_parts),
				'address_parts' => $address_parts,
				'has_address' => !empty($pd['address']),
				'has_street' => !empty($pd['street']),
				'has_region' => !empty($pd['region']),
				'has_city' => !empty($pd['city']),
				'has_communityName' => !empty($pd['communityName']),
				'address_value' => isset($pd['address']) ? $pd['address'] : 'NOT_SET',
				'region_value' => isset($pd['region']) ? $pd['region'] : 'NOT_SET',
				'city_value' => isset($pd['city']) ? $pd['city'] : 'NOT_SET'
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		// Fallback to meta fields if API data not available - Format: community, area, city
		if(empty($address_parts)) {
			// Try to get community from taxonomy or meta
			$community_terms = wp_get_post_terms($post_id, 'community', array('fields' => 'names'));
			if(!empty($community_terms) && !is_wp_error($community_terms)) {
				$address_parts[] = $community_terms[0];
			}
			
			$property_area = get_post_meta($post_id, 'fave_property_area', true);
			if(!empty($property_area)) {
				$address_parts[] = $property_area;
			}
			
			$property_city = get_post_meta($post_id, 'fave_property_city', true);
			if(!empty($property_city)) {
				$address_parts[] = $property_city;
			}
		}
		
		// Set the address
		$address_saved = false;
		if(!empty($address_parts)) {
			$full_address = implode(', ', $address_parts);
			$address_result = update_post_meta($post_id, 'fave_property_map_address', $full_address);
			$address_saved = true;
		} else {
			// Build fallback address in format: community, area, city
			$fallback_parts = array();
			
			// Try community
			if(!empty($pd['community'])) {
				if(is_array($pd['community']) && isset($pd['community']['name'])) {
					$fallback_parts[] = $pd['community']['name'];
				} elseif(is_object($pd['community']) && isset($pd['community']->name)) {
					$fallback_parts[] = $pd['community']->name;
				} elseif(is_string($pd['community'])) {
					$fallback_parts[] = $pd['community'];
				}
			} elseif(!empty($pd['communityName'])) {
				$fallback_parts[] = $pd['communityName'];
			}
			
			// Try region/area
			if(!empty($pd['region'])) {
				$fallback_parts[] = $pd['region'];
			}
			
			// Try city
			if(!empty($pd['city'])) {
				$fallback_parts[] = $pd['city'];
			} elseif(!empty($pd['cityName'])) {
				$fallback_parts[] = $pd['cityName'];
			}
			
			if(!empty($fallback_parts)) {
				$full_address = implode(', ', $fallback_parts);
				$address_result = update_post_meta($post_id, 'fave_property_map_address', $full_address);
				$address_saved = true;
			} elseif(!empty($pd['region'])) {
				// At minimum, use region if available
				$address_result = update_post_meta($post_id, 'fave_property_map_address', $pd['region']);
				$full_address = $pd['region'];
				$address_saved = true;
			} elseif(!empty($pd['city'])) {
				// Fallback to city
				$address_result = update_post_meta($post_id, 'fave_property_map_address', $pd['city']);
				$full_address = $pd['city'];
				$address_saved = true;
			}
		}
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H3',
			'location' => 'set_houzez_map_data:address_saved',
			'message' => 'Address meta saved',
			'data' => array(
				'post_id' => $post_id,
				'address_saved' => $address_saved,
				'address_result' => isset($address_result) ? $address_result : 'NOT_SET',
				'full_address' => isset($full_address) ? $full_address : 'NOT_SET',
				'address_value_read_back' => get_post_meta($post_id, 'fave_property_map_address', true)
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		// Also update ACF if active
		if(function_exists('update_field')) {
			update_field('fave_property_map', 1, $post_id);
			update_field('fave_property_location', $lat . ',' . $lng . ',' . $zoom, $post_id);
			$final_address = isset($full_address) && !empty($full_address) ? $full_address : (!empty($pd['region']) ? $pd['region'] : (!empty($pd['city']) ? $pd['city'] : ''));
			if(!empty($final_address)) {
				update_field('fave_property_map_address', $final_address, $post_id);
			}
		}
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H3',
			'location' => 'set_houzez_map_data:final_read_back',
			'message' => 'Final values read back after all updates',
			'data' => array(
				'post_id' => $post_id,
				'fave_property_map' => get_post_meta($post_id, 'fave_property_map', true),
				'fave_property_location' => get_post_meta($post_id, 'fave_property_location', true),
				'fave_property_map_address' => get_post_meta($post_id, 'fave_property_map_address', true)
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		return true;
	}
	
	function get_mapped_field_value($post_id, $wp_field, $pd, $default_api_field = '') {
		// Get saved mappings
		$opt = get_option('pixxi_opt_val');
		if(!$opt) {
			$opt = get_option('lw_opt_val');
			if($opt) {
				update_option('pixxi_opt_val', $opt);
			}
		}
		$pixxi_option = json_decode($opt, true);
		$mappings = isset($pixxi_option['pixxi_field_mappings']) && is_array($pixxi_option['pixxi_field_mappings']) ? $pixxi_option['pixxi_field_mappings'] : array();
		
		// Check if there's a custom mapping for this field
		foreach($mappings as $mapping) {
			if(isset($mapping['wp_field']) && $mapping['wp_field'] === $wp_field && !empty($mapping['api_field'])) {
				$api_field = $mapping['api_field'];
				$default_value = isset($mapping['default_value']) ? $mapping['default_value'] : '';
				return $this->get_field_value($pd, $api_field, $default_value);
			}
		}
		
		// If no mapping found, use default API field name
		if(!empty($default_api_field)) {
			return $this->get_field_value($pd, $default_api_field, '');
		}
		
		return '';
	}
	
	function apply_field_mappings($post_id, $pd) {
		// #region agent log
		$log_file = __DIR__ . '/../.cursor/debug.log';
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'A,B',
			'location' => 'apply_field_mappings:entry',
			'message' => 'Function entry - checking listingType and taxonomy',
			'data' => array(
				'post_id' => $post_id,
				'has_listingType' => isset($pd['listingType']),
				'listingType_value' => isset($pd['listingType']) ? $pd['listingType'] : 'NOT_SET',
				'has_newParam' => isset($pd['newParam']),
				'newParam_is_array' => isset($pd['newParam']) && is_array($pd['newParam']),
				'newParam_keys' => isset($pd['newParam']) && is_array($pd['newParam']) ? array_keys($pd['newParam']) : array()
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// Check taxonomy term for property_status
		$status_terms = wp_get_post_terms($post_id, 'property_status', array('fields' => 'all'));
		if(!empty($status_terms) && !is_wp_error($status_terms)) {
			$status_slugs = array();
			$status_names = array();
			foreach($status_terms as $term) {
				$status_slugs[] = $term->slug;
				$status_names[] = $term->name;
			}
			$log_entry = json_encode(array(
				'sessionId' => 'debug-session',
				'runId' => 'run1',
				'hypothesisId' => 'B',
				'location' => 'apply_field_mappings:taxonomy_check',
				'message' => 'Property status taxonomy terms',
				'data' => array(
					'post_id' => $post_id,
					'status_slugs' => $status_slugs,
					'status_names' => $status_names,
					'has_off_plan_slug' => in_array('off-plan', $status_slugs),
					'has_off_plan_name' => in_array('Off Plan', $status_names)
				),
				'timestamp' => time() * 1000
			)) . "\n";
			@file_put_contents($log_file, $log_entry, FILE_APPEND);
		}
		// #endregion
		// Log all available fields that might contain coordinates
		$coordinate_fields = array();
		$all_keys = array_keys($pd);
		foreach($all_keys as $key) {
			if(stripos($key, 'lat') !== false || stripos($key, 'lng') !== false || 
			   stripos($key, 'lon') !== false || stripos($key, 'position') !== false ||
			   stripos($key, 'coord') !== false || stripos($key, 'location') !== false ||
			   stripos($key, 'geo') !== false) {
				$coordinate_fields[$key] = is_array($pd[$key]) ? 'ARRAY' : (is_object($pd[$key]) ? 'OBJECT' : $pd[$key]);
			}
		}
		// Also check nested structures
		$nested_coords = array();
		if(isset($pd['location']) && is_array($pd['location'])) {
			$nested_coords['location'] = $pd['location'];
		}
		if(isset($pd['coordinates']) && is_array($pd['coordinates'])) {
			$nested_coords['coordinates'] = $pd['coordinates'];
		}
		if(isset($pd['geo']) && is_array($pd['geo'])) {
			$nested_coords['geo'] = $pd['geo'];
		}
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run2',
			'hypothesisId' => 'H1',
			'location' => 'apply_field_mappings:entry',
			'message' => 'Function entry - checking for coordinate fields',
			'data' => array(
				'post_id' => $post_id,
				'has_position' => isset($pd['position']),
				'position_value' => isset($pd['position']) ? (is_array($pd['position']) ? 'ARRAY' : (is_object($pd['position']) ? 'OBJECT' : $pd['position'])) : 'NOT_SET',
				'coordinate_fields_found' => $coordinate_fields,
				'nested_coords' => $nested_coords,
				'all_top_level_keys' => array_slice($all_keys, 0, 40) // First 40 keys to see structure
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		// Hard-coded default field mappings for Houzez
		$default_mappings = array(
			'post_title' => 'title',
			'post_content' => 'description',
			'fave_property_id' => 'propertyId',
			'fave_property_status' => 'Status',
			'fave_property_type' => 'propertyType',
			'fave_property_images' => 'photos',
			'fave_property_address' => 'region',
			'fave_property_city' => 'city',
			'fave_property_area' => 'region',
			'houzez_geolocation_lat' => 'position',
			'fave_latitude' => 'position',
			'fave_longitude' => 'position',
			'houzez_geolocation_long' => 'position',
			'fave_property_bedrooms' => 'bedRooms', // Try bedRooms first, fallback to bedRoomNum and bedrooms
			'fave_property_bathrooms' => 'rentParam.bathrooms', // Try rentParam.bathrooms first, fallback to sellParam.bathrooms and washRoom fields
			'fave_property_garage' => 'newParam.parking',
			'fave_property_size' => 'size',
			'_fave_property_land' => 'plotSize',
			'fave_property_year' => 'rentParam.buildYear',
			'fave_parking' => 'rentParam.parking',
			'fave_ac-charge' => 'rentParam.acFee',
			'fave_property_price' => 'price',
			'fave_developer' => 'developer',
			'fave_developer-logo' => 'developerLogo',
			'fave_permit_number' => 'permitNumber',
			'fave_permit_qr_code' => 'permitQRCode',
			'fave_permit_url' => 'permitUrl',
			'fave_agent_email' => 'agent.email',
			'fave_agent_mobile' => 'agent.phone',
			'fave_agent_whatsapp' => 'agent.phone',
			'fave_agent_photo' => 'agent.originalAvatar',
			'fave_additional_features' => 'amenities'
		);
		
		// Apply each mapping
		foreach($default_mappings as $wp_field => $api_field) {
			if(empty($api_field)) {
				continue;
			}
			
			// Skip fave_property_images - photos are handled separately by attach_product_thumbnail()
			if($wp_field === 'fave_property_images') {
				continue;
			}
			
			// Check if property is Off Plan (listingType == 'NEW')
			$is_off_plan = isset($pd['listingType']) && $pd['listingType'] == 'NEW';
			
			// #region agent log
			if($wp_field === 'fave_property_bedrooms' || $wp_field === 'fave_property_area') {
				$log_entry = json_encode(array(
					'sessionId' => 'debug-session',
					'runId' => 'run1',
					'hypothesisId' => 'A,B',
					'location' => 'apply_field_mappings:off_plan_check',
					'message' => 'Checking off-plan status for field mapping',
					'data' => array(
						'post_id' => $post_id,
						'wp_field' => $wp_field,
						'is_off_plan' => $is_off_plan,
						'listingType' => isset($pd['listingType']) ? $pd['listingType'] : 'NOT_SET',
						'skipping_field' => $is_off_plan && ($wp_field === 'fave_property_bedrooms' || $wp_field === 'fave_property_area')
					),
					'timestamp' => time() * 1000
				)) . "\n";
				@file_put_contents($log_file, $log_entry, FILE_APPEND);
			}
			// #endregion
			
			// Skip bedroom and area mapping for Off Plan properties - they will be handled separately with ranges
			if($is_off_plan && ($wp_field === 'fave_property_bedrooms' || $wp_field === 'fave_property_area')) {
				continue;
			}
			
			// Get value from API data
			$value = $this->get_field_value($pd, $api_field, '');
			
			// Special handling for bedrooms - try multiple field names
			if($wp_field === 'fave_property_bedrooms' && empty($value)) {
				// Try alternative field names as fallback (default mapping already tries bedRooms)
				if(isset($pd['bedRoomNum']) && !empty($pd['bedRoomNum'])) {
					$value = $pd['bedRoomNum'];
				} elseif(isset($pd['bedrooms']) && !empty($pd['bedrooms'])) {
					$value = $pd['bedrooms'];
				}
			}
			
			// Special handling for bathrooms - try multiple field names
			if($wp_field === 'fave_property_bathrooms') {
				// If primary mapping (rentParam.bathrooms) returned empty, try sellParam.bathrooms first
				if(empty($value) && isset($pd['sellParam']['bathrooms']) && !empty($pd['sellParam']['bathrooms'])) {
					$value = $pd['sellParam']['bathrooms'];
				}
				// If still empty, try washRoom fields for backward compatibility
				if(empty($value)) {
					if(isset($pd['rentParam']['washRoom']) && !empty($pd['rentParam']['washRoom'])) {
						$value = $pd['rentParam']['washRoom'];
					} elseif(isset($pd['sellParam']['washRoom']) && !empty($pd['sellParam']['washRoom'])) {
						$value = $pd['sellParam']['washRoom'];
					} elseif(isset($pd['bathrooms']) && !empty($pd['bathrooms'])) {
						$value = $pd['bathrooms'];
					} elseif(isset($pd['washRoom']) && !empty($pd['washRoom'])) {
						$value = $pd['washRoom'];
					}
				}
			}
			
			$original_value = $value; // Keep original for ACF
			
			// Handle position field - extract lat/lng from comma-separated string
			if($api_field === 'position' && is_string($value) && !empty($value)) {
				$position_parts = explode(',', $value);
				if(count($position_parts) >= 2) {
					$lat = trim($position_parts[0]);
					$lng = trim($position_parts[1]);
					if(is_numeric($lat) && is_numeric($lng)) {
						// Set latitude for fields that need it
						if($wp_field === 'houzez_geolocation_lat' || $wp_field === 'fave_latitude') {
							$value = $lat;
						}
						// Set longitude for fields that need it
						if($wp_field === 'houzez_geolocation_long' || $wp_field === 'fave_longitude') {
							$value = $lng;
						}
						
						// Store coordinates for later map setup (only once when processing lat)
						if($wp_field === 'houzez_geolocation_lat') {
							// Store coordinates and flag to enable map later
							update_post_meta($post_id, '_pixxi_has_coordinates', true);
							update_post_meta($post_id, '_pixxi_map_lat', $lat);
							update_post_meta($post_id, '_pixxi_map_lng', $lng);
							// Store API data for address building
							update_post_meta($post_id, '_pixxi_map_pd', $pd);
							update_post_meta($post_id, '_pixxi_map_pd', $pd); // Store API data for address
						}
					}
				}
			}
			
			// Handle arrays - convert to JSON string for storage to avoid "Array to string conversion" warnings
			if(is_array($value)) {
				// Convert arrays to JSON string for post meta storage
				// This prevents "Array to string conversion" warnings
				$value = json_encode($value);
			}
			
			// Update post meta (works with ACF)
			// Only update if value is not empty and is scalar (string, int, float, bool)
			if($value !== '' && $value !== null && $value !== false && is_scalar($value)) {
				update_post_meta($post_id, $wp_field, $value);
				// Also update ACF field if ACF is active
				// ACF can handle arrays, so pass original value
				if(function_exists('update_field')) {
					update_field($wp_field, $original_value, $post_id);
				}
			} else {
				// Clear the field if no value
				delete_post_meta($post_id, $wp_field);
				if(function_exists('update_field')) {
					update_field($wp_field, '', $post_id);
				}
			}
		}
		
		// Special handling for off-plan properties (newParam)
		// Only apply range logic for properties with Off Plan status (listingType == 'NEW')
		$is_off_plan = isset($pd['listingType']) && $pd['listingType'] == 'NEW';
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'A,B,C,D',
			'location' => 'apply_field_mappings:off_plan_range_logic',
			'message' => 'Off-plan range logic entry',
			'data' => array(
				'post_id' => $post_id,
				'is_off_plan' => $is_off_plan,
				'listingType' => isset($pd['listingType']) ? $pd['listingType'] : 'NOT_SET',
				'has_newParam' => isset($pd['newParam']),
				'newParam_is_array' => isset($pd['newParam']) && is_array($pd['newParam']),
				'newParam_not_empty' => isset($pd['newParam']) && !empty($pd['newParam']),
				'condition_passes' => $is_off_plan && !empty($pd['newParam']) && is_array($pd['newParam'])
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		if($is_off_plan && !empty($pd['newParam']) && is_array($pd['newParam'])) {
			$newParam = $pd['newParam'];
			
			// #region agent log
			$log_entry = json_encode(array(
				'sessionId' => 'debug-session',
				'runId' => 'run1',
				'hypothesisId' => 'C',
				'location' => 'apply_field_mappings:newParam_data',
				'message' => 'newParam data values',
				'data' => array(
					'post_id' => $post_id,
					'has_bedroomMin' => isset($newParam['bedroomMin']),
					'bedroomMin_value' => isset($newParam['bedroomMin']) ? $newParam['bedroomMin'] : 'NOT_SET',
					'has_bedroomMax' => isset($newParam['bedroomMax']),
					'bedroomMax_value' => isset($newParam['bedroomMax']) ? $newParam['bedroomMax'] : 'NOT_SET',
					'has_minSize' => isset($newParam['minSize']),
					'minSize_value' => isset($newParam['minSize']) ? $newParam['minSize'] : 'NOT_SET',
					'has_maxSize' => isset($newParam['maxSize']),
					'maxSize_value' => isset($newParam['maxSize']) ? $newParam['maxSize'] : 'NOT_SET'
				),
				'timestamp' => time() * 1000
			)) . "\n";
			@file_put_contents($log_file, $log_entry, FILE_APPEND);
			// #endregion
			
			// Handle bedroom range (bedroomMin to bedroomMax) -> both _fave_property_bedrooms and fave_property_bedrooms
			if(isset($newParam['bedroomMin']) && isset($newParam['bedroomMax'])) {
				$bedroomMin = $newParam['bedroomMin'];
				$bedroomMax = $newParam['bedroomMax'];
				
				// #region agent log
				$log_entry = json_encode(array(
					'sessionId' => 'debug-session',
					'runId' => 'run1',
					'hypothesisId' => 'C',
					'location' => 'apply_field_mappings:bedroom_range_before_validation',
					'message' => 'Bedroom range values before validation',
					'data' => array(
						'post_id' => $post_id,
						'bedroomMin_raw' => $bedroomMin,
						'bedroomMax_raw' => $bedroomMax,
						'bedroomMin_type' => gettype($bedroomMin),
						'bedroomMax_type' => gettype($bedroomMax),
						'bedroomMin_is_null' => $bedroomMin === null,
						'bedroomMax_is_null' => $bedroomMax === null,
						'bedroomMin_is_empty' => $bedroomMin === '',
						'bedroomMax_is_empty' => $bedroomMax === ''
					),
					'timestamp' => time() * 1000
				)) . "\n";
				@file_put_contents($log_file, $log_entry, FILE_APPEND);
				// #endregion
				
				// Skip if values are null or empty
				if($bedroomMin !== null && $bedroomMax !== null && 
				   $bedroomMin !== '' && $bedroomMax !== '') {
					// Convert to string and trim
					$bedroomMin = trim((string)$bedroomMin);
					$bedroomMax = trim((string)$bedroomMax);
					
					// #region agent log
					$log_entry = json_encode(array(
						'sessionId' => 'debug-session',
						'runId' => 'run1',
						'hypothesisId' => 'C',
						'location' => 'apply_field_mappings:bedroom_range_after_trim',
						'message' => 'Bedroom range values after trim',
						'data' => array(
							'post_id' => $post_id,
							'bedroomMin_trimmed' => $bedroomMin,
							'bedroomMax_trimmed' => $bedroomMax,
							'bedroomMin_empty_after_trim' => empty($bedroomMin),
							'bedroomMax_empty_after_trim' => empty($bedroomMax)
						),
						'timestamp' => time() * 1000
					)) . "\n";
					@file_put_contents($log_file, $log_entry, FILE_APPEND);
					// #endregion
					
					if(!empty($bedroomMin) && !empty($bedroomMax)) {
						$bedroomRange = $bedroomMin . '-' . $bedroomMax;
						
						// #region agent log
						$log_entry = json_encode(array(
							'sessionId' => 'debug-session',
							'runId' => 'run1',
							'hypothesisId' => 'E',
							'location' => 'apply_field_mappings:bedroom_range_set',
							'message' => 'Setting bedroom range meta',
							'data' => array(
								'post_id' => $post_id,
								'bedroomRange' => $bedroomRange,
								'before_meta_bedrooms' => get_post_meta($post_id, 'fave_property_bedrooms', true),
								'before_meta_bedrooms_underscore' => get_post_meta($post_id, '_fave_property_bedrooms', true)
							),
							'timestamp' => time() * 1000
						)) . "\n";
						@file_put_contents($log_file, $log_entry, FILE_APPEND);
						// #endregion
						
						// Set both underscore-prefixed and regular field names
						update_post_meta($post_id, '_fave_property_bedrooms', $bedroomRange);
						update_post_meta($post_id, 'fave_property_bedrooms', $bedroomRange);
						if(function_exists('update_field')) {
							update_field('_fave_property_bedrooms', $bedroomRange, $post_id);
							update_field('fave_property_bedrooms', $bedroomRange, $post_id);
						}
						
						// #region agent log
						$log_entry = json_encode(array(
							'sessionId' => 'debug-session',
							'runId' => 'run1',
							'hypothesisId' => 'E',
							'location' => 'apply_field_mappings:bedroom_range_after_set',
							'message' => 'Bedroom range meta after setting',
							'data' => array(
								'post_id' => $post_id,
								'after_meta_bedrooms' => get_post_meta($post_id, 'fave_property_bedrooms', true),
								'after_meta_bedrooms_underscore' => get_post_meta($post_id, '_fave_property_bedrooms', true)
							),
							'timestamp' => time() * 1000
						)) . "\n";
						@file_put_contents($log_file, $log_entry, FILE_APPEND);
						// #endregion
					} else {
						// #region agent log
						$log_entry = json_encode(array(
							'sessionId' => 'debug-session',
							'runId' => 'run1',
							'hypothesisId' => 'C',
							'location' => 'apply_field_mappings:bedroom_range_empty_after_trim',
							'message' => 'Bedroom range empty after trim - skipping',
							'data' => array(
								'post_id' => $post_id,
								'bedroomMin_trimmed' => $bedroomMin,
								'bedroomMax_trimmed' => $bedroomMax
							),
							'timestamp' => time() * 1000
						)) . "\n";
						@file_put_contents($log_file, $log_entry, FILE_APPEND);
						// #endregion
					}
				} else {
					// #region agent log
					$log_entry = json_encode(array(
						'sessionId' => 'debug-session',
						'runId' => 'run1',
						'hypothesisId' => 'C',
						'location' => 'apply_field_mappings:bedroom_range_validation_failed',
						'message' => 'Bedroom range validation failed - null or empty',
						'data' => array(
							'post_id' => $post_id,
							'bedroomMin' => $bedroomMin,
							'bedroomMax' => $bedroomMax
						),
						'timestamp' => time() * 1000
					)) . "\n";
					@file_put_contents($log_file, $log_entry, FILE_APPEND);
					// #endregion
				}
			} else {
				// #region agent log
				$log_entry = json_encode(array(
					'sessionId' => 'debug-session',
					'runId' => 'run1',
					'hypothesisId' => 'C',
					'location' => 'apply_field_mappings:bedroom_range_not_set',
					'message' => 'Bedroom min/max not set in newParam',
					'data' => array(
						'post_id' => $post_id,
						'has_bedroomMin' => isset($newParam['bedroomMin']),
						'has_bedroomMax' => isset($newParam['bedroomMax'])
					),
					'timestamp' => time() * 1000
				)) . "\n";
				@file_put_contents($log_file, $log_entry, FILE_APPEND);
				// #endregion
			}
			
			// Handle area size range (minSize to maxSize) -> fave_property_size
			if(isset($newParam['minSize']) && isset($newParam['maxSize'])) {
				$minSize = $newParam['minSize'];
				$maxSize = $newParam['maxSize'];
				
				// #region agent log
				$log_entry = json_encode(array(
					'sessionId' => 'debug-session',
					'runId' => 'run1',
					'hypothesisId' => 'C',
					'location' => 'apply_field_mappings:area_range_before_validation',
					'message' => 'Area range values before validation',
					'data' => array(
						'post_id' => $post_id,
						'minSize_raw' => $minSize,
						'maxSize_raw' => $maxSize,
						'minSize_type' => gettype($minSize),
						'maxSize_type' => gettype($maxSize),
						'minSize_is_null' => $minSize === null,
						'maxSize_is_null' => $maxSize === null,
						'minSize_is_empty' => $minSize === '',
						'maxSize_is_empty' => $maxSize === ''
					),
					'timestamp' => time() * 1000
				)) . "\n";
				@file_put_contents($log_file, $log_entry, FILE_APPEND);
				// #endregion
				
				// Skip if values are null or empty
				// #region agent log
				$log_entry = json_encode(array(
					'sessionId' => 'debug-session',
					'runId' => 'run1',
					'hypothesisId' => 'C',
					'location' => 'apply_field_mappings:area_range_validation_check',
					'message' => 'Area range validation conditions',
					'data' => array(
						'post_id' => $post_id,
						'minSize_not_null' => $minSize !== null,
						'maxSize_not_null' => $maxSize !== null,
						'minSize_not_empty_string' => $minSize !== '',
						'maxSize_not_empty_string' => $maxSize !== '',
						'validation_passes' => ($minSize !== null && $maxSize !== null && $minSize !== '' && $maxSize !== '')
					),
					'timestamp' => time() * 1000
				)) . "\n";
				@file_put_contents($log_file, $log_entry, FILE_APPEND);
				// #endregion
				
				if($minSize !== null && $maxSize !== null && 
				   $minSize !== '' && $maxSize !== '') {
					// Convert to string and trim
					$minSize = trim((string)$minSize);
					$maxSize = trim((string)$maxSize);
					
					// #region agent log
					$log_entry = json_encode(array(
						'sessionId' => 'debug-session',
						'runId' => 'run1',
						'hypothesisId' => 'C',
						'location' => 'apply_field_mappings:area_range_after_trim',
						'message' => 'Area range values after trim',
						'data' => array(
							'post_id' => $post_id,
							'minSize_trimmed' => $minSize,
							'maxSize_trimmed' => $maxSize,
							'minSize_empty_after_trim' => empty($minSize),
							'maxSize_empty_after_trim' => empty($maxSize),
							'minSize_is_numeric' => is_numeric($minSize),
							'maxSize_is_numeric' => is_numeric($maxSize)
						),
						'timestamp' => time() * 1000
					)) . "\n";
					@file_put_contents($log_file, $log_entry, FILE_APPEND);
					// #endregion
					
					if(!empty($minSize) && !empty($maxSize)) {
						$areaRange = $minSize . '-' . $maxSize;
						
						// #region agent log
						$log_entry = json_encode(array(
							'sessionId' => 'debug-session',
							'runId' => 'run1',
							'hypothesisId' => 'E',
							'location' => 'apply_field_mappings:area_range_set',
							'message' => 'Setting area range meta',
							'data' => array(
								'post_id' => $post_id,
								'areaRange' => $areaRange,
								'before_meta_size' => get_post_meta($post_id, 'fave_property_size', true)
							),
							'timestamp' => time() * 1000
						)) . "\n";
						@file_put_contents($log_file, $log_entry, FILE_APPEND);
						// #endregion
						
						// Delete existing meta to clear any array format
						delete_post_meta($post_id, 'fave_property_size');
						
						// Set fave_property_size as string
						update_post_meta($post_id, 'fave_property_size', $areaRange);
						
						// Update ACF field - ensure value is set as string, not array
						if(function_exists('update_field')) {
							// Clear ACF field by setting to empty first, then set the value
							update_field('fave_property_size', '', $post_id);
							
							// Set as string value (ACF will handle the conversion if needed)
							update_field('fave_property_size', $areaRange, $post_id);
						}
						
						// #region agent log
						$log_entry = json_encode(array(
							'sessionId' => 'debug-session',
							'runId' => 'run1',
							'hypothesisId' => 'E',
							'location' => 'apply_field_mappings:area_range_after_set',
							'message' => 'Area range meta after setting',
							'data' => array(
								'post_id' => $post_id,
								'after_meta_size' => get_post_meta($post_id, 'fave_property_size', true)
							),
							'timestamp' => time() * 1000
						)) . "\n";
						@file_put_contents($log_file, $log_entry, FILE_APPEND);
						// #endregion
					} else {
						// #region agent log
						$log_entry = json_encode(array(
							'sessionId' => 'debug-session',
							'runId' => 'run1',
							'hypothesisId' => 'C',
							'location' => 'apply_field_mappings:area_range_empty_after_trim',
							'message' => 'Area range empty after trim - skipping',
							'data' => array(
								'post_id' => $post_id,
								'minSize_trimmed' => $minSize,
								'maxSize_trimmed' => $maxSize
							),
							'timestamp' => time() * 1000
						)) . "\n";
						@file_put_contents($log_file, $log_entry, FILE_APPEND);
						// #endregion
					}
				} else {
					// #region agent log
					$log_entry = json_encode(array(
						'sessionId' => 'debug-session',
						'runId' => 'run1',
						'hypothesisId' => 'C',
						'location' => 'apply_field_mappings:area_range_validation_failed',
						'message' => 'Area range validation failed - null or empty',
						'data' => array(
							'post_id' => $post_id,
							'minSize' => $minSize,
							'maxSize' => $maxSize
						),
						'timestamp' => time() * 1000
					)) . "\n";
					@file_put_contents($log_file, $log_entry, FILE_APPEND);
					// #endregion
				}
			} else {
				// #region agent log
				$log_entry = json_encode(array(
					'sessionId' => 'debug-session',
					'runId' => 'run1',
					'hypothesisId' => 'C',
					'location' => 'apply_field_mappings:area_range_not_set',
					'message' => 'Area min/max not set in newParam',
					'data' => array(
						'post_id' => $post_id,
						'has_minSize' => isset($newParam['minSize']),
						'has_maxSize' => isset($newParam['maxSize'])
					),
					'timestamp' => time() * 1000
				)) . "\n";
				@file_put_contents($log_file, $log_entry, FILE_APPEND);
				// #endregion
			}
		}
		
		// Enable property map if coordinates exist - extract from API data using helper function
		// #region agent log
		$log_file = __DIR__ . '/../.cursor/debug.log';
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H2',
			'location' => 'apply_field_mappings:before_map_extraction',
			'message' => 'Before extracting coordinates',
			'data' => array('post_id' => $post_id),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		$coords = $this->extract_coordinates_from_api($pd);
		
		// #region agent log
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H2',
			'location' => 'apply_field_mappings:after_map_extraction',
			'message' => 'After extracting coordinates',
			'data' => array(
				'post_id' => $post_id,
				'coords_found' => !empty($coords),
				'coords_value' => $coords
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		if($coords) {
			$this->set_houzez_map_data($post_id, $pd, $coords);
		} else {
			// No coordinates found - set map to Hide (0)
			update_post_meta($post_id, 'fave_property_map', 0);
			
			// Still set address if we have community/region/city data - Format: community, area, city
			$address_parts = array();
			
			// 1. Community (first priority)
			if(!empty($pd['community'])) {
				if(is_array($pd['community']) && isset($pd['community']['name'])) {
					$address_parts[] = $pd['community']['name'];
				} elseif(is_object($pd['community']) && isset($pd['community']->name)) {
					$address_parts[] = $pd['community']->name;
				} elseif(is_string($pd['community'])) {
					$address_parts[] = $pd['community'];
				}
			} elseif(!empty($pd['communityName'])) {
				$address_parts[] = $pd['communityName'];
			}
			
			// 2. Area/Region (second priority)
			if(!empty($pd['region'])) {
				$address_parts[] = $pd['region'];
			}
			
			// 3. City (third priority)
			if(!empty($pd['city'])) {
				$address_parts[] = $pd['city'];
			} elseif(!empty($pd['cityName'])) {
				$address_parts[] = $pd['cityName'];
			}
			
			if(!empty($address_parts)) {
				$full_address = implode(', ', $address_parts);
				update_post_meta($post_id, 'fave_property_map_address', $full_address);
			}
		}
		
		// Clean up temporary flags if they exist
		delete_post_meta($post_id, '_pixxi_has_coordinates');
		delete_post_meta($post_id, '_pixxi_map_lat');
		delete_post_meta($post_id, '_pixxi_map_lng');
		delete_post_meta($post_id, '_pixxi_map_pd');
		
		// #region agent log
		$log_file = __DIR__ . '/../.cursor/debug.log';
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'E',
			'location' => 'apply_field_mappings:final_check',
			'message' => 'Final meta values after all processing',
			'data' => array(
				'post_id' => $post_id,
				'final_fave_property_bedrooms' => get_post_meta($post_id, 'fave_property_bedrooms', true),
				'final_fave_property_bedrooms_underscore' => get_post_meta($post_id, '_fave_property_bedrooms', true),
				'final_fave_property_area' => get_post_meta($post_id, 'fave_property_area', true),
				'final_fave_property_area_underscore' => get_post_meta($post_id, '_fave_property_area', true),
				'listingType' => isset($pd['listingType']) ? $pd['listingType'] : 'NOT_SET'
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
		
		// #region agent log
		$log_file = __DIR__ . '/../.cursor/debug.log';
		$log_entry = json_encode(array(
			'sessionId' => 'debug-session',
			'runId' => 'run1',
			'hypothesisId' => 'H5',
			'location' => 'apply_field_mappings:function_exit',
			'message' => 'Function exit - final values check',
			'data' => array(
				'post_id' => $post_id,
				'fave_property_map_final' => get_post_meta($post_id, 'fave_property_map', true),
				'fave_property_location_final' => get_post_meta($post_id, 'fave_property_location', true),
				'fave_property_map_address_final' => get_post_meta($post_id, 'fave_property_map_address', true)
			),
			'timestamp' => time() * 1000
		)) . "\n";
		@file_put_contents($log_file, $log_entry, FILE_APPEND);
		// #endregion
	}
	
	function apply_taxonomy_mappings($post_id, $pd) {
		// Hard-coded default taxonomy mappings for Houzez
		$default_taxonomy_mappings = array(
			'property_developers' => 'developer',
			'property_type' => 'houseType',
			'property_status' => 'listingType',
			'property_feature' => 'amenities',
			'property_area' => 'region', // Region goes to area, not state
			'property_city' => 'city',
			'community' => 'community' // Community taxonomy mapping
		);
		
		// Process taxonomy mappings
		foreach($default_taxonomy_mappings as $taxonomy => $api_field) {
			// Get value from API data - use null as default to distinguish between empty string and not found
			$value = $this->get_field_value($pd, $api_field, null);
			
			// If field not found, try alternative field names for property_type
			if($taxonomy === 'property_type' && ($value === null || (is_array($value) && empty($value)))) {
				// Try houseType if propertyType is not found or empty
				if(isset($pd['houseType']) && is_array($pd['houseType']) && !empty($pd['houseType'])) {
					$value = $pd['houseType'];
				} elseif(isset($pd['propertyType'])) {
					$value = is_array($pd['propertyType']) ? $pd['propertyType'] : array($pd['propertyType']);
				}
			}
			
			// Add fallback for property_area - try alternative field names
			if($taxonomy === 'property_area' && ($value === null || empty($value))) {
				// Try region first (primary field), then try communityName, then community
				if(isset($pd['region']) && !empty($pd['region'])) {
					$value = $pd['region'];
				} elseif(array_key_exists('community', $pd) && $pd['community'] !== null) {
					// Check if community is an object/array with a name property
					if(is_array($pd['community']) && isset($pd['community']['name'])) {
						$value = $pd['community']['name'];
					} elseif(is_object($pd['community']) && isset($pd['community']->name)) {
						$value = $pd['community']->name;
					} elseif(!empty($pd['community'])) {
						$value = $pd['community'];
					}
				} elseif(array_key_exists('communityName', $pd) && !empty($pd['communityName'])) {
					$value = $pd['communityName'];
				}
			}
			
			// Add fallback for property_city - try alternative field names
			if($taxonomy === 'property_city' && ($value === null || empty($value))) {
				// Try city first (primary field), then try cityName
				if(isset($pd['city']) && !empty($pd['city'])) {
					$value = $pd['city'];
				} elseif(isset($pd['cityName']) && !empty($pd['cityName'])) {
					$value = $pd['cityName'];
				}
			}
			
			// Add fallback for community taxonomy - try alternative field names
			if($taxonomy === 'community' && ($value === null || empty($value))) {
				// Try community first (primary field), then try communityName
				if(array_key_exists('community', $pd) && $pd['community'] !== null) {
					// Check if community is an object/array with a name property
					if(is_array($pd['community']) && isset($pd['community']['name'])) {
						$value = $pd['community']['name'];
					} elseif(is_object($pd['community']) && isset($pd['community']->name)) {
						$value = $pd['community']->name;
					} elseif(!empty($pd['community'])) {
						$value = $pd['community'];
					}
				} elseif(array_key_exists('communityName', $pd) && !empty($pd['communityName'])) {
					$value = $pd['communityName'];
				}
			}
			
			// Special handling for property_status - convert listingType to status
			if($taxonomy === 'property_status' && !empty($value) && !is_array($value)) {
				if($value == 'NEW'){
					$value = 'Off Plan';
				} elseif($value == 'SELL'){
					$value = 'Buy';
				} elseif($value == 'RENT'){
					$value = 'Rent';
				}
			}
			
			// Handle arrays (for property_type, property_feature, etc.)
			if(is_array($value)) {
				// Filter out empty values and ensure array is not empty
				$value = array_filter($value, function($item) {
					return !empty($item) && $item !== null && $item !== '';
				});
				
				if(!empty($value)) {
					// Reset array keys to ensure sequential indexing
					$value = array_values($value);
					$this->pixxi_create_propertyfeatures($post_id, $value, $taxonomy);
				}
			} elseif($value !== null && !empty($value)) {
				// Single value - ensure it's not empty and not null
				// Trim whitespace and ensure it's a valid string
				$value = is_string($value) ? trim($value) : $value;
				if(!empty($value)) {
					$this->pixxi_create_propertyterm($post_id, $value, $taxonomy);
				}
			}
		}
	}
	
	
	function insert_sku($insert_sku, $api_data){
		// Start output buffering
		ob_start();
		
		// Check property limit from license
		$max_properties = -1; // Unlimited by default
		if (class_exists('Pixxi_License_Feature_Gates')) {
			$max_properties = Pixxi_License_Feature_Gates::get_max_properties();
			
			// Get current property count
			global $wpdb;
			$current_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}postmeta WHERE meta_key = '_xm_api_sku'");
			
			// Limit insertions if we're at or near the limit
			if ($max_properties > 0 && $current_count >= $max_properties) {
				ob_end_clean();
				return array('error' => "Property limit reached ($max_properties). Please upgrade your plan to add more properties.");
			}
			
			// Limit the number of items to insert based on remaining quota
			if ($max_properties > 0) {
				$remaining = $max_properties - $current_count;
				if ($remaining < count($insert_sku)) {
					$insert_sku = array_slice($insert_sku, 0, $remaining);
				}
			}
		}
		
		$inserted_count = 0;
		$error_count = 0;
		
		foreach($insert_sku as $sku){
			try {
				if(!isset($api_data[$sku])) {
					continue;
				}
				
				$pd = json_decode($api_data[$sku],true);
				
				if(!$pd || !isset($pd['id'])) {
					continue; // Skip invalid data
				}
				
				// Handle payment plan - only for NEW listing type
				$paymentPlan = array('one' => '', 'two' => '', 'three' => '', 'four' => '');
				if(isset($pd['newParam']['paymentPlan']) && !empty($pd['newParam']['paymentPlan'])) {
					$paymentPlanData = is_string($pd['newParam']['paymentPlan']) ? json_decode($pd['newParam']['paymentPlan'], true) : $pd['newParam']['paymentPlan'];
					if(is_array($paymentPlanData)) {
						$paymentPlan = array_merge($paymentPlan, $paymentPlanData);
					}
				}
				
				if($this->check_if_mvalue_exist($pd['id'])){
				$desc = '';
				if(!empty($pd['description'])){
					$desc = $pd['description'];
				}
				
				// API uses 'name' field, not 'title'
				$title = '';
				if(!empty($pd['name'])){
					$title = $pd['name'];
				} elseif(!empty($pd['title'])){
					$title = $pd['title'];
				}
				
				if(empty($title)) {
					continue; // Skip if no title/name
				}
					$post = array(
						'post_author' => 1,
						'post_content' => $desc, //$pd->description,
						'post_status' => "publish",
						'post_title' => $title, //$pd->title,
						'post_type' => "property",
					);
					$post_id = wp_insert_post( $post );
					if(!$post_id)
					continue;
					update_post_meta( $post_id, '_xm_api_sku', $pd['id']);
					
					// Save full API data
					update_post_meta( $post_id, 'pixi_import_data', json_encode($pd));
					
					// Apply all field mappings from data mapping configuration
					$this->apply_field_mappings($post_id, $pd);
					
					// Apply taxonomy mappings from data mapping configuration
					$this->apply_taxonomy_mappings($post_id, $pd);
					
			// Geolocation - extract coordinates and set map data
			$coords = $this->extract_coordinates_from_api($pd);
			if($coords) {
				// Also set houzez geolocation fields for compatibility
				update_post_meta($post_id, 'houzez_geolocation_lat', $coords['lat']);
				update_post_meta($post_id, 'houzez_geolocation_long', $coords['lng']);
				// Set map data using helper function
				$this->set_houzez_map_data($post_id, $pd, $coords);
			} else {
				// No coordinates found - set map to Hide (0)
				update_post_meta($post_id, 'fave_property_map', 0);
				
				// Still set address if we have region/city data
				$address_parts = array();
				if(!empty($pd['region'])) {
					$address_parts[] = $pd['region'];
				}
				if(!empty($pd['city'])) {
					$address_parts[] = $pd['city'];
				}
				if(!empty($pd['cityName'])) {
					$address_parts[] = $pd['cityName'];
				}
				if(!empty($address_parts)) {
					$full_address = implode(', ', $address_parts);
					update_post_meta($post_id, 'fave_property_map_address', $full_address);
				}
			}
					
					// Agent - Check multiple possible field names
					$agent_id = '';
					$agent_email = '';
					$agent_name = '';
					$agent_phone = '';
					$agent_avatar = '';
					
					// Try different field name variations
					if(!empty($pd['agentEmail'])) {
						$agent_email = $pd['agentEmail'];
					} elseif(!empty($pd['agent']['email'])) {
						$agent_email = $pd['agent']['email'];
					}
					
					if(!empty($pd['agentName'])) {
						$agent_name = $pd['agentName'];
					} elseif(!empty($pd['agent']['name'])) {
						$agent_name = $pd['agent']['name'];
					} elseif(!empty($pd['agent']['agentName'])) {
						$agent_name = $pd['agent']['agentName'];
					}
					
					if(!empty($pd['agentPhone'])) {
						$agent_phone = $pd['agentPhone'];
					} elseif(!empty($pd['agent']['phone'])) {
						$agent_phone = $pd['agent']['phone'];
					} elseif(!empty($pd['agent']['mobile'])) {
						$agent_phone = $pd['agent']['mobile'];
					}
					
					if(!empty($pd['agentAvatar'])) {
						$agent_avatar = $pd['agentAvatar'];
					} elseif(!empty($pd['agent']['avatar'])) {
						$agent_avatar = $pd['agent']['avatar'];
					} elseif(!empty($pd['agent']['image'])) {
						$agent_avatar = $pd['agent']['image'];
					} elseif(!empty($pd['agent']['originalAvatar'])) {
						$agent_avatar = $pd['agent']['originalAvatar'];
					}
					
					// If we have agent email, try to find or create agent
					if(!empty($agent_email)) {
						// First, try to find existing agent by email
						$existing_agents = get_posts(array(
							'post_type' => 'houzez_agent',
							'meta_key' => 'fave_agent_email',
							'meta_value' => sanitize_email($agent_email),
							'posts_per_page' => 1,
							'fields' => 'ids',
						));
						
						if(!empty($existing_agents)) {
							$agent_id = $existing_agents[0];
						} elseif(!empty($agent_name) && !empty($agent_email)) {
							// Create new agent
							$agent_data = array(
								'name' => $agent_name,
								'email' => $agent_email,
								'phone' => $agent_phone,
								'avatar' => $agent_avatar,
								'subCompanyConfigId' => isset($pd['agent']['subCompanyConfigId']) ? $pd['agent']['subCompanyConfigId'] : ''
							);
							$agent_id = $this->create_houzez_agent_from_array($agent_data);
						}
					}
					
					// Assign agent to property if we have one
					if(!empty($agent_id) && is_numeric($agent_id)) {
						update_post_meta( $post_id, 'fave_agent_display_option', 'agent_info');
						update_post_meta( $post_id, 'fave_agents', $agent_id);
					}
					
					// Floor plans (only for NEW projects)
					if(!empty($pd['newParam']['floorPlan']) && is_array($pd['newParam']['floorPlan'])){
						$this->add_floor_plans_to_houzez($post_id, $pd['newParam']['floorPlan']);
					}

					// Photos
					if (!empty($pd['photos']) && is_array($pd['photos'])) {
						$property_id = !empty($pd['propertyId']) ? $pd['propertyId'] : (isset($pd['id']) ? $pd['id'] : '');
						if(!empty($property_id)) {
							try {
								$this->attach_product_thumbnail($property_id, $post_id, $pd['photos']);
							} catch (Exception $e) {
								error_log('Error attaching photos to property ' . $post_id . ': ' . $e->getMessage());
							}
						}
					}
					
					// Update database table with post ID, status, and updated timestamp
					global $wpdb;
					$current_time = (int)time();
					$wpdb->update(
						PIXXI_TABLE,
						array(
							'pid' => $post_id,
							'status' => 1,
							'updated' => $current_time
						),
						array('sku' => (string)$pd['id']),
						array('%d', '%d', '%d'),
						array('%s')
					);
					
					$inserted_count++;
				}
			} catch (Exception $e) {
				$error_count++;
				error_log('Error inserting property SKU ' . $sku . ': ' . $e->getMessage());
				continue;
			}
		}
		
		// Report usage to license API
		if ($inserted_count > 0 && class_exists('Pixxi_License_Feature_Gates')) {
			Pixxi_License_Feature_Gates::report_usage('property_sync', $inserted_count);
		}
		
		// Discard any output
		ob_end_clean();
	}
	// Function to add floor plans to Houzez theme
	function add_floor_plans_to_houzez($post_id, $floorPlans) {
		if(empty($floorPlans) || !is_array($floorPlans)) {
			return;
		}
		
		// Delete existing floor plans if any
		$floor_plan_data = [];
		delete_post_meta($post_id, 'floor_plans');
		
		foreach ($floorPlans as $plan) {
			if(empty($plan) || !is_array($plan)) {
				continue;
			}
			// Prepare the floor plan data
			$floor_plan_data[] = array(
				'fave_plan_title' => isset($plan['title']) ? $plan['title'] : '',
				'fave_plan_rooms' => isset($plan['name']) ? $plan['name'] : '',
				'fave_plan_price' => isset($plan['price']) ? $plan['price'] : '',
				'fave_plan_price_postfix' => '',
				'fave_plan_size' => isset($plan['area']) ? $plan['area'] : '',
				'fave_plan_size_postfix' => 'sq ft',
				'fave_plan_image' => isset($plan['imgUrl'][0]) ? $plan['imgUrl'][0] : '',
				'fave_plan_description' => '',
				'fave_plan_rooms_' => '',
				'fave_plan_bathrooms' => '',
				'fave_plan_bedrooms' => ''
			);
			// Add the floor plan to the property
		}
		add_post_meta($post_id, 'floor_plans', $floor_plan_data);
	}
	/**
	 * Update existing properties in place (does not delete/recreate)
	 * Uses update_post_meta() and wp_set_object_terms() which replaces existing terms
	 * This ensures properties are updated without losing data or creating duplicates
	 */
	function update_sku($update_sku,$wp_data, $api_data){
		// Start output buffering
		ob_start();
		foreach($update_sku as $sku){
			$post_id = $wp_data[$sku];
			if(!$post_id)
				continue;
			
			$pd = json_decode($api_data[$sku],true);
			
			if(!$pd) {
				continue; // Skip invalid data
			}
			
			// Save full API data
			update_post_meta( $post_id, 'pixi_import_data', json_encode($pd));
			
			// Apply all field mappings from data mapping configuration
			// This updates post meta in place, does not delete/recreate the post
			$this->apply_field_mappings($post_id, $pd);
			
			// Apply taxonomy mappings from data mapping configuration
			// wp_set_object_terms() replaces existing terms (does not accumulate)
			$this->apply_taxonomy_mappings($post_id, $pd);

			// Geolocation - extract coordinates and set map data
			$coords = $this->extract_coordinates_from_api($pd);
			if($coords) {
				// Also set houzez geolocation fields for compatibility
				update_post_meta($post_id, 'houzez_geolocation_lat', $coords['lat']);
				update_post_meta($post_id, 'houzez_geolocation_long', $coords['lng']);
				// Set map data using helper function
				$this->set_houzez_map_data($post_id, $pd, $coords);
			} else {
				// No coordinates found - set map to Hide (0)
				update_post_meta($post_id, 'fave_property_map', 0);
				
				// Still set address if we have community/region/city data - Format: community, area, city
				$address_parts = array();
				
				// 1. Community (first priority)
				if(!empty($pd['community'])) {
					if(is_array($pd['community']) && isset($pd['community']['name'])) {
						$address_parts[] = $pd['community']['name'];
					} elseif(is_object($pd['community']) && isset($pd['community']->name)) {
						$address_parts[] = $pd['community']->name;
					} elseif(is_string($pd['community'])) {
						$address_parts[] = $pd['community'];
					}
				} elseif(!empty($pd['communityName'])) {
					$address_parts[] = $pd['communityName'];
				}
				
				// 2. Area/Region (second priority)
				if(!empty($pd['region'])) {
					$address_parts[] = $pd['region'];
				}
				
				// 3. City (third priority)
				if(!empty($pd['city'])) {
					$address_parts[] = $pd['city'];
				} elseif(!empty($pd['cityName'])) {
					$address_parts[] = $pd['cityName'];
				}
				
				if(!empty($address_parts)) {
					$full_address = implode(', ', $address_parts);
					update_post_meta($post_id, 'fave_property_map_address', $full_address);
				}
			}
			
			// Listing Status
			$status = '';
			if(isset($pd['listingType'])) {
				if($pd['listingType'] == 'NEW'){
					$status = 'Off Plan';
				}elseif($pd['listingType'] == 'SELL'){
					$status = 'Buy';
				}elseif($pd['listingType'] == 'RENT'){
					$status = 'Rent';
				}
			}
			if(!empty($status)){
				$this->pixxi_create_propertyterm($post_id, $status, 'property_status');
			}
			
			// Amenities
			if(!empty($pd['amenities']) && is_array($pd['amenities'])){
				$this->pixxi_create_propertyfeatures($post_id, array_filter($pd['amenities']), 'property_feature');
			}
			
			// Developer term
			$developer_name = !empty($pd['developer']) ? $pd['developer'] : (isset($pd['developerName']) ? $pd['developerName'] : '');
			if(!empty($developer_name)){
				$this->pixxi_create_propertyterm($post_id, $developer_name, 'property_developers');
			}
			
			// Floor plans (only for NEW projects)
			if(!empty($pd['newParam']['floorPlan']) && is_array($pd['newParam']['floorPlan'])){
				$this->add_floor_plans_to_houzez($post_id, $pd['newParam']['floorPlan']);
			}
			
			// Property Type - API uses houseType array
			$property_types = array();
			if(!empty($pd['houseType']) && is_array($pd['houseType'])){
				$property_types = $pd['houseType'];
			} elseif(!empty($pd['propertyType'])){
				$property_types = is_array($pd['propertyType']) ? $pd['propertyType'] : array($pd['propertyType']);
			}
			if(!empty($property_types)){
				$this->pixxi_create_propertyfeatures($post_id, $property_types, 'property_type');
			}
			
			// Agent - Check multiple possible field names
			$agent_id = '';
			$agent_email = '';
			$agent_name = '';
			$agent_phone = '';
			$agent_avatar = '';
			
			// Try different field name variations
			if(!empty($pd['agentEmail'])) {
				$agent_email = $pd['agentEmail'];
			} elseif(!empty($pd['agent']['email'])) {
				$agent_email = $pd['agent']['email'];
			}
			
			if(!empty($pd['agentName'])) {
				$agent_name = $pd['agentName'];
			} elseif(!empty($pd['agent']['name'])) {
				$agent_name = $pd['agent']['name'];
			} elseif(!empty($pd['agent']['agentName'])) {
				$agent_name = $pd['agent']['agentName'];
			}
			
			if(!empty($pd['agentPhone'])) {
				$agent_phone = $pd['agentPhone'];
			} elseif(!empty($pd['agent']['phone'])) {
				$agent_phone = $pd['agent']['phone'];
			} elseif(!empty($pd['agent']['mobile'])) {
				$agent_phone = $pd['agent']['mobile'];
			}
			
			if(!empty($pd['agentAvatar'])) {
				$agent_avatar = $pd['agentAvatar'];
			} elseif(!empty($pd['agent']['avatar'])) {
				$agent_avatar = $pd['agent']['avatar'];
			} elseif(!empty($pd['agent']['image'])) {
				$agent_avatar = $pd['agent']['image'];
			} elseif(!empty($pd['agent']['originalAvatar'])) {
				$agent_avatar = $pd['agent']['originalAvatar'];
			}
			
			// If we have agent email, try to find or create agent
			if(!empty($agent_email)) {
				// First, try to find existing agent by email
				$existing_agents = get_posts(array(
					'post_type' => 'houzez_agent',
					'meta_key' => 'fave_agent_email',
					'meta_value' => sanitize_email($agent_email),
					'posts_per_page' => 1,
					'fields' => 'ids',
				));
				
				if(!empty($existing_agents)) {
					$agent_id = $existing_agents[0];
				} elseif(!empty($agent_name) && !empty($agent_email)) {
					// Create new agent
					$agent_data = array(
						'name' => $agent_name,
						'email' => $agent_email,
						'phone' => $agent_phone,
						'avatar' => $agent_avatar,
						'subCompanyConfigId' => isset($pd['agent']['subCompanyConfigId']) ? $pd['agent']['subCompanyConfigId'] : ''
					);
					$agent_id = $this->create_houzez_agent_from_array($agent_data);
				}
			}
			
			// Assign agent to property if we have one
			if(!empty($agent_id) && is_numeric($agent_id)) {
				update_post_meta( $post_id, 'fave_agent_display_option', 'agent_info');
				update_post_meta( $post_id, 'fave_agents', $agent_id);
			}
			
			// Photos - only update if default thumbnail exists or no thumbnail
			if (!has_post_thumbnail($post_id) || (has_post_thumbnail($post_id) && get_post_thumbnail_id($post_id) == 28546)) {
				if (!empty($pd['photos']) && is_array($pd['photos'])) {
					$property_id = !empty($pd['propertyId']) ? $pd['propertyId'] : (isset($pd['id']) ? $pd['id'] : '');
					if(!empty($property_id)) {
						try {
							$this->attach_product_thumbnail($property_id, $post_id, $pd['photos']);
						} catch (Exception $e) {
							error_log('Error attaching photos to property ' . $post_id . ': ' . $e->getMessage());
						}
					}
				}
			}
			
			// Update database table with post ID, status, and updated timestamp
			global $wpdb;
			$current_time = (int)time();
			$wpdb->update(
				PIXXI_TABLE,
				array(
					'pid' => $post_id,
					'status' => 1,
					'updated' => $current_time
				),
				array('sku' => (string)$sku),
				array('%d', '%d', '%d'),
				array('%s')
			);
		}
		
		// Discard any output
		ob_end_clean();
	}
	/**
	 * Clean up orphaned taxonomy terms (terms with no associated posts)
	 * This is called after property deletion to remove unused taxonomy terms
	 * Only runs if the option is enabled in settings
	 */
	function cleanup_orphaned_taxonomy_terms(){
		// Check if cleanup is enabled in settings
		$opt = get_option('pixxi_opt_val');
		// Migration: Check for old option name
		if(!$opt) {
			$opt = get_option('lw_opt_val');
			if($opt) {
				update_option('pixxi_opt_val', $opt);
			}
		}
		$pixxi_option = $opt ? json_decode($opt, true) : array();
		
		// Check if cleanup is enabled (default to disabled if not set)
		$cleanup_enabled = isset($pixxi_option['pixxi_cleanup_orphaned_taxonomy']) && $pixxi_option['pixxi_cleanup_orphaned_taxonomy'] == 1;
		
		if(!$cleanup_enabled) {
			// Cleanup is disabled, skip
			return;
		}
		
		// List of all property taxonomies used by the plugin
		$taxonomies = array(
			'property_status',
			'property_type', 
			'property_feature',
			'property_developers',
			'property_area',
			'property_city',
			'community'
		);
		
		foreach($taxonomies as $taxonomy) {
			// Check if taxonomy exists
			if(!taxonomy_exists($taxonomy)) {
				continue;
			}
			
			// Get all terms for this taxonomy, including empty ones
			$terms = get_terms(array(
				'taxonomy' => $taxonomy,
				'hide_empty' => false
			));
			
			// Skip if get_terms returned an error or empty array
			if(is_wp_error($terms) || empty($terms)) {
				continue;
			}
			
			// Delete terms with zero post count
			foreach($terms as $term) {
				if($term->count == 0) {
					wp_delete_term($term->term_id, $taxonomy);
				}
			}
		}
	}
	
	function delete_sku($delete_sku,$wp_data){
		// Start output buffering
		ob_start();
		foreach($delete_sku as $sku){
			$post_id = $wp_data[$sku];
			if(!$post_id)
				continue;
			wp_delete_post($post_id, true);
		}
		
		// Clean up orphaned taxonomy terms after deleting properties
		$this->cleanup_orphaned_taxonomy_terms();
		
		// Discard any output
		ob_end_clean();
	}
	function smart_chunk($arr, $lw){
		$chunk = $lw['chunk']; //0
		$size = $lw['size']; //1000
		$count = count($arr); //1
		if($count == 0){
			return [];
		}
		$total_chunks= $count/$size;
		$chunks = ceil($count/$total_chunks);
		$arr_chunk = array_chunk($arr, $chunks );
		if(isset($arr_chunk[$chunk]))
			return $arr_chunk[$chunk];
		return [];
	}
	function update_products($lw){
		// Start output buffering to capture any echo statements
		ob_start();
		
		// Validate $lw array
		if(!is_array($lw) || !isset($lw['action'])) {
			ob_end_clean();
			return;
		}
		
		global $wpdb;
		$wp_data = $this->get_save_sku();
		$api_data = $this->get_products_data();
		
		// Ensure arrays are valid
		if(!is_array($wp_data)) {
			$wp_data = array();
		}
		if(!is_array($api_data)) {
			$api_data = array();
		}
		
		$wp_sku = array_keys($wp_data);
		$api_sku = array_keys($api_data);
		
		// Check for pause or stop before processing
		if(isset($lw['action']) && $lw['action'] == 'pause') {
			// Save last action before pause for resume
			if(!isset($lw['last_action'])) {
				// Try to determine last action from context
				$wp_sku = array_keys($wp_data);
				$api_sku = array_keys($api_data);
				$delete_sku = array_diff($wp_sku, $api_sku);
				$update_sku = array_intersect($wp_sku, $api_sku);
				$insert_sku = array_diff($api_sku, $wp_sku);
				
				if(!empty($delete_sku)) {
					$lw['last_action'] = 'delete';
				} elseif(!empty($update_sku)) {
					$lw['last_action'] = 'update';
				} elseif(!empty($insert_sku)) {
					$lw['last_action'] = 'insert';
				} else {
					$lw['last_action'] = 'update'; // Default
				}
			}
			// Save current state and return
			update_option('pixxi_api_info', json_encode($lw));
			return;
		}
		
		if(isset($lw['action']) && $lw['action'] == 'stop') {
			return;
		}
		
		// Save current action before switching (for pause resume)
		if(isset($lw['action']) && in_array($lw['action'], array('delete', 'update', 'insert'))) {
			$lw['last_action'] = $lw['action'];
		}
		
		switch($lw['action']){
			case 'delete':
				// Check if auto delete is allowed by license
				$can_delete = true;
				if (class_exists('Pixxi_License_Feature_Gates')) {
					$can_delete = Pixxi_License_Feature_Gates::can_auto_delete();
				} else {
					// Fallback: check option setting
					$opt = get_option('pixxi_opt_val');
					if ($opt) {
						$pixxi_option = json_decode($opt, true);
						$can_delete = isset($pixxi_option['pixxi_auto_delete']) && $pixxi_option['pixxi_auto_delete'] == 1;
					}
				}
				
				if (!$can_delete) {
					// Skip delete, move to update
					$lw['chunk'] = 0;
					$lw['action'] = 'update';
				} else {
					$delete_sku = array_diff ($wp_sku, $api_sku);
					$delete_sku = $this->smart_chunk($delete_sku, $lw);
					if(!empty($delete_sku)){
						$lw['chunk']++;
						$this->delete_sku($delete_sku,$wp_data);
					}else{
						// Delete phase complete - cleanup orphaned taxonomy terms
						$this->cleanup_orphaned_taxonomy_terms();
						$lw['chunk'] = 0;
						$lw['action'] = 'update';
					}
				}
			break;
			case 'update':
				$update_sku = array_intersect($wp_sku, $api_sku);
				$lw['size'] = 200;
				$update_sku = $this->smart_chunk($update_sku, $lw);
				if(!empty($update_sku)){
					$lw['chunk']++;
					$this->update_sku($update_sku,$wp_data, $api_data);
				}else{
					// Update phase complete - cleanup orphaned taxonomy terms
					$this->cleanup_orphaned_taxonomy_terms();
					$lw['chunk'] = 0;
					$lw['action'] = 'insert';
				}
			break;
			case 'insert':
				$insert_sku = array_diff ( $api_sku, $wp_sku);
				$lw['size'] = 5;
				$insert_sku = $this->smart_chunk($insert_sku, $lw);
				if(!empty($insert_sku)){
					$lw['chunk']++;
					$this->insert_sku($insert_sku,$api_data);
				}else{
					// Insert phase complete - cleanup orphaned taxonomy terms
					$this->cleanup_orphaned_taxonomy_terms();
					$lw['chunk'] = 0;
					$lw['action'] = 'stop';
				}
			break;
		}
		
		// Ensure $lw is still valid before saving
		if(is_array($lw)) {
			update_option('pixxi_api_info',json_encode($lw));
		}
		
		// Discard any output before returning
		ob_end_clean();
		
		// Don't use exit() as it breaks AJAX JSON responses
		// Return instead - the caller can handle the response
		return;
	}
}
function pixxi_properties_import(){
	$pixxi = new pixxi_property_class();
	$pixxi->init();
}
add_action('init', 'pixxi_properties_import');