/**
 * Dashboard JavaScript for Pixxi Property Importer Plugin
 * Author: Dev Mizan
 * Website: https://devmizan.com
 */

(function($) {
	'use strict';

	const Dashboard = {
		init: function() {
			this.bindEvents();
			this.startStatsRefresh();
			this.initCronToggleState();
			this.checkSyncStatus();
			this.updateUnreadBadge();
			// Update unread badge every 30 seconds
			setInterval(() => {
				this.updateUnreadBadge();
			}, 30000);
			// Preload audio on page load
			this.preloadAudio();
		},
		
		preloadAudio: function() {
			const audio = document.getElementById('pixxi-notification-sound');
			if(audio) {
				// Load the audio to ensure it's ready
				audio.load();
				// Try to play and immediately pause to "unlock" audio for autoplay
				audio.play().then(() => {
					audio.pause();
					audio.currentTime = 0;
				}).catch(() => {
					// Ignore - will need user interaction
				});
			}
		},
		
		handleRefreshLicenseStatus: function(e) {
			e.preventDefault();
			const $btn = $('#pixxi-refresh-license-status');
			const originalHtml = $btn.html();
			
			$btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite; vertical-align: middle;"></span> Refreshing...');
			
			// Clear license cache and reload page
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_refresh_license',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					// Reload page to show updated license status
					setTimeout(() => {
						location.reload();
					}, 500);
				},
				error: () => {
					$btn.prop('disabled', false).html(originalHtml);
					Swal.fire({
						title: 'Error',
						text: 'Failed to refresh license status',
						icon: 'error',
						confirmButtonColor: '#EF4444',
						timer: 2000
					});
				}
			});
		},
		
		
		checkSyncStatus: function() {
			// Check if sync is in progress on page load
			const statusText = $('#sync-status-text').text().trim();
			if(statusText && statusText !== 'Ready to sync' && statusText !== 'Paused') {
				// Sync is active, show progress bar
				$('#sync-progress').show();
				$('#progress-text').text(statusText);
				// Set progress to 50% if active (will be updated by polling)
				$('#progress-fill').css('width', '50%');
			}
		},

		bindEvents: function() {
			// Download button
			$('#pixxi-download-btn').on('click', this.handleDownload.bind(this));
			
			// Full sync button
			$('#pixxi-full-sync-btn').on('click', this.handleFullSync.bind(this));
			
			// Pause button
			$('#pixxi-pause-btn').on('click', this.handlePause.bind(this));
			
			// Stop button
			$('#pixxi-stop-btn').on('click', this.handleStop.bind(this));
			
			// Settings form submission
			$('#pixxi_import').on('submit', this.handleSettingsSave.bind(this));
			
			// Cron settings toggle
			$('#pixxi_cron_enabled').on('change', this.handleCronToggle.bind(this));
			$('#pixxi_cron_interval').on('change', this.handleCronIntervalChange.bind(this));
			
			// License activation - Upgrade Now button
			$('#pixxi-upgrade-now-btn').on('click', this.handleLicenseActivation.bind(this));
			
			
			// Accordion handlers
			$(document).on('click', '.pixxi-accordion-header', this.handleAccordionToggle.bind(this));
			
			// Refresh license status button
			$('#pixxi-refresh-license-status').on('click', this.handleRefreshLicenseStatus.bind(this));
			
			// Support ticket handlers - use event delegation to ensure it works
			$(document).on('click', '#pixxi-support-btn', this.handleSupportButtonClick.bind(this));
			$('#pixxi-support-offcanvas-close, #pixxi-support-offcanvas-close-detail, .pixxi-offcanvas-overlay').on('click', this.handleCloseOffCanvas.bind(this));
			$('#pixxi-back-to-tickets').on('click', this.handleBackToTickets.bind(this));
			$('#pixxi-toggle-new-ticket-form').on('click', this.handleToggleNewTicketForm.bind(this));
			$('#pixxi-cancel-new-ticket').on('click', this.handleCancelNewTicket.bind(this));
			// Bind to both forms (modal and offcanvas)
			$('#pixxi-support-form, #pixxi-support-form-modal').on('submit', this.handleSupportFormSubmit.bind(this));
			$('#pixxi-comment-form').on('submit', this.handleCommentFormSubmit.bind(this));
			$(document).on('click', '.pixxi-ticket-item', this.handleTicketClick.bind(this));
			
		},
		
		handleAccordionToggle: function(e) {
			const $header = $(e.currentTarget);
			const $item = $header.closest('.pixxi-accordion-item');
			const $content = $item.find('.pixxi-accordion-content');
			const isActive = $header.hasClass('active');
			
			// Close all other accordions (optional - remove if you want multiple open)
			// $('.pixxi-accordion-header').not($header).removeClass('active');
			// $('.pixxi-accordion-content').not($content).removeClass('active');
			
			// Toggle current accordion
			if(isActive) {
				$header.removeClass('active');
				$content.removeClass('active');
			} else {
				$header.addClass('active');
				$content.addClass('active');
			}
		},
		
		// Support Ticket Handlers
		handleSupportButtonClick: function(e) {
			e.preventDefault();
			e.stopPropagation();
			console.log('Support button clicked');
			// Open off-canvas directly with ticket list
			this.openSupportOffCanvas();
		},
		
		openTicketsList: function() {
			// Open off-canvas with ticket list
			this.openSupportOffCanvas();
		},
		
		handleToggleNewTicketForm: function(e) {
			e.preventDefault();
			$('#pixxi-new-ticket-form').slideToggle(300);
		},
		
		handleCancelNewTicket: function(e) {
			e.preventDefault();
			$('#pixxi-new-ticket-form').slideUp(300);
			const $form = $('#pixxi-support-form');
			if($form.length && $form[0]) {
				$form[0].reset();
			}
		},
		
		handleCloseOffCanvas: function(e) {
			if($(e.target).hasClass('pixxi-offcanvas-overlay') || $(e.target).closest('.pixxi-offcanvas-close').length) {
				this.closeSupportOffCanvas();
			}
		},
		
		closeSupportOffCanvas: function() {
			const $offcanvas = $('#pixxi-support-offcanvas');
			$offcanvas.removeClass('active');
			
			// Wait for animation to complete before hiding
			setTimeout(() => {
				$offcanvas.css('display', 'none');
			}, 300);
			
			if(this.commentPollInterval) {
				clearInterval(this.commentPollInterval);
				this.commentPollInterval = null;
			}
		},
		
		handleSupportFormSubmit: function(e) {
			e.preventDefault();
			e.stopPropagation();
			const $form = $(e.currentTarget);
			const $submitBtn = $form.find('#pixxi-support-submit');
			const originalText = $submitBtn.html();
			const self = this;
			
			$submitBtn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Creating...');
			
			// Get form values from the current form being submitted (scoped to avoid duplicate IDs)
			// Use serializeArray for more reliable form data extraction
			const formArray = $form.serializeArray();
			const formData = {
				action: 'pixxi_create_support_ticket',
				nonce: pixxiDashboard.nonce
			};
			
			// Convert form array to object
			$.each(formArray, function(i, field) {
				formData[field.name] = field.value;
			});
			
			// Fallback to direct field access if serializeArray doesn't work
			if (!formData.name) formData.name = $form.find('[name="name"]').val() || '';
			if (!formData.site_url) formData.site_url = $form.find('[name="site_url"]').val() || '';
			if (!formData.subject) formData.subject = $form.find('[name="subject"]').val() || '';
			if (!formData.email) formData.email = $form.find('[name="email"]').val() || '';
			if (!formData.issue_description) formData.issue_description = $form.find('[name="issue_description"]').val() || '';
			if (!formData.login_creds) formData.login_creds = $form.find('[name="login_creds"]').val() || '';
			if (!formData.wp_version) formData.wp_version = $form.find('[name="wp_version"]').val() || '';
			if (!formData.license_key) formData.license_key = $form.find('[name="license_key"]').val() || '';
			
			// Validate required fields before submitting
			const missingFields = [];
			if (!formData.name || formData.name.trim() === '') missingFields.push('Name');
			if (!formData.site_url || formData.site_url.trim() === '') missingFields.push('Site URL');
			if (!formData.subject || formData.subject.trim() === '') missingFields.push('Subject');
			if (!formData.email || formData.email.trim() === '') missingFields.push('Email');
			if (!formData.issue_description || formData.issue_description.trim() === '') missingFields.push('Issue Description');
			
			if (missingFields.length > 0) {
				$submitBtn.prop('disabled', false).html(originalText);
				Swal.fire({
					title: 'Validation Error',
					text: 'Please fill in all required fields. Missing: ' + missingFields.join(', '),
					icon: 'error',
					confirmButtonColor: '#EF4444'
				});
				return;
			}
			
			// Trim all string values
			formData.name = formData.name.trim();
			formData.site_url = formData.site_url.trim();
			formData.subject = formData.subject.trim();
			formData.email = formData.email.trim();
			formData.issue_description = formData.issue_description.trim();
			
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: formData,
				success: function(response) {
					if(response.success) {
						// Hide form and reset
						$('#pixxi-new-ticket-form').slideUp(300);
						if($form.length && $form[0]) {
							$form[0].reset();
						}
						
						// Close offcanvas first
						self.closeSupportOffCanvas();
						
						// Show success alert after offcanvas closes (wait for animation)
						setTimeout(() => {
							Swal.fire({
								title: 'Ticket Created!',
								text: 'Your support ticket #' + response.data.ticket_number + ' has been created successfully.',
								icon: 'success',
								confirmButtonColor: '#58c080',
								timer: 3000,
								timerProgressBar: true,
								allowOutsideClick: false,
								allowEscapeKey: false
							}).then(() => {
								// Reload tickets list after alert closes
								self.loadAllTickets();
								
								// Reopen offcanvas and show the newly created ticket
								self.openSupportOffCanvas();
								setTimeout(() => {
									self.showTicketDetail(response.data.ticket_id, response.data.ticket_number);
								}, 350);
							});
						}, 350);
					} else {
						Swal.fire({
							title: 'Error',
							text: response.data.message || 'Failed to create ticket. Please try again.',
							icon: 'error',
							confirmButtonColor: '#EF4444'
						});
					}
					$submitBtn.prop('disabled', false).html(originalText);
				},
				error: function() {
					Swal.fire({
						title: 'Error',
						text: 'An error occurred. Please try again.',
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
					$submitBtn.prop('disabled', false).html(originalText);
				}
			});
		},
		
		openSupportOffCanvas: function(ticketId, ticketNumber) {
			const $offcanvas = $('#pixxi-support-offcanvas');
			
			if($offcanvas.length === 0) {
				console.error('Off-canvas element not found!');
				return;
			}
			
			// Remove inline styles that might be blocking
			$offcanvas.removeAttr('style');
			
			// Set display first, then trigger reflow
			$offcanvas.css('display', 'flex');
			
			// Trigger reflow to ensure display is set
			$offcanvas[0].offsetHeight;
			
			// Add active class to trigger animation
			setTimeout(() => {
				$offcanvas.addClass('active');
			}, 10);
			
			// Show ticket list view
			$('#pixxi-tickets-list-view').show();
			$('#pixxi-ticket-detail-view').hide();
			
			// Load all tickets
			this.loadAllTickets();
			
			// If ticket ID provided, show that ticket detail
			if(ticketId && ticketNumber) {
				setTimeout(() => {
					this.showTicketDetail(ticketId, ticketNumber);
				}, 350);
			}
		},
		
		handleBackToTickets: function(e) {
			e.preventDefault();
			$('#pixxi-ticket-detail-view').hide();
			$('#pixxi-tickets-list-view').show();
			
			// Stop polling
			if(this.commentPollInterval) {
				clearInterval(this.commentPollInterval);
				this.commentPollInterval = null;
			}
			
			// Reload tickets list
			this.loadAllTickets();
		},
		
		handleTicketClick: function(e) {
			e.preventDefault();
			const $ticket = $(e.currentTarget);
			const ticketId = $ticket.data('ticket-id');
			const ticketNumber = $ticket.data('ticket-number');
			
			if(ticketId && ticketNumber) {
				this.showTicketDetail(ticketId, ticketNumber);
			}
		},
		
		showTicketDetail: function(ticketId, ticketNumber) {
			$('#pixxi-tickets-list-view').hide();
			$('#pixxi-ticket-detail-view').show();
			
			$('#pixxi-comment-ticket-id').val(ticketId);
			$('#pixxi-ticket-number').text('#' + ticketNumber);
			
			// Mark comments as read when viewing ticket
			this.markCommentsRead(ticketId);
			
			// Load comments
			this.loadTicketComments(ticketId);
			
			// Start polling
			this.startCommentPolling(ticketId);
		},
		
		updateUnreadBadge: function() {
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_get_unread_count',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					if(response.success) {
						const count = parseInt(response.data.unread_count) || 0;
						const $badge = $('#pixxi-support-unread-badge');
						if(count > 0) {
							$badge.text(count > 99 ? '99+' : count).show();
						} else {
							$badge.hide();
						}
					} else {
						console.log('Unread count error:', response.data);
					}
				},
				error: (xhr, status, error) => {
					// Log error for debugging
					console.log('Failed to get unread count:', error, xhr.responseText);
				}
			});
		},
		
		markCommentsRead: function(ticketId) {
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_mark_comments_read',
					nonce: pixxiDashboard.nonce,
					ticket_id: ticketId
				},
				success: () => {
					// Update badge after marking as read
					this.updateUnreadBadge();
				}
			});
		},
		
		loadAllTickets: function() {
			const $ticketsList = $('#pixxi-tickets-list');
			$ticketsList.html('<div class="pixxi-loading-tickets"><span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Loading tickets...</div>');
			
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_get_all_tickets',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					if(response.success) {
						this.renderTicketsList(response.data.tickets);
						// Update unread badge after loading tickets
						this.updateUnreadBadge();
					} else {
						$ticketsList.html('<div class="pixxi-ticket-empty">Failed to load tickets. Please try again.</div>');
					}
				},
				error: () => {
					$ticketsList.html('<div class="pixxi-ticket-empty">An error occurred. Please try again.</div>');
				}
			});
		},
		
		renderTicketsList: function(tickets) {
			const $ticketsList = $('#pixxi-tickets-list');
			$ticketsList.empty();
			
			if(tickets.length === 0) {
				$ticketsList.html('<div class="pixxi-ticket-empty">No support tickets yet. Create your first ticket!</div>');
				return;
			}
			
			tickets.forEach((ticket) => {
				const statusClass = ticket.status === 'open' ? 'open' : (ticket.status === 'closed' ? 'closed' : 'pending');
				const commentCount = parseInt(ticket.comment_count) || 0;
				const lastCommentAt = ticket.last_comment_at ? new Date(ticket.last_comment_at) : null;
				const updatedAt = new Date(ticket.updated_at);
				const hasUnread = parseInt(ticket.has_unread) > 0;
				
				// Format date
				const dateStr = updatedAt.toLocaleDateString() + ' ' + updatedAt.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
				
				// Get preview of subject (fallback to issue_description if subject not available)
				const previewText = (ticket.subject && ticket.subject.trim()) || (ticket.issue_description && ticket.issue_description.trim()) || 'No subject';
				const issuePreview = previewText.length > 100 
					? previewText.substring(0, 100) + '...' 
					: previewText;
				
				const $ticket = $('<div class="pixxi-ticket-item' + (hasUnread ? ' unread' : '') + '" data-ticket-id="' + ticket.ticket_id + '" data-ticket-number="' + ticket.ticket_number + '">' +
					'<div class="pixxi-ticket-item-header">' +
					'<div class="pixxi-ticket-item-title">' +
					'<span class="pixxi-ticket-number-badge">#' + ticket.ticket_number + '</span>' +
					'<span class="pixxi-ticket-status-badge status-' + statusClass + '">' + ticket.status.charAt(0).toUpperCase() + ticket.status.slice(1) + '</span>' +
					'</div>' +
					'<div class="pixxi-ticket-item-date">' + dateStr + '</div>' +
					'</div>' +
					'<div class="pixxi-ticket-item-body">' +
					'<div class="pixxi-ticket-item-preview">' + this.escapeHtml(issuePreview) + '</div>' +
					'<div class="pixxi-ticket-item-meta">' +
					'<span class="pixxi-ticket-comment-count"><span class="dashicons dashicons-format-chat"></span> ' + commentCount + ' comment' + (commentCount !== 1 ? 's' : '') + '</span>' +
					(lastCommentAt ? '<span class="pixxi-ticket-last-comment">Last reply: ' + lastCommentAt.toLocaleDateString() + '</span>' : '') +
					'</div>' +
					'</div>' +
					'</div>');
				
				$ticketsList.append($ticket);
			});
		},
		
		loadTicketComments: function(ticketId) {
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_get_ticket_comments',
					nonce: pixxiDashboard.nonce,
					ticket_id: ticketId
				},
			success: (response) => {
				if(response.success) {
					$('#pixxi-ticket-number').text('#' + response.data.ticket_number);
					$('#pixxi-ticket-status').text(response.data.status).attr('data-status', response.data.status);
					
					// Display subject in header (create header info if it doesn't exist)
					let $headerInfo = $('#pixxi-ticket-header-info');
					if($headerInfo.length === 0) {
						// Insert header info at the beginning of the offcanvas body
						$headerInfo = $('<div id="pixxi-ticket-header-info" class="pixxi-ticket-header-info"></div>');
						$('#pixxi-ticket-detail-view .pixxi-offcanvas-body').prepend($headerInfo);
					}
					
					const subject = response.data.subject || response.data.issue_description || 'No subject';
					$headerInfo.html('<h2>' + this.escapeHtml(subject) + '</h2>');
					
					// Store previous comment count
					const previousCount = $('#pixxi-support-comments .pixxi-comment-item').length;
					
					this.renderComments(response.data.comments);
					
					// Play notification sound if new comments received (more than before)
					const currentCount = response.data.comments ? response.data.comments.length : 0;
					if(currentCount > previousCount && previousCount > 0) {
						// Play sound after a short delay
						setTimeout(() => {
							this.playNotificationSound();
						}, 100);
					}
					
					// Update unread badge after loading comments
					this.updateUnreadBadge();
				}
			}
			});
		},
		
		playNotificationSound: function() {
			const audio = document.getElementById('pixxi-notification-sound');
			if(audio) {
				// Reset audio to beginning
				audio.currentTime = 0;
				// Try to play
				const playPromise = audio.play();
				if(playPromise !== undefined) {
					playPromise.catch(err => {
						// Autoplay was prevented - try again after user interaction
						console.log('Audio play failed, trying alternative method:', err);
						// Create a new audio element and play it
						const newAudio = new Audio(audio.src);
						newAudio.volume = 0.5;
						newAudio.play().catch(e => {
							console.log('Alternative audio play also failed:', e);
						});
					});
				}
			} else {
				console.log('Audio element not found');
			}
		},
		
		renderComments: function(comments) {
			const $commentsList = $('#pixxi-support-comments');
			$commentsList.empty();
			
			if(comments.length === 0) {
				$commentsList.html('<div class="pixxi-comment-empty">No comments yet. Start the conversation!</div>');
				return;
			}
			
			comments.forEach((comment) => {
				const isAdmin = comment.user_type === 'admin';
				const commentClass = isAdmin ? 'pixxi-comment-admin' : 'pixxi-comment-user';
				const commentDate = new Date(comment.created_at).toLocaleString();
				
				const $comment = $('<div class="pixxi-comment ' + commentClass + '">' +
					'<div class="pixxi-comment-header">' +
					'<span class="pixxi-comment-author">' + (isAdmin ? 'Support Team' : 'You') + '</span>' +
					'<span class="pixxi-comment-date">' + commentDate + '</span>' +
					'</div>' +
					'<div class="pixxi-comment-body">' + this.escapeHtml(comment.message) + '</div>' +
					'</div>');
				
				$commentsList.append($comment);
			});
			
			// Scroll to bottom
			$commentsList.scrollTop($commentsList[0].scrollHeight);
		},
		
		escapeHtml: function(text) {
			const map = {
				'&': '&amp;',
				'<': '&lt;',
				'>': '&gt;',
				'"': '&quot;',
				"'": '&#039;'
			};
			return text.replace(/[&<>"']/g, (m) => map[m]);
		},
		
		handleCommentFormSubmit: function(e) {
			e.preventDefault();
			const $form = $(this);
			const ticketId = $('#pixxi-comment-ticket-id').val();
			const message = $('#pixxi-comment-message').val().trim();
			
			if(!message) {
				return;
			}
			
			const $submitBtn = $form.find('button[type="submit"]');
			const originalText = $submitBtn.html();
			$submitBtn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Sending...');
			
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_add_ticket_comment',
					nonce: pixxiDashboard.nonce,
					ticket_id: ticketId,
					message: message
				},
				success: (response) => {
					if(response.success) {
						$('#pixxi-comment-message').val('');
						this.renderComments(response.data.comments);
						// Play notification sound after a short delay to ensure audio is ready
						setTimeout(() => {
							this.playNotificationSound();
						}, 100);
					} else {
						Swal.fire({
							title: 'Error',
							text: response.data.message || 'Failed to send message.',
							icon: 'error',
							confirmButtonColor: '#EF4444'
						});
					}
					$submitBtn.prop('disabled', false).html(originalText);
				},
				error: () => {
					Swal.fire({
						title: 'Error',
						text: 'An error occurred. Please try again.',
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
					$submitBtn.prop('disabled', false).html(originalText);
				}
			});
		},
		
		startCommentPolling: function(ticketId) {
			// Clear existing interval
			if(this.commentPollInterval) {
				clearInterval(this.commentPollInterval);
			}
			
			// Poll every 30 seconds
			this.commentPollInterval = setInterval(() => {
				this.loadTicketComments(ticketId);
				// Update unread badge during polling
				this.updateUnreadBadge();
			}, 30000);
		},
		
		
		handleCronToggle: function() {
			const enabled = $('#pixxi_cron_enabled').is(':checked');
			if(enabled) {
				$('#cron-settings-wrapper').slideDown();
				// Show minute wrapper only if interval is set to minute
				const interval = $('#pixxi_cron_interval').val();
				if(interval === 'minute') {
					$('#cron-minute-wrapper').slideDown();
				}
			} else {
				$('#cron-settings-wrapper').slideUp();
				$('#cron-minute-wrapper').slideUp();
			}
		},
		
		handleCronIntervalChange: function() {
			const interval = $('#pixxi_cron_interval').val();
			const enabled = $('#pixxi_cron_enabled').is(':checked');
			// Only show minute wrapper if cron is enabled AND interval is minute
			if(enabled && interval === 'minute') {
				$('#cron-minute-wrapper').slideDown();
			} else {
				$('#cron-minute-wrapper').slideUp();
			}
		},
		
		initCronToggleState: function() {
			// Ensure minute wrapper is hidden if cron is disabled on page load
			const enabled = $('#pixxi_cron_enabled').is(':checked');
			const interval = $('#pixxi_cron_interval').val();
			// Hide if cron is disabled OR if interval is not minute
			if(!enabled || interval !== 'minute') {
				$('#cron-minute-wrapper').hide();
			}
		},

		handleDownload: function(e) {
			e.preventDefault();
			this.showProgress('Updating properties from API...');
			this.disableButtons(true);

			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_download_properties',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					if (response && response.success) {
						Swal.fire({
							title: 'Success!',
							text: response.data?.message || 'Properties downloaded successfully!',
							icon: 'success',
							confirmButtonColor: '#58c080',
							timer: 3000,
							timerProgressBar: true
						});
						this.updateStats();
						// Automatically trigger update process to insert properties
						setTimeout(() => {
							this.handleUpdate();
						}, 3000);
					} else {
						// Handle error response
						let errorMsg = 'Unknown error';
						if (response && response.data) {
							if (response.data.message) {
								errorMsg = response.data.message;
							} else if (typeof response.data === 'string') {
								errorMsg = response.data;
							}
						}
						Swal.fire({
							title: 'Error',
							text: errorMsg,
							icon: 'error',
							confirmButtonColor: '#EF4444'
						});
					}
					this.hideProgress();
					this.disableButtons(false);
				},
				error: (xhr, status, error) => {
					let errorMsg = error;
					// Try to parse error response
					if (xhr.responseText) {
						try {
							const errorResponse = JSON.parse(xhr.responseText);
							if (errorResponse.data && errorResponse.data.message) {
								errorMsg = errorResponse.data.message;
							}
						} catch (e) {
							// Not JSON, use the response text
							errorMsg = xhr.responseText.substring(0, 200);
						}
					}
					Swal.fire({
						title: 'Error',
						text: errorMsg,
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
					this.hideProgress();
					this.disableButtons(false);
				}
			});
		},

		handleUpdate: function(e) {
			if (e) e.preventDefault();
			this.showProgress('Updating properties...');
			this.disableButtons(true);
			this.updateStatus('Updating properties...', true);

			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_update_properties',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					if (response.success) {
						// Check if process should continue
						if (response.data && response.data.continue) {
							// Process is still running, continue automatically
							setTimeout(() => {
								this.handleUpdate();
							}, 2000);
						} else if (response.data && response.data.message && response.data.message.includes('paused')) {
							// Process was paused
							this.updateButtonVisibility('pause');
							this.updateStatus('Paused', false);
							this.hideProgress();
							this.disableButtons(false);
						} else {
							// Process completed
							Swal.fire({
								title: 'Success!',
								text: response.data?.message || 'Properties updated successfully!',
								icon: 'success',
								confirmButtonColor: '#58c080',
								timer: 3000,
								timerProgressBar: true
							});
							this.updateStats();
							this.updateButtonVisibility('stop');
							this.hideProgress();
							this.disableButtons(false);
							this.updateStatus('Ready to sync', false);
						}
					} else {
						Swal.fire({
							title: 'Error',
							text: response.data?.message || 'Unknown error',
							icon: 'error',
							confirmButtonColor: '#EF4444'
						});
						this.updateButtonVisibility('stop');
						this.hideProgress();
						this.disableButtons(false);
						this.updateStatus('Ready to sync', false);
					}
				},
				error: (xhr, status, error) => {
					Swal.fire({
						title: 'Error',
						text: error || 'An error occurred during update',
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
					this.updateButtonVisibility('stop');
					this.hideProgress();
					this.disableButtons(false);
					this.updateStatus('Ready to sync', false);
				}
			});
		},

		handleFullSync: function(e) {
			e.preventDefault();
			
			Swal.fire({
				title: 'Full Sync Confirmation',
				text: 'This will perform a full sync (download + update). This may take several minutes. Continue?',
				icon: 'warning',
				showCancelButton: true,
				confirmButtonColor: '#58c080',
				cancelButtonColor: '#6b7280',
				confirmButtonText: 'Yes, continue',
				cancelButtonText: 'Cancel',
				reverseButtons: true
			}).then((result) => {
				if (result.isConfirmed) {
					this.showProgress('Starting full sync...');
					this.disableButtons(true);
					this.updateStatus('Full sync in progress...', true);

					$.ajax({
						url: pixxiDashboard.ajax_url,
						type: 'POST',
						data: {
							action: 'pixxi_full_sync',
							nonce: pixxiDashboard.nonce
						},
						success: (response) => {
							if (response.success) {
								Swal.fire({
									title: 'Success!',
									text: 'Full sync started! Updates will continue in the background.',
									icon: 'success',
									confirmButtonColor: '#58c080',
									timer: 3000,
									timerProgressBar: true
								});
								this.updateStats();
								
								// Start update process after download
								setTimeout(() => {
									this.handleUpdate();
								}, 3000);
							} else {
								Swal.fire({
									title: 'Error',
									text: response.data?.message || 'Unknown error',
									icon: 'error',
									confirmButtonColor: '#EF4444'
								});
								this.hideProgress();
								this.disableButtons(false);
								this.updateButtonVisibility('stop');
								this.updateStatus('Ready to sync', false);
							}
						},
						error: (xhr, status, error) => {
							Swal.fire({
								title: 'Error',
								text: error || 'An error occurred during full sync',
								icon: 'error',
								confirmButtonColor: '#EF4444'
							});
							this.hideProgress();
							this.disableButtons(false);
							this.updateStatus('Ready to sync', false);
						}
					});
				}
			});
		},

		handleSettingsSave: function(e) {
			// Let the form submit normally, just show a loading state
			const $form = $(e.target);
			const $submitBtn = $form.find('#pixxi_save');
			
			// Don't prevent default - let form submit normally
			$submitBtn.prop('disabled', true);
			$submitBtn.html('<span class="pixxi-spinner"></span> Saving...');
			
			// Form will submit and page will reload, so we don't need to re-enable
		},

		showProgress: function(message) {
			$('#sync-progress').slideDown(300);
			$('#progress-text').text(message || 'Processing...');
			$('#progress-fill').css('width', '0%');
			
			// Animate progress bar
			let progress = 0;
			const interval = setInterval(() => {
				progress += Math.random() * 15;
				if (progress > 90) progress = 90;
				$('#progress-fill').css('width', progress + '%');
				
				if (progress >= 90) {
					clearInterval(interval);
				}
			}, 500);
		},

		hideProgress: function() {
			$('#progress-fill').css('width', '100%');
			setTimeout(() => {
				$('#sync-progress').slideUp(300, function() {
					$(this).hide();
					$('#progress-fill').css('width', '0%');
				});
			}, 800);
		},

		showMessage: function(message, type) {
			const $message = $('#sync-message');
			$message.removeClass('success error')
				.addClass(type)
				.text(message)
				.fadeIn(300);
			
			setTimeout(() => {
				$message.fadeOut(300);
			}, 5000);
		},

		updateStatus: function(text, active) {
			$('#sync-status-text').text(text);
			const $dot = $('.pixxi-status-dot');
			if (active) {
				$dot.addClass('pixxi-status-active');
			} else {
				$dot.removeClass('pixxi-status-active');
			}
		},

		disableButtons: function(disable) {
			$('#pixxi-download-btn, #pixxi-full-sync-btn').prop('disabled', disable);
			if (disable) {
				$('#pixxi-download-btn, #pixxi-full-sync-btn').addClass('loading');
			} else {
				$('#pixxi-download-btn, #pixxi-full-sync-btn').removeClass('loading');
			}
		},

		updateStats: function() {
			return $.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				dataType: 'json',
				data: {
					action: 'pixxi_get_stats',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					if (response && response.success && response.data) {
						this.updateStatsFromData(response.data);
					}
				},
				error: (xhr, status, error) => {
					// Check if it's a JSON parse error - try to extract JSON from response
					if (xhr.responseText) {
						try {
							// Try to find JSON in the response (might have BOM or whitespace)
							const cleaned = xhr.responseText.trim().replace(/^\xEF\xBB\xBF/, '');
							const response = JSON.parse(cleaned);
							if (response && response.success && response.data) {
								// If we can parse it, use it
								this.updateStatsFromData(response.data);
								return;
							}
						} catch (e) {
							// Not valid JSON, ignore silently for background updates
						}
					}
					// Silently fail - don't show errors for background updates
				}
			});
		},
		
		updateStatsFromData: function(data) {
			// Update stat values
			$('#stat-total-properties').text(this.formatNumber(data.total_properties));
			$('#stat-synced').text(this.formatNumber(data.total_synced));
			$('#stat-pending').text(this.formatNumber(data.total_pending));
			$('#stat-last-sync').text(data.last_sync);
			
			// Update progress bar visibility based on sync status
			if(data.current_action && data.current_action !== 'stop' && data.current_action !== 'pause') {
				if($('#sync-progress').is(':hidden')) {
					$('#sync-progress').slideDown(300);
				}
				// Update progress text based on action
				let progressText = this.getActionText(data.current_action);
				$('#progress-text').text(progressText);
				// Keep progress bar animated (set to 50% if not already higher)
				const currentWidth = parseInt($('#progress-fill').css('width')) || 0;
				if(currentWidth < 50) {
					$('#progress-fill').css('width', '50%');
				}
			} else {
				// Sync is stopped or paused, hide progress bar
				if($('#sync-progress').is(':visible')) {
					$('#sync-progress').slideUp(300, function() {
						$('#progress-fill').css('width', '0%');
					});
				}
			}
			
			// Update status and button visibility
			if (data.current_action === 'pause') {
				this.updateStatus('Paused', false);
				this.updateButtonVisibility('pause');
			} else if (data.current_action !== 'stop') {
				this.updateStatus(this.getActionText(data.current_action), true);
				this.updateButtonVisibility('running');
			} else {
				this.updateStatus('Ready to sync', false);
				this.updateButtonVisibility('stop');
			}
		},

		getActionText: function(action) {
			const actions = {
				'delete': 'Deleting properties...',
				'update': 'Updating properties...',
				'insert': 'Inserting properties...',
				'stop': 'Ready to sync'
			};
			return actions[action] || 'Processing...';
		},

		formatNumber: function(num) {
			return new Intl.NumberFormat().format(num || 0);
		},

		handlePause: function(e) {
			e.preventDefault();
			
			Swal.fire({
				title: 'Pause Sync?',
				text: 'The sync will pause and can be resumed later.',
				icon: 'question',
				showCancelButton: true,
				confirmButtonColor: '#58c080',
				cancelButtonColor: '#6b7280',
				confirmButtonText: 'Yes, pause',
				cancelButtonText: 'Cancel'
			}).then((result) => {
				if (result.isConfirmed) {
					$.ajax({
						url: pixxiDashboard.ajax_url,
						type: 'POST',
						data: {
							action: 'pixxi_pause_sync',
							nonce: pixxiDashboard.nonce
						},
						success: (response) => {
							if (response && response.success) {
								Swal.fire({
									title: 'Paused',
									text: 'Sync has been paused.',
									icon: 'info',
									confirmButtonColor: '#58c080',
									timer: 2000,
									timerProgressBar: true
								});
								this.updateButtonVisibility('pause');
								this.updateStatus('Paused', false);
							}
						},
						error: () => {
							Swal.fire({
								title: 'Error',
								text: 'Failed to pause sync',
								icon: 'error',
								confirmButtonColor: '#EF4444'
							});
						}
					});
				}
			});
		},

		handleStop: function(e) {
			e.preventDefault();
			
			Swal.fire({
				title: 'Stop Sync?',
				text: 'This will completely stop the sync process. You will need to start over.',
				icon: 'warning',
				showCancelButton: true,
				confirmButtonColor: '#EF4444',
				cancelButtonColor: '#6b7280',
				confirmButtonText: 'Yes, stop',
				cancelButtonText: 'Cancel'
			}).then((result) => {
				if (result.isConfirmed) {
					$.ajax({
						url: pixxiDashboard.ajax_url,
						type: 'POST',
						data: {
							action: 'pixxi_stop_sync',
							nonce: pixxiDashboard.nonce
						},
						success: (response) => {
							if (response && response.success) {
								Swal.fire({
									title: 'Stopped',
									text: 'Sync has been stopped.',
									icon: 'success',
									confirmButtonColor: '#58c080',
									timer: 2000,
									timerProgressBar: true
								});
								this.updateButtonVisibility('stop');
								this.updateStatus('Ready to sync', false);
								this.hideProgress();
								this.disableButtons(false);
							}
						},
						error: () => {
							Swal.fire({
								title: 'Error',
								text: 'Failed to stop sync',
								icon: 'error',
								confirmButtonColor: '#EF4444'
							});
						}
					});
				}
			});
		},

		updateButtonVisibility: function(state) {
			if (state === 'running') {
				$('#pixxi-download-btn, #pixxi-full-sync-btn').hide();
				$('#pixxi-pause-btn').show().html('<span class="pixxi-btn-icon-wrapper"><span class="dashicons dashicons-controls-pause"></span></span><span class="pixxi-btn-text">Pause</span>').off('click').on('click', this.handlePause.bind(this));
				$('#pixxi-stop-btn').show();
			} else if (state === 'pause') {
				$('#pixxi-download-btn, #pixxi-full-sync-btn').hide();
				$('#pixxi-pause-btn').show().html('<span class="pixxi-btn-icon-wrapper"><span class="dashicons dashicons-controls-play"></span></span><span class="pixxi-btn-text">Resume</span>').off('click').on('click', this.handleResume.bind(this));
				$('#pixxi-stop-btn').show();
			} else {
				$('#pixxi-download-btn, #pixxi-full-sync-btn').show();
				$('#pixxi-pause-btn, #pixxi-stop-btn').hide();
			}
		},

		handleLicenseActivation: function(e) {
			e.preventDefault();
			
			Swal.fire({
				title: 'Activate Your License',
				html: `
					<div style="text-align: left; margin: 20px 0;">
						<p style="color: #475569; margin-bottom: 16px;">
							Enter your license key to activate Pixxi Property Importer. You can obtain a license key from your administrator.
						</p>
						<label for="swal-license-key" style="display: block; margin-bottom: 8px; font-weight: 500; color: #0F172A;">
							License Key
						</label>
						<input type="text" 
							   id="swal-license-key" 
							   class="swal2-input" 
							   placeholder="PIXXI-XXXX-XXXX-XXXX"
							   maxlength="24"
							   style="font-family: monospace; text-transform: uppercase; letter-spacing: 1px; width: 100%; margin: 0;">
						<p style="font-size: 13px; color: #64748B; margin-top: 8px;">
							Format: PIXXI-XXXX-XXXX-XXXX (4 characters per segment)
						</p>
					</div>
				`,
				icon: 'info',
				showCancelButton: true,
				confirmButtonColor: '#58c080',
				cancelButtonColor: '#6b7280',
				confirmButtonText: '<span class="dashicons dashicons-yes" style="vertical-align: middle; margin-right: 4px;"></span> Activate',
				cancelButtonText: 'Cancel',
				focusConfirm: false,
				preConfirm: () => {
					const licenseKey = document.getElementById('swal-license-key').value.trim().toUpperCase();
					
					// Validate format
					const licensePattern = /^PIXXI-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;
					if (!licensePattern.test(licenseKey)) {
						Swal.showValidationMessage('Invalid license key format. Please use format: PIXXI-XXXX-XXXX-XXXX');
						return false;
					}
					
					return licenseKey;
				},
				didOpen: () => {
					// Auto-format license key input
					const input = document.getElementById('swal-license-key');
					input.addEventListener('input', function() {
						let val = this.value.toUpperCase().replace(/[^A-Z0-9-]/g, '');
						this.value = val;
					});
					input.focus();
				}
			}).then((result) => {
				if (result.isConfirmed) {
					const licenseKey = result.value;
					
					// Show loading
					Swal.fire({
						title: 'Activating License...',
						html: 'Please wait while we validate your license key.',
						allowOutsideClick: false,
						allowEscapeKey: false,
						showConfirmButton: false,
						didOpen: () => {
							Swal.showLoading();
						}
					});
					
					// AJAX request to activate license
					$.ajax({
						url: pixxiDashboard.ajax_url,
						type: 'POST',
						data: {
							action: 'pixxi_activate_license',
							license_key: licenseKey,
							nonce: pixxiDashboard.license_nonce || pixxiDashboard.nonce
						},
						success: (response) => {
							if (response && response.success) {
								Swal.fire({
									title: 'Success!',
									html: response.data.message || 'License activated successfully!',
									icon: 'success',
									confirmButtonColor: '#10B981',
									confirmButtonText: 'Continue'
								}).then(() => {
									// Reload the page to show activated status
									location.reload();
								});
							} else {
								Swal.fire({
									title: 'Activation Failed',
									text: response.data || 'Failed to activate license. Please check your license key and try again.',
									icon: 'error',
									confirmButtonColor: '#EF4444'
								});
							}
						},
						error: () => {
							Swal.fire({
								title: 'Connection Error',
								text: 'Failed to connect to license server. Please check your internet connection and try again.',
								icon: 'error',
								confirmButtonColor: '#EF4444'
							});
						}
					});
				}
			});
		},

		handleResume: function(e) {
			e.preventDefault();
			
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_resume_sync',
					nonce: pixxiDashboard.nonce
				},
				success: (response) => {
					if (response && response.success) {
						Swal.fire({
							title: 'Resumed',
							text: 'Sync has been resumed.',
							icon: 'success',
							confirmButtonColor: '#58c080',
							timer: 2000,
							timerProgressBar: true
						});
						this.updateButtonVisibility('running');
						this.updateStatus('Resuming...', true);
						// Auto-continue the update process
						setTimeout(() => {
							this.handleUpdate();
						}, 1000);
					}
				},
				error: () => {
					Swal.fire({
						title: 'Error',
						text: 'Failed to resume sync',
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
				}
			});
		},

		autoResumeIfRunning: function() {
			// Check if sync is running and auto-resume
			this.updateStats().done((response) => {
				if (response && response.success && response.data) {
					const action = response.data.current_action;
					if (action && action !== 'stop' && action !== 'pause') {
						// Sync is running, show pause/stop buttons and continue processing
						this.updateButtonVisibility('running');
						this.updateStatus(this.getActionText(action), true);
						// Continue the update process
						this.continueUpdateProcess();
					} else if (action === 'pause') {
						// Sync is paused, show resume button
						this.updateButtonVisibility('pause');
						this.updateStatus('Paused', false);
					}
				}
			});
		},

		continueUpdateProcess: function() {
			// Continuously poll and process updates
			const processUpdate = () => {
				$.ajax({
					url: pixxiDashboard.ajax_url,
					type: 'POST',
					data: {
						action: 'pixxi_update_properties',
						nonce: pixxiDashboard.nonce
					},
				success: (response) => {
					if (response && response.success) {
						// Check response data to see if we should continue
						if (response.data && response.data.continue) {
							// Process is still running, continue after delay
							setTimeout(processUpdate, 2000);
						} else if (response.data && response.data.paused) {
							// Process was paused
							this.updateButtonVisibility('pause');
							this.updateStatus('Paused', false);
							this.hideProgress();
							this.disableButtons(false);
						} else {
							// Check status to determine next step
							this.updateStats().done((statsResponse) => {
								if (statsResponse && statsResponse.success && statsResponse.data) {
									const action = statsResponse.data.current_action;
									if (action && action !== 'stop' && action !== 'pause') {
										// Still running, continue processing
										setTimeout(processUpdate, 2000);
									} else if (action === 'pause') {
										// Process was paused
										this.updateButtonVisibility('pause');
										this.updateStatus('Paused', false);
										this.hideProgress();
										this.disableButtons(false);
									} else {
										// Process completed
										this.updateButtonVisibility('stop');
										this.updateStatus('Ready to sync', false);
										this.hideProgress();
										this.disableButtons(false);
									}
								}
							});
						}
					}
				},
					error: () => {
						// On error, check status and retry if still running
						setTimeout(() => {
							this.updateStats().done((statsResponse) => {
								if (statsResponse && statsResponse.success && statsResponse.data) {
									const action = statsResponse.data.current_action;
									if (action && action !== 'stop' && action !== 'pause') {
										processUpdate();
									}
								}
							});
						}, 5000);
					}
				});
			};
			
			// Start the continuous process
			processUpdate();
		},

		startStatsRefresh: function() {
			// Check for auto-resume on page load
			setTimeout(() => {
				this.autoResumeIfRunning();
			}, 1000);
			
			// Wait a bit before starting auto-refresh to avoid conflicts
			setTimeout(() => {
				// Refresh stats every 5 seconds
				setInterval(() => {
					this.updateStats();
				}, 5000); // Check every 5 seconds
			}, 2000); // Wait 2 seconds before first refresh
		}
	};

	// Initialize on document ready
	$(document).ready(function() {
		Dashboard.init();
		
		// Billing Toggle for Subscriptions Page
		$('.pixxi-billing-option').on('click', function() {
			var billing = $(this).data('billing');
			$('.pixxi-billing-option').removeClass('active');
			$(this).addClass('active');
			
			if (billing === 'monthly') {
				$('.pixxi-plan-price-monthly').show();
				$('.pixxi-plan-price-yearly').hide();
				// Update billing period on subscribe buttons
				$('.pixxi-subscribe-btn').data('billing-period', 'monthly');
			} else {
				$('.pixxi-plan-price-monthly').hide();
				$('.pixxi-plan-price-yearly').show();
				// Update billing period on subscribe buttons
				$('.pixxi-subscribe-btn').data('billing-period', 'yearly');
			}
		});
		
		// Refresh Status Button
		$('#pixxi-refresh-status').on('click', function() {
			var $btn = $(this);
			var originalText = $btn.html();
			$btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Refreshing...');
			
			// Reload the page after a short delay
			setTimeout(function() {
				location.reload();
			}, 1000);
		});
		
		// Cancel subscription button
		$('#pixxi-cancel-subscription').on('click', function() {
			var $btn = $(this);
			var originalText = $btn.html();
			Swal.fire({
				title: 'Cancel subscription?',
				text: 'Your license will be cancelled and you will lose access. This cannot be undone.',
				icon: 'warning',
				showCancelButton: true,
				confirmButtonColor: '#EF4444',
				cancelButtonColor: '#64748B',
				confirmButtonText: 'Yes, cancel subscription',
				cancelButtonText: 'Keep subscription'
			}).then(function(result) {
				if (!result.isConfirmed) return;
				$btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Cancelling...');
				$.ajax({
					url: pixxiDashboard.ajax_url,
					type: 'POST',
					data: {
						action: 'pixxi_cancel_subscription',
						nonce: pixxiDashboard.nonce
					},
					success: function(res) {
						if (res && res.success) {
							Swal.fire({
								title: 'Cancelled',
								text: res.data && res.data.message ? res.data.message : 'Your subscription has been cancelled.',
								icon: 'success',
								confirmButtonColor: '#58c080'
							}).then(function() { location.reload(); });
						} else {
							$btn.prop('disabled', false).html(originalText);
							Swal.fire({ title: 'Error', text: (res && res.data && res.data.message) ? res.data.message : 'Failed to cancel.', icon: 'error', confirmButtonColor: '#EF4444' });
						}
					},
					error: function(xhr) {
						$btn.prop('disabled', false).html(originalText);
						var msg = (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) ? xhr.responseJSON.data.message : 'Failed to cancel subscription';
						Swal.fire({ title: 'Error', text: msg, icon: 'error', confirmButtonColor: '#EF4444' });
					}
				});
			});
		});
		
		// Subscribe button handler
		$(document).on('click', '.pixxi-subscribe-btn', function() {
			var $btn = $(this);
			var planId = $btn.data('plan-id');
			var domain = $btn.data('domain');
			var billingPeriod = $btn.data('billing-period') || 'monthly';
			var originalText = $btn.html();
			
			$btn.prop('disabled', true).html('Creating checkout...');
			
			// Call API to create Stripe checkout session
			$.ajax({
				url: 'https://devmizan.com/wp-json/pixxi-license/v1/create-checkout',
				type: 'POST',
				contentType: 'application/json',
				data: JSON.stringify({
					plan_id: planId,
					domain: domain,
					billing_period: billingPeriod
				}),
				success: function(response) {
					if (response && response.success && response.checkout_url) {
						// Redirect to Stripe checkout
						window.location.href = response.checkout_url;
					} else {
						$btn.prop('disabled', false).html(originalText);
						Swal.fire({
							title: 'Error',
							text: 'Failed to create checkout session',
							icon: 'error',
							confirmButtonColor: '#EF4444'
						});
					}
				},
				error: function(xhr, status, error) {
					$btn.prop('disabled', false).html(originalText);
					var errorMsg = 'Failed to create checkout session';
					if (xhr.responseJSON && xhr.responseJSON.message) {
						errorMsg = xhr.responseJSON.message;
					}
					Swal.fire({
						title: 'Error',
						text: errorMsg,
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
				}
			});
		});
		
		// Retry loading plans button
		$('#pixxi-retry-plans').on('click', function() {
			var $btn = $(this);
			var originalText = $btn.html();
			$btn.prop('disabled', true).html('Loading...');
			
			// Clear plans cache and reload page
			$.ajax({
				url: pixxiDashboard.ajax_url,
				type: 'POST',
				data: {
					action: 'pixxi_refresh_license',
					nonce: pixxiDashboard.nonce
				},
				success: function() {
					location.reload();
				},
				error: function() {
					$btn.prop('disabled', false).html(originalText);
					Swal.fire({
						title: 'Error',
						text: 'Failed to refresh plans',
						icon: 'error',
						confirmButtonColor: '#EF4444'
					});
				}
			});
		});
	});

})(jQuery);

