<?php
/**
 * Plugin Name: Pixxi Property Importer
 * Plugin URI:  https://devmizan.com
 * Description: Import properties and insert/update in the Houzez Theme from PIXXI CRM in an automatic CRON Scheduling.
 * Version: 1.0.0
 * Author: Dev Mizan
 * Author URI:  https://devmizan.com
 * License: GPL-3.0+
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 */
set_time_limit(300);

define ('PIXXI_PATH', plugin_dir_path( __FILE__ ));
define ('PIXXI_URL', plugin_dir_url( __FILE__ ));
define ('PIXXI_LOCAL', PIXXI_PATH.'data/properties_data.json');
define ('PIXXI_DOWN', PIXXI_PATH.'data/download.txt');
define ('PIXXI_UPDATE', PIXXI_PATH.'data/update.txt');
define ('PIXXI_TABLE', 'wp_properties_import');
require_once plugin_dir_path( __FILE__ ) . 'includes/pixxi-property-class.php';
require_once plugin_dir_path( __FILE__ ) . 'includes/class-license-api-client.php';
require_once plugin_dir_path( __FILE__ ) . 'includes/class-feature-gates.php';
/***********************************************************************************************************************************************/

add_action('admin_enqueue_scripts', 'pixxi_dashboard_scripts');

function pixxi_dashboard_scripts($hook) {
	// Check if we're on one of our plugin pages
	$current_page = isset($_GET['page']) ? $_GET['page'] : '';
	$valid_pages = array('pixxi-property-importer', 'pixxi-subscriptions', 'pixxi-view-all-data');
	if (!in_array($current_page, $valid_pages)) {
		return;
	}
	
	$version = time();
	wp_enqueue_style("pixxi_dashboard_style", plugin_dir_url(__FILE__) . 'css/dashboard.css', $version, false);
	// Enqueue SweetAlert2
	wp_enqueue_style("sweetalert2", 'https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css', array(), '11');
	wp_enqueue_script("sweetalert2", 'https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.js', array(), '11', false);
	
	$script_deps = array('jquery', 'sweetalert2');
	
	wp_enqueue_script("pixxi_dashboard_script", plugin_dir_url(__FILE__) . 'js/dashboard.js', $script_deps, $version, false);
	
	wp_localize_script('pixxi_dashboard_script', 'pixxiDashboard', array(
		'ajax_url' => admin_url('admin-ajax.php'),
		'nonce' => wp_create_nonce('pixxi_dashboard_nonce'),
		'license_nonce' => wp_create_nonce('pixxi_license_nonce'),
		'notification_sound' => plugin_dir_url(__FILE__) . '../pixxi-license-manager/notification.mp3'
	));
}
// on plugin activation create a custom table that stores the jobs in wordpress database
register_activation_hook(__FILE__, 'pixxi_plugin_create_db');
function pixxi_plugin_create_db()
{

    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    $table_name = PIXXI_TABLE;
    $sql = "CREATE TABLE $table_name (
			`id` int(11) NOT NULL AUTO_INCREMENT,
			`pid` int(11) NOT NULL,
			`sku` varchar(255) NOT NULL,
			`data` longtext COLLATE utf8mb4_unicode_ci NOT NULL,
			`status` tinyint(1) NOT NULL,
			`updated` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY id (id)
			) $charset_collate;";
    require_once (ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
    
    // Create support tickets table
    $support_tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
    $support_tickets_sql = "CREATE TABLE $support_tickets_table (
        `ticket_id` int(11) NOT NULL AUTO_INCREMENT,
        `ticket_number` varchar(50) NOT NULL,
        `user_name` varchar(255) NOT NULL,
        `site_url` varchar(500) NOT NULL,
        `login_creds` text,
        `issue_description` text NOT NULL,
        `wp_version` varchar(50),
        `license_key` varchar(255),
        `status` varchar(50) DEFAULT 'open',
        `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
        `updated_at` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`ticket_id`),
        UNIQUE KEY `ticket_number` (`ticket_number`)
    ) $charset_collate;";
    dbDelta($support_tickets_sql);
    
    // Create support comments table
    $support_comments_table = $wpdb->prefix . 'pixxi_support_comments';
    $support_comments_sql = "CREATE TABLE $support_comments_table (
        `comment_id` int(11) NOT NULL AUTO_INCREMENT,
        `ticket_id` int(11) NOT NULL,
        `user_type` varchar(20) NOT NULL,
        `message` text NOT NULL,
        `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`comment_id`),
        KEY `ticket_id` (`ticket_id`)
    ) $charset_collate;";
    dbDelta($support_comments_sql);
    
    // Schedule cron if enabled
    pixxi_activate_cron();
}


//adding menu & settings page
add_action('admin_menu','pixxi_import_products_menu');
function pixxi_import_products_menu(){
	// Add as main menu item (positioned after Posts which is at position 5)
	add_menu_page('Pixxi Property Importer', 'Pixxi Importer', 'manage_options', 'pixxi-property-importer', 'pixxi_import_autos_data', 'dashicons-building', 6);
	
	// Add submenu for viewing all Pixxi data
	add_submenu_page('pixxi-property-importer', 'View All Pixxi Data', 'View All Data', 'manage_options', 'pixxi-view-all-data', 'pixxi_view_all_data_page');
	
	add_submenu_page('pixxi-property-importer', 'Subscriptions', 'Subscriptions', 'manage_options', 'pixxi-subscriptions', 'pixxi_subscriptions_page');
}

//Adding Settings Links
add_filter( 'plugin_action_links_'.plugin_basename( __FILE__ ) , 'pixxi_settings_link');

function pixxi_settings_link( $links ){

	$links[] = '<a href="' .
	admin_url( 'admin.php?page=pixxi-property-importer' ) .'">' . __('Settings') . '</a>';
	return $links;
}
//saving settings page data and cron options
function pixxi_import_autos_data(){
	global $wpdb;
	
	$save_success = false;
	$save_error = '';
	
		if(isset($_POST['pixxi_save']) || isset($_REQUEST['pixxi_save'])){
		// Verify nonce
			if(!isset($_POST['pixxi_settings_nonce']) || !wp_verify_nonce($_POST['pixxi_settings_nonce'], 'pixxi_settings_save')){
			$save_error = 'Security check failed. Please try again.';
		} else {
			// Get form values - preserve URL encoding
			$apiurl_raw = isset($_POST['pixxi_apiurl']) ? stripslashes($_POST['pixxi_apiurl']) : '';
			
			// Extract and preserve the company name encoding
			$apiurl = '';
			if(!empty($apiurl_raw)) {
				// Extract base URL and company name
				if(preg_match('#^(https?://[^/]+/pixxiapi/v1/properties/)(.+)$#', $apiurl_raw, $matches)) {
					$base_url = $matches[1];
					$company_name = $matches[2];
					
					// If company name has spaces, encode them using rawurlencode (uses %20, not +)
					// Check if it's already encoded by looking for %20 or +
					if(strpos($company_name, '%20') === false && strpos($company_name, '+') === false && strpos($company_name, ' ') !== false) {
						// Has spaces but no encoding - encode it with %20
						$company_name = rawurlencode($company_name);
					} elseif(strpos($company_name, '+') !== false) {
						// Has + encoding - convert to %20
						$decoded = urldecode($company_name); // This handles + as spaces
						$company_name = rawurlencode($decoded); // Re-encode with %20
					} elseif(strpos($company_name, '%20') !== false) {
						// Already has %20 - keep as is, but ensure it's properly formatted
						// Decode and re-encode to ensure proper encoding
						$decoded = rawurldecode($company_name);
						$company_name = rawurlencode($decoded);
					}
					
					// Validate base URL
					$base_url = esc_url_raw($base_url);
					$apiurl = $base_url . $company_name;
				} else {
					// If pattern doesn't match, try to sanitize the whole URL
					$apiurl = esc_url_raw($apiurl_raw);
				}
			}
			
			// Set default if empty
			if(empty($apiurl)) {
				$apiurl = 'https://dataapi.pixxicrm.ae/pixxiapi/v1/properties/Woolhawk%20Real%20Estate%20LLC';
			}
			
			$token = isset($_POST['pixxi_token']) ? sanitize_text_field($_POST['pixxi_token']) : '';
			
			// License settings are now domain-based and auto-detected
			// No manual input required
			
			// Check if user has "easy" plan for auto-delete features
			$has_easy_plan = false;
			if (class_exists('Pixxi_License_Feature_Gates')) {
				$has_easy_plan = Pixxi_License_Feature_Gates::has_easy_plan();
			}
			
			// Get automatic delete setting (only if plan allows)
			$auto_delete = 0;
			if ($has_easy_plan && isset($_POST['pixxi_auto_delete'])) {
				$auto_delete = (int)$_POST['pixxi_auto_delete'];
			}
			
			// Get taxonomy cleanup setting (only if plan allows)
			$cleanup_orphaned_taxonomy = 0;
			if ($has_easy_plan && isset($_POST['pixxi_cleanup_orphaned_taxonomy'])) {
				$cleanup_orphaned_taxonomy = (int)$_POST['pixxi_cleanup_orphaned_taxonomy'];
			}
			
			// Get cron settings
			$cron_enabled = isset($_POST['pixxi_cron_enabled']) ? (int)$_POST['pixxi_cron_enabled'] : 0;
			$cron_interval = isset($_POST['pixxi_cron_interval']) ? sanitize_text_field($_POST['pixxi_cron_interval']) : 'hourly';
			$cron_minute = isset($_POST['pixxi_cron_minute']) ? (int)$_POST['pixxi_cron_minute'] : 5;
			
			$pixxi_opt_val = array(
				"pixxi_apiurl" => $apiurl,
				"pixxi_token" => $token,
				// Keep old keys for backward compatibility
				"lw_fsapi" => $apiurl,
				"lw_pixiapi" => $token,
				// Automatic delete setting
				"pixxi_auto_delete" => $auto_delete,
				// Taxonomy cleanup setting
				"pixxi_cleanup_orphaned_taxonomy" => $cleanup_orphaned_taxonomy,
				// Cron settings
				"pixxi_cron_enabled" => $cron_enabled,
				"pixxi_cron_interval" => $cron_interval,
				"pixxi_cron_minute" => $cron_minute,
			);
			
			// Preserve existing data mappings if they exist
			$opt = get_option('pixxi_opt_val');
			if(!$opt) {
		$opt = get_option('lw_opt_val');
			}
			$existing_option = json_decode($opt, true);
			if(is_array($existing_option) && isset($existing_option['pixxi_field_mappings'])) {
				$pixxi_opt_val['pixxi_field_mappings'] = $existing_option['pixxi_field_mappings'];
			}
			
			$saved = update_option('pixxi_opt_val', json_encode($pixxi_opt_val));
		$save_success = true;
			
			// Schedule or unschedule cron based on settings
			pixxi_schedule_cron($cron_enabled, $cron_interval, $cron_minute);
			
			// Reload options immediately after save
			$pixxi_option = $pixxi_opt_val;
		}
	}
	
	if(!$save_success) {
	// Load existing options
		$opt = get_option('pixxi_opt_val');
		// Migration: Check for old option name
		if(!$opt) {
	$opt = get_option('lw_opt_val');
			if($opt) {
				update_option('pixxi_opt_val', $opt);
			}
		}
		$pixxi_option = json_decode($opt, true);
		
		// Initialize if empty
		if(!is_array($pixxi_option)) {
			$pixxi_option = array();
		}
		
		// Backward compatibility: migrate old keys to new keys
		if(empty($pixxi_option['pixxi_apiurl']) && !empty($pixxi_option['lw_apiurl'])) {
			$pixxi_option['pixxi_apiurl'] = $pixxi_option['lw_apiurl'];
		}
		if(empty($pixxi_option['pixxi_apiurl']) && !empty($pixxi_option['lw_fsapi'])) {
			$pixxi_option['pixxi_apiurl'] = $pixxi_option['lw_fsapi'];
		}
		if(empty($pixxi_option['pixxi_token']) && !empty($pixxi_option['lw_token'])) {
			$pixxi_option['pixxi_token'] = $pixxi_option['lw_token'];
		}
		if(empty($pixxi_option['pixxi_token']) && !empty($pixxi_option['lw_pixiapi'])) {
			$pixxi_option['pixxi_token'] = $pixxi_option['lw_pixiapi'];
		}
		
		// Backward compatibility: if API URL is base URL and company endpoint exists, construct full URL
		if(!empty($pixxi_option['pixxi_apiurl']) && !empty($pixxi_option['lw_cmenp'])) {
			$api_url = rtrim($pixxi_option['pixxi_apiurl'], '/');
			$company_encoded = urlencode($pixxi_option['lw_cmenp']);
			// Check if URL already contains /v1/properties/ to avoid duplication
			if(strpos($api_url, '/v1/properties/') === false) {
				$pixxi_option['pixxi_apiurl'] = $api_url . '/v1/properties/' . $company_encoded;
			}
		}
	}
	
	// Get statistics
	$total_properties = $wpdb->get_var("SELECT COUNT(*) FROM " . PIXXI_TABLE);
	$total_synced = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}postmeta WHERE meta_key='_xm_api_sku'");
	$total_pending = $total_properties - $total_synced;
	
	// Get last sync time
	$last_download = file_exists(PIXXI_DOWN) ? file_get_contents(PIXXI_DOWN) : 0;
	$last_update = file_exists(PIXXI_UPDATE) ? file_get_contents(PIXXI_UPDATE) : 0;
	$last_sync = max($last_download, $last_update);
	
	// Get API status
	$api_status_data = pixxi_check_api_status();
	$api_status = $api_status_data['status'];
	$api_last_check = $api_status_data['last_check'];
	
	// Get API info
	$api_info = get_option('pixxi_api_info');
	// Migration: Check for old option name
	if(!$api_info) {
		$api_info = get_option('lw_api_info');
		if($api_info) {
			update_option('pixxi_api_info', $api_info);
		}
	}
	$api_data = $api_info ? json_decode($api_info, true) : array('action' => 'stop', 'chunk' => 0);
	
	// Auto-resume if paused or running
	if($api_data && is_array($api_data) && isset($api_data['action'])) {
		if($api_data['action'] == 'pause') {
			// Keep paused state - user can resume manually
		} elseif($api_data['action'] != 'stop') {
			// Auto-resume if running (delete, update, or insert)
			// This will be handled by JavaScript polling
		}
	}
	
	// Get recent properties
	// First, update any synced properties that have post IDs but missing updated timestamps
	$synced_properties = $wpdb->get_results("SELECT * FROM " . PIXXI_TABLE . " WHERE pid > 0 AND (updated = 0 OR updated = '0000-00-00 00:00:00' OR updated IS NULL)", ARRAY_A);
	foreach($synced_properties as $synced_prop) {
		if(!empty($synced_prop['pid'])) {
			$post_modified = get_post_field('post_modified', $synced_prop['pid']);
			if($post_modified && $post_modified != '0000-00-00 00:00:00') {
				$timestamp = strtotime($post_modified);
				if($timestamp !== false) {
					$wpdb->update(
						PIXXI_TABLE,
						array('updated' => $timestamp),
						array('id' => $synced_prop['id']),
						array('%d'),
						array('%d')
					);
				}
			}
		}
	}
	
	// Get recent properties
	$recent_properties = $wpdb->get_results("SELECT * FROM " . PIXXI_TABLE . " ORDER BY updated DESC LIMIT 10", ARRAY_A);
	
	// Get import history
	$import_history = get_option('pixxi_import_history');
	// Migration: Check for old option name
	if(!$import_history) {
		$import_history = get_option('lw_import_history');
		if($import_history) {
			update_option('pixxi_import_history', $import_history);
		}
	}
	// Handle both string and array formats
	if(is_array($import_history)) {
		$history_data = $import_history;
	} elseif(is_string($import_history) && !empty($import_history)) {
		$history_data = json_decode($import_history, true);
		if(!is_array($history_data)) {
			$history_data = array();
		}
	} else {
		$history_data = array();
	}
	
	// Build history array from available data
	$history_list = array();
	if($last_download > 0) {
		$history_list[] = array(
			'date' => $last_download,
			'type' => 'Manual',
			'action' => 'Download',
			'status' => 'Success'
		);
	}
	if($last_update > 0) {
		$history_list[] = array(
			'date' => $last_update,
			'type' => 'Manual',
			'action' => 'Update',
			'status' => 'Success'
		);
	}
	// Add stored history
	if(is_array($history_data) && !empty($history_data)) {
		// Filter out invalid entries and ensure they have required keys
		foreach($history_data as $entry) {
			if(is_array($entry) && isset($entry['date']) && isset($entry['type']) && isset($entry['action']) && isset($entry['status'])) {
				$history_list[] = $entry;
			}
		}
	}
	// Sort by date descending and limit to 10
	usort($history_list, function($a, $b) {
		$a_date = isset($a['date']) ? (int)$a['date'] : 0;
		$b_date = isset($b['date']) ? (int)$b['date'] : 0;
		return $b_date - $a_date;
	});
	$history_list = array_slice($history_list, 0, 10);
	
	?>
	<!-- Audio notification -->
	<audio id="pixxi-notification-sound" preload="auto" style="display: none;">
		<source src="<?php echo esc_url(plugin_dir_url(__FILE__) . '../pixxi-license-manager/notification.mp3'); ?>" type="audio/mpeg">
		Your browser does not support the audio element.
	</audio>
	
	<div class="pixxi-dashboard-wrapper">
		<div class="pixxi-dashboard-header">
			<div class="pixxi-header-content">
				<div class="pixxi-header-left">
					<h1 class="pixxi-dashboard-title">Pixxi Property Importer Dashboard</h1>
					<p class="pixxi-dashboard-subtitle">Manage property imports and synchronization</p>
				</div>
				<div class="pixxi-header-actions">
					<button type="button" id="pixxi-support-btn" class="pixxi-header-btn pixxi-btn-secondary" style="margin-right: 8px; position: relative;">
						<span class="pixxi-btn-icon-wrapper">
							<span class="dashicons dashicons-sos"></span>
						</span>
						<span class="pixxi-btn-text">Support</span>
						<span id="pixxi-support-unread-badge" class="pixxi-unread-badge" style="display: none;">0</span>
					</button>
					<button type="button" id="pixxi-download-btn" class="pixxi-header-btn pixxi-btn-primary">
						<span class="pixxi-btn-icon-wrapper">
							<span class="dashicons dashicons-download"></span>
						</span>
						<span class="pixxi-btn-text">Update Properties</span>
					</button>
					<button type="button" id="pixxi-full-sync-btn" class="pixxi-header-btn pixxi-btn-success">
						<span class="pixxi-btn-icon-wrapper">
							<span class="dashicons dashicons-update-alt"></span>
						</span>
						<span class="pixxi-btn-text">Full Sync</span>
					</button>
					<button type="button" id="pixxi-pause-btn" class="pixxi-header-btn pixxi-btn-warning" style="display: none;">
						<span class="pixxi-btn-icon-wrapper">
							<span class="dashicons dashicons-controls-pause"></span>
						</span>
						<span class="pixxi-btn-text">Pause</span>
					</button>
					<button type="button" id="pixxi-stop-btn" class="pixxi-header-btn pixxi-btn-danger" style="display: none;">
						<span class="pixxi-btn-icon-wrapper">
							<span class="dashicons dashicons-controls-stop"></span>
						</span>
						<span class="pixxi-btn-text">Stop</span>
					</button>
				</div>
			</div>
		</div>

		<!-- Top Progress Bar -->
		<div id="sync-progress" class="pixxi-top-progress" style="<?php echo ($api_data['action'] != 'stop' && $api_data['action'] != 'pause') ? '' : 'display: none;'; ?>">
			<div class="pixxi-progress-bar-wrapper">
				<div class="pixxi-progress-text" id="progress-text">
					<?php 
					if($api_data['action'] == 'delete') echo 'Deleting properties...';
					elseif($api_data['action'] == 'update') echo 'Updating properties...';
					elseif($api_data['action'] == 'insert') echo 'Inserting properties...';
					else echo 'Processing...';
					?>
				</div>
				<div class="pixxi-progress-bar">
					<div class="pixxi-progress-fill" id="progress-fill" style="width: <?php echo ($api_data['action'] != 'stop' && $api_data['action'] != 'pause') ? '50%' : '0%'; ?>;"></div>
				</div>
			</div>
		</div>

		<div class="pixxi-dashboard-content">
			<?php
			// Display license status card
			if (class_exists('Pixxi_License_API_Client')) {
				$api_client = new Pixxi_License_API_Client();
				$license_data = $api_client->validate_license();
				$is_license_valid = $license_data['valid'] ?? false;
			?>
			<div class="pixxi-stat-card pixxi-license-status-card" style="grid-column: 1 / -1; margin-bottom: 24px; <?php echo !$is_license_valid ? 'border-left: 4px solid #EF4444;' : 'border-left: 4px solid #10B981;'; ?>">
				<div class="pixxi-stat-icon">
					<span class="dashicons dashicons-<?php echo $is_license_valid ? 'yes-alt' : 'warning'; ?>" style="color: <?php echo $is_license_valid ? '#10B981' : '#EF4444'; ?>;"></span>
				</div>
				<div class="pixxi-stat-content" style="flex: 1;">
					<div class="pixxi-stat-value" style="color: <?php echo $is_license_valid ? '#10B981' : '#EF4444'; ?>; font-size: 20px;">
						<?php echo $is_license_valid ? 'License Active' : 'License Invalid'; ?>
					</div>
					<?php if ($is_license_valid): ?>
					<div class="pixxi-stat-label">
						<strong>Plan:</strong> <?php echo esc_html(ucfirst($license_data['plan_id'] ?? 'None')); ?>
						<?php if (isset($license_data['limits']['max_properties'])): ?>
							| <strong>Max Properties:</strong> <?php echo $license_data['limits']['max_properties'] == -1 ? 'Unlimited' : number_format($license_data['limits']['max_properties']); ?>
						<?php endif; ?>
						<?php if (!empty($license_data['expires_at'])): ?>
							| <strong>Expires:</strong> <?php echo esc_html(date('F j, Y', strtotime($license_data['expires_at']))); ?>
						<?php endif; ?>
					</div>
					<?php else: ?>
					<div class="pixxi-stat-label" style="color: #475569;">
						<p style="margin: 8px 0 12px;">
							Your license is not active. Please enter a valid license key to activate the plugin features.
						</p>
						<button type="button" class="button button-primary" id="pixxi-upgrade-now-btn">
							<span class="dashicons dashicons-unlock" style="vertical-align: middle;"></span>
							Activate License
						</button>
					</div>
					<?php endif; ?>
				</div>
				<?php if ($is_license_valid): ?>
				<div class="pixxi-license-actions">
					<button type="button" class="button button-secondary" id="pixxi-refresh-license-status" title="Refresh License Status">
						<span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
						Refresh
					</button>
				</div>
				<?php endif; ?>
			</div>
			<?php } ?>
			
			<!-- Statistics Cards -->
			<div class="pixxi-stats-grid">
				<div class="pixxi-stat-card">
					<div class="pixxi-stat-icon">
						<span class="dashicons dashicons-building"></span>
					</div>
					<div class="pixxi-stat-content">
						<div class="pixxi-stat-value" id="stat-total-properties"><?php echo number_format($total_properties); ?></div>
						<div class="pixxi-stat-label">Total Properties</div>
					</div>
				</div>
				
				<div class="pixxi-stat-card">
					<div class="pixxi-stat-icon">
						<span class="dashicons dashicons-yes-alt"></span>
					</div>
					<div class="pixxi-stat-content">
						<div class="pixxi-stat-value" id="stat-synced"><?php echo number_format($total_synced); ?></div>
						<div class="pixxi-stat-label">Synced Properties</div>
					</div>
				</div>
				
				<div class="pixxi-stat-card">
					<div class="pixxi-stat-icon">
						<span class="dashicons dashicons-clock"></span>
					</div>
					<div class="pixxi-stat-content">
						<div class="pixxi-stat-value" id="stat-pending"><?php echo number_format($total_pending); ?></div>
						<div class="pixxi-stat-label">Pending Sync</div>
					</div>
				</div>
				
				<div class="pixxi-stat-card">
					<div class="pixxi-stat-icon">
						<span class="dashicons dashicons-update"></span>
					</div>
					<div class="pixxi-stat-content">
						<div class="pixxi-stat-value" id="stat-last-sync">
							<?php echo $last_sync ? human_time_diff($last_sync, current_time('timestamp')) . ' ago' : 'Never'; ?>
						</div>
						<div class="pixxi-stat-label">Last Sync</div>
					</div>
				</div>
				
				<div class="pixxi-stat-card">
					<div class="pixxi-stat-icon">
						<span class="dashicons dashicons-<?php echo $api_status == 'online' ? 'yes-alt' : 'dismiss'; ?>"></span>
					</div>
					<div class="pixxi-stat-content">
						<div class="pixxi-stat-value" style="color: <?php echo $api_status == 'online' ? '#10B981' : '#EF4444'; ?>;">
							<?php echo ucfirst($api_status); ?>
						</div>
						<div class="pixxi-stat-label">API Status</div>
						<?php if($api_last_check > 0): ?>
							<div class="pixxi-stat-sublabel" style="font-size: 11px; color: #94A3B8; margin-top: 4px;">
								Last check: <?php echo human_time_diff($api_last_check, current_time('timestamp')); ?> ago
							</div>
						<?php endif; ?>
					</div>
				</div>
			</div>

			<!-- Main Content Grid -->
			<div class="pixxi-main-grid">
				<!-- Left Column -->
			<div class="pixxi-main-left">
					<!-- Settings Card -->
				<div class="pixxi-card">
					<div class="pixxi-card-header">
							<h2 class="pixxi-card-title">API Settings</h2>
							<p class="pixxi-card-description">Configure your PIXXI API credentials</p>
					</div>
					<div class="pixxi-card-body">
							<?php if($save_success): ?>
							<div class="pixxi-sync-message success" style="display: block; margin-bottom: 24px;">
									Settings saved successfully!
								</div>
							<?php elseif(!empty($save_error)): ?>
							<div class="pixxi-sync-message error" style="display: block; margin-bottom: 24px;">
									<?php echo esc_html($save_error); ?>
								</div>
							<?php endif; ?>
							<form name="pixxi_import" id="pixxi_import" method="post" action="<?php echo admin_url('admin.php?page=pixxi-property-importer'); ?>">
								<?php wp_nonce_field('pixxi_settings_save', 'pixxi_settings_nonce'); ?>
								<div class="pixxi-form-group">
									<label for="pixxi_apiurl" class="pixxi-label">API Endpoint URL</label>
									<input type="text" id="pixxi_apiurl" name="pixxi_apiurl" class="pixxi-input" 
										value="<?php echo isset($pixxi_option['pixxi_apiurl']) && !empty($pixxi_option['pixxi_apiurl']) ? esc_attr($pixxi_option['pixxi_apiurl']) : (isset($pixxi_option['lw_apiurl']) && !empty($pixxi_option['lw_apiurl']) ? esc_attr($pixxi_option['lw_apiurl']) : ''); ?>" 
										placeholder="https://dataapi.pixxicrm.ae/pixxiapi/v1/properties/YourCompanyName">
									<p class="pixxi-field-description">The complete API endpoint URL including company name (e.g., https://dataapi.pixxicrm.ae/pixxiapi/v1/properties/Woolhawk%20Real%20Estate%20LLC)</p>
								</div>
								
								<div class="pixxi-form-group">
									<label for="pixxi_token" class="pixxi-label">API Token</label>
									<input type="text" id="pixxi_token" name="pixxi_token" class="pixxi-input" 
										value="<?php echo isset($pixxi_option['pixxi_token']) ? esc_attr($pixxi_option['pixxi_token']) : (isset($pixxi_option['lw_token']) ? esc_attr($pixxi_option['lw_token']) : ''); ?>" 
										placeholder="Enter API token">
									<p class="pixxi-field-description">Enter your PIXXI API token</p>
								</div>
								
								<div class="pixxi-form-divider"></div>
								
								<?php 
								// Check if user has "easy" plan for auto-delete features
								$has_easy_plan = false;
								if (class_exists('Pixxi_License_Feature_Gates')) {
									$has_easy_plan = Pixxi_License_Feature_Gates::has_easy_plan();
								}
								$feature_locked = !$has_easy_plan;
								?>
								
								<div class="pixxi-form-group <?php echo $feature_locked ? 'pixxi-feature-locked' : ''; ?>">
									<div class="pixxi-toggle-wrapper">
										<label for="pixxi_auto_delete" class="pixxi-toggle-label <?php echo $feature_locked ? 'pixxi-toggle-disabled' : ''; ?>">
											<span class="pixxi-toggle-switch">
												<input type="checkbox" id="pixxi_auto_delete" name="pixxi_auto_delete" value="1" 
													<?php checked(isset($pixxi_option['pixxi_auto_delete']) && $pixxi_option['pixxi_auto_delete'] == 1 && !$feature_locked); ?>
													<?php echo $feature_locked ? 'disabled' : ''; ?>>
												<span class="pixxi-toggle-slider"></span>
											</span>
											<span class="pixxi-toggle-text">
												<strong>Automatic Delete</strong>
												<?php if ($feature_locked): ?>
													<span class="pixxi-feature-lock-badge"><span class="dashicons dashicons-lock"></span> Easy Plan Required</span>
												<?php endif; ?>
												<span class="pixxi-toggle-description">Automatically delete properties that are no longer in the API response</span>
											</span>
										</label>
									</div>
									<p class="pixxi-field-description">When enabled, properties that are removed from the API will be automatically deleted from WordPress during sync operations</p>
									<?php if ($feature_locked): ?>
									<p class="pixxi-upgrade-notice"><span class="dashicons dashicons-info"></span> <a href="<?php echo admin_url('admin.php?page=pixxi-subscriptions'); ?>">Upgrade to Easy plan</a> to unlock this feature</p>
									<?php endif; ?>
								</div>
								
								<div class="pixxi-form-group <?php echo $feature_locked ? 'pixxi-feature-locked' : ''; ?>">
									<div class="pixxi-toggle-wrapper">
										<label for="pixxi_cleanup_orphaned_taxonomy" class="pixxi-toggle-label <?php echo $feature_locked ? 'pixxi-toggle-disabled' : ''; ?>">
											<span class="pixxi-toggle-switch">
												<input type="checkbox" id="pixxi_cleanup_orphaned_taxonomy" name="pixxi_cleanup_orphaned_taxonomy" value="1" 
													<?php checked(isset($pixxi_option['pixxi_cleanup_orphaned_taxonomy']) && $pixxi_option['pixxi_cleanup_orphaned_taxonomy'] == 1 && !$feature_locked); ?>
													<?php echo $feature_locked ? 'disabled' : ''; ?>>
												<span class="pixxi-toggle-slider"></span>
											</span>
											<span class="pixxi-toggle-text">
												<strong>Delete Orphaned Taxonomy Terms</strong>
												<?php if ($feature_locked): ?>
													<span class="pixxi-feature-lock-badge"><span class="dashicons dashicons-lock"></span> Easy Plan Required</span>
												<?php endif; ?>
												<span class="pixxi-toggle-description">Automatically delete taxonomy terms with zero associated properties</span>
											</span>
										</label>
									</div>
									<p class="pixxi-field-description">When enabled, taxonomy terms (status, type, features, developers, etc.) with no associated properties will be automatically deleted during sync operations</p>
									<?php if ($feature_locked): ?>
									<p class="pixxi-upgrade-notice"><span class="dashicons dashicons-info"></span> <a href="<?php echo admin_url('admin.php?page=pixxi-subscriptions'); ?>">Upgrade to Easy plan</a> to unlock this feature</p>
									<?php endif; ?>
								</div>
								
								<div class="pixxi-form-divider"></div>
								
								<div class="pixxi-form-group">
									<h3 class="pixxi-section-title">Automatic Sync Schedule</h3>
									<p class="pixxi-field-description">Configure automatic property synchronization</p>
								</div>
								
								<div class="pixxi-form-group">
									<div class="pixxi-toggle-wrapper">
										<label for="pixxi_cron_enabled" class="pixxi-toggle-label">
											<span class="pixxi-toggle-switch">
												<input type="checkbox" id="pixxi_cron_enabled" name="pixxi_cron_enabled" value="1" 
													<?php checked(isset($pixxi_option['pixxi_cron_enabled']) && $pixxi_option['pixxi_cron_enabled'] == 1); ?>>
												<span class="pixxi-toggle-slider"></span>
											</span>
											<span class="pixxi-toggle-text">
												<strong>Enable Automatic Sync</strong>
												<span class="pixxi-toggle-description">Enable automatic property synchronization on schedule</span>
											</span>
										</label>
									</div>
									<p class="pixxi-field-description">When enabled, properties will be automatically synchronized based on the schedule below</p>
								</div>
								
								<div class="pixxi-form-group" id="cron-settings-wrapper" style="<?php echo (isset($pixxi_option['pixxi_cron_enabled']) && $pixxi_option['pixxi_cron_enabled'] == 1) ? '' : 'display: none;'; ?>">
									<label for="pixxi_cron_interval" class="pixxi-label">Sync Interval</label>
									<select id="pixxi_cron_interval" name="pixxi_cron_interval" class="pixxi-select">
										<?php
										// Get allowed sync interval from license
										$allowed_intervals = array();
										if (class_exists('Pixxi_License_Feature_Gates')) {
											$sync_interval = Pixxi_License_Feature_Gates::get_sync_interval_minutes();
											if ($sync_interval <= 5) {
												$allowed_intervals = array('minute', 'hourly', 'daily', 'weekly');
											} elseif ($sync_interval <= 15) {
												$allowed_intervals = array('hourly', 'daily', 'weekly');
											} else {
												$allowed_intervals = array('hourly', 'daily', 'weekly');
											}
										} else {
											$allowed_intervals = array('minute', 'hourly', 'daily', 'weekly');
										}
										
										$intervals = array(
											'minute' => 'Every Minute',
											'hourly' => 'Hourly',
											'daily' => 'Daily',
											'weekly' => 'Weekly'
										);
										
										foreach ($intervals as $value => $label):
											if (in_array($value, $allowed_intervals) || empty($allowed_intervals)):
										?>
										<option value="<?php echo esc_attr($value); ?>" <?php selected(isset($pixxi_option['pixxi_cron_interval']) ? $pixxi_option['pixxi_cron_interval'] : 'hourly', $value); ?>><?php echo esc_html($label); ?></option>
										<?php
											endif;
										endforeach;
										?>
									</select>
									<p class="pixxi-field-description">How often to automatically sync properties (based on your plan)</p>
								</div>
								
								<div class="pixxi-form-group" id="cron-minute-wrapper" style="<?php echo (isset($pixxi_option['pixxi_cron_interval']) && $pixxi_option['pixxi_cron_interval'] == 'minute') ? '' : 'display: none;'; ?>">
									<label for="pixxi_cron_minute" class="pixxi-label">Every X Minutes</label>
									<input type="number" id="pixxi_cron_minute" name="pixxi_cron_minute" class="pixxi-input" min="1" max="60" 
										value="<?php echo isset($pixxi_option['pixxi_cron_minute']) ? esc_attr($pixxi_option['pixxi_cron_minute']) : 5; ?>">
									<p class="pixxi-field-description">Sync every X minutes (1-60)</p>
								</div>
								
								<?php 
								$next_run = pixxi_get_next_cron_run();
								if($next_run): ?>
									<div class="pixxi-form-group">
										<div class="pixxi-info-box">
											<strong>Next Scheduled Run:</strong> <?php echo esc_html($next_run); ?>
										</div>
									</div>
									<?php endif; ?>
									
								<div class="pixxi-form-actions">
									<input type="hidden" name="pixxi_save" value="1">
									<button type="submit" id="pixxi_save" class="pixxi-btn pixxi-btn-primary">
										<span class="dashicons dashicons-saved"></span> Save Settings
									</button>
												</div>
							</form>
											</div>
					</div>

					<!-- Automatic Import History Card -->
					<div class="pixxi-card">
						<div class="pixxi-card-header">
							<h2 class="pixxi-card-title">Automatic Import History</h2>
							<p class="pixxi-card-description">Recent synchronization history</p>
						</div>
						<div class="pixxi-card-body">
							<div class="pixxi-history-table-wrapper">
								<table class="pixxi-history-table">
									<thead>
										<tr>
											<th>Date & Time</th>
											<th>Type</th>
											<th>Action</th>
											<th>Status</th>
										</tr>
									</thead>
									<tbody>
										<?php if(!empty($history_list)): ?>
											<?php foreach($history_list as $history): ?>
												<tr>
													<td><?php echo date('Y-m-d H:i:s', $history['date']); ?></td>
													<td>
														<span class="pixxi-history-type pixxi-history-type-<?php echo strtolower($history['type']); ?>">
															<?php echo esc_html($history['type']); ?>
														</span>
													</td>
													<td><?php echo esc_html($history['action']); ?></td>
													<td>
														<span class="pixxi-history-status pixxi-history-status-<?php echo strtolower($history['status']); ?>">
															<?php echo esc_html($history['status']); ?>
														</span>
													</td>
												</tr>
														<?php endforeach; ?>
										<?php else: ?>
											<tr>
												<td colspan="4" class="pixxi-history-empty">No import history available</td>
											</tr>
													<?php endif; ?>
									</tbody>
								</table>
											</div>
											</div>
										</div>
				</div>

				<!-- Right Column -->
				<div class="pixxi-main-right">
					<!-- Recent Properties Card -->
					<div class="pixxi-card">
						<div class="pixxi-card-header">
							<h2 class="pixxi-card-title">Recent Properties</h2>
							<a href="<?php echo admin_url('edit.php?post_type=property'); ?>" class="pixxi-card-link">View All</a>
						</div>
						<div class="pixxi-card-body-2">
							<div class="pixxi-properties-table-wrapper">
								<table class="pixxi-properties-table">
									<thead>
										<tr>
											<th>Property ID</th>
											<th>Property Name</th>
											<th>Agent Name</th>
											<th>Updated</th>
										</tr>
									</thead>
									<tbody>
										<?php if(!empty($recent_properties)): ?>
											<?php foreach($recent_properties as $prop): 
												$prop_data = json_decode($prop['data'], true);
												
												// Get property post ID from SKU
												$property_post_id = '';
												$property_name = '';
												$property_id = '';
												$agent_name = '';
												
												// Find property post by SKU
												$property_posts = get_posts(array(
													'post_type' => 'property',
													'meta_key' => '_xm_api_sku',
													'meta_value' => $prop['sku'],
													'posts_per_page' => 1,
													'fields' => 'ids',
												));
												
												if(!empty($property_posts)) {
													$property_post_id = $property_posts[0];
													$property_name = get_the_title($property_post_id);
													$property_id = get_post_meta($property_post_id, 'fave_property_id', true);
													
													// Get agent name
													$agent_id = get_post_meta($property_post_id, 'fave_agents', true);
													if(!empty($agent_id) && is_numeric($agent_id)) {
														$agent_name = get_the_title($agent_id);
													}
												}
												
												// Fallback: Get from API data if property not synced yet
												if(empty($property_name) && !empty($prop_data)) {
													$property_name = isset($prop_data['name']) ? $prop_data['name'] : (isset($prop_data['title']) ? $prop_data['title'] : '');
													$property_id = isset($prop_data['propertyId']) ? $prop_data['propertyId'] : '';
													
													// Get agent name from API data
													if(empty($agent_name) && !empty($prop_data)) {
														if(!empty($prop_data['agentName'])) {
															$agent_name = $prop_data['agentName'];
														} elseif(!empty($prop_data['agent']['name'])) {
															$agent_name = $prop_data['agent']['name'];
														} elseif(!empty($prop_data['agent']['agentName'])) {
															$agent_name = $prop_data['agent']['agentName'];
														}
													}
												}
												
												// Get updated time - use database field first, then fallback to post modified date
												$updated_time = 0;
												
												// Check database updated field
												if(isset($prop['updated'])) {
													$updated_value = $prop['updated'];
													
													// Skip if it's clearly empty or invalid
													if($updated_value !== null && $updated_value !== '' && $updated_value !== '0' && $updated_value !== 0 && $updated_value !== '0000-00-00 00:00:00') {
														// Handle both integer timestamps and MySQL TIMESTAMP strings
														if(is_numeric($updated_value)) {
															$updated_time = intval($updated_value);
															// If it's a very small number, it might be a date issue - check if it's a valid timestamp
															if($updated_time < 1000000000) {
																$updated_time = 0; // Invalid timestamp, reset
															}
														} else {
															// Convert MySQL TIMESTAMP string to Unix timestamp
															$updated_time = strtotime($updated_value);
															if($updated_time === false) {
																$updated_time = 0;
															}
														}
													}
												}
												
												// Fallback to post modified date if database field is empty or invalid
												if($updated_time <= 0 && !empty($property_post_id)) {
													$post_modified = get_post_field('post_modified', $property_post_id);
													if($post_modified && $post_modified != '0000-00-00 00:00:00') {
														$updated_time = strtotime($post_modified);
														if($updated_time === false) {
															$updated_time = 0;
														}
													}
												}
												
												// Fallback to API data updateTime if available
												if($updated_time <= 0 && !empty($prop_data)) {
													if(isset($prop_data['updateTime']) && !empty($prop_data['updateTime'])) {
														$updated_time = strtotime($prop_data['updateTime']);
														if($updated_time === false) {
															$updated_time = 0;
														}
													} elseif(isset($prop_data['createTime']) && !empty($prop_data['createTime'])) {
														// Last resort: use createTime
														$updated_time = strtotime($prop_data['createTime']);
														if($updated_time === false) {
															$updated_time = 0;
														}
													}
												}
											?>
												<tr>
													<td><?php echo esc_html($property_id ? $property_id : '-'); ?></td>
													<td><?php echo esc_html($property_name ? $property_name : '-'); ?></td>
													<td><?php echo esc_html($agent_name ? $agent_name : '-'); ?></td>
													<td>
														<?php 
														if($updated_time > 0) {
															echo human_time_diff($updated_time, current_time('timestamp')) . ' ago';
														} else {
															echo 'Never';
														}
														?>
													</td>
												</tr>
									<?php endforeach; ?>
										<?php else: ?>
											<tr>
												<td colspan="4" class="pixxi-empty-state">No properties found</td>
											</tr>
													<?php endif; ?>
									</tbody>
								</table>
											</div>
											</div>
										</div>

					<!-- System Info Card -->
					<div class="pixxi-card">
						<div class="pixxi-card-header">
							<h2 class="pixxi-card-title">System Information</h2>
						</div>
						<div class="pixxi-card-body">
							<div class="pixxi-info-list">
								<div class="pixxi-info-item">
									<span class="pixxi-info-label">Current Action:</span>
									<span class="pixxi-info-value"><?php echo ucfirst($api_data['action']); ?></span>
								</div>
								<div class="pixxi-info-item">
									<span class="pixxi-info-label">Current Chunk:</span>
									<span class="pixxi-info-value"><?php echo $api_data['chunk']; ?></span>
								</div>
								<div class="pixxi-info-item">
									<span class="pixxi-info-label">Chunk Size:</span>
									<span class="pixxi-info-value"><?php echo isset($api_data['size']) ? $api_data['size'] : 'N/A'; ?></span>
								</div>
								<div class="pixxi-info-item">
									<span class="pixxi-info-label">Last Download:</span>
									<span class="pixxi-info-value">
										<?php echo $last_download ? date('Y-m-d H:i:s', $last_download) : 'Never'; ?>
									</span>
								</div>
								<div class="pixxi-info-item">
									<span class="pixxi-info-label">Last Update:</span>
									<span class="pixxi-info-value">
										<?php echo $last_update ? date('Y-m-d H:i:s', $last_update) : 'Never'; ?>
									</span>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>
		</div>
	</div>
	
	<!-- Support Ticket Modal (Hidden - using off-canvas instead) -->
	<div id="pixxi-support-modal" class="pixxi-modal" style="display: none !important; visibility: hidden;">
		<div class="pixxi-modal-overlay"></div>
		<div class="pixxi-modal-content">
			<div class="pixxi-modal-header">
				<h2 class="pixxi-modal-title">Create Support Ticket</h2>
				<button type="button" class="pixxi-modal-close" id="pixxi-support-modal-close">
					<span class="dashicons dashicons-no-alt"></span>
				</button>
			</div>
			<div class="pixxi-modal-body">
				<form id="pixxi-support-form-modal" onsubmit="return false;">
					<div class="pixxi-form-group">
						<label for="support_name" class="pixxi-form-label">Name <span class="pixxi-required">*</span></label>
						<input type="text" id="support_name" name="name" class="pixxi-form-input" value="<?php echo esc_attr(wp_get_current_user()->display_name); ?>" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_site_url" class="pixxi-form-label">Site URL <span class="pixxi-required">*</span></label>
						<input type="url" id="support_site_url" name="site_url" class="pixxi-form-input" value="<?php echo esc_url(home_url()); ?>" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_subject" class="pixxi-form-label">Subject <span class="pixxi-required">*</span></label>
						<input type="text" id="support_subject" name="subject" class="pixxi-form-input" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_email" class="pixxi-form-label">Email <span class="pixxi-required">*</span></label>
						<input type="email" id="support_email" name="email" class="pixxi-form-input" value="<?php echo esc_attr(wp_get_current_user()->user_email); ?>" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_issue" class="pixxi-form-label">Issue Description <span class="pixxi-required">*</span></label>
						<textarea id="support_issue" name="issue_description" class="pixxi-form-textarea" rows="5" placeholder="Please describe your issue in detail..." required></textarea>
					</div>
					<div class="pixxi-form-group">
						<label for="support_login_creds" class="pixxi-form-label">Login Credentials (Optional)</label>
						<textarea id="support_login_creds" name="login_creds" class="pixxi-form-textarea" rows="3" placeholder="If you want us to access your site, please provide login credentials here"></textarea>
					</div>
					<input type="hidden" id="support_wp_version" name="wp_version" value="<?php echo esc_attr(get_bloginfo('version')); ?>">
					<?php
					// Get license key/domain
					$opt = get_option('pixxi_opt_val');
					if(!$opt) {
						$opt = get_option('lw_opt_val');
					}
					$pixxi_option = json_decode($opt, true);
					$license_domain = !empty($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : '';
					?>
					<input type="hidden" id="support_license_key" name="license_key" value="<?php echo esc_attr($license_domain); ?>">
					<div class="pixxi-form-actions">
						<button type="button" class="pixxi-btn pixxi-btn-secondary" id="pixxi-support-cancel">Cancel</button>
						<button type="submit" class="pixxi-btn pixxi-btn-primary" id="pixxi-support-submit">
							<span class="dashicons dashicons-sos"></span> Create Ticket
						</button>
					</div>
				</form>
			</div>
		</div>
	</div>
	
	<!-- Support Ticket Off-Canvas Panel -->
	<div id="pixxi-support-offcanvas" class="pixxi-offcanvas">
		<div class="pixxi-offcanvas-overlay"></div>
		<div class="pixxi-offcanvas-content">
			<!-- Ticket List View -->
			<div id="pixxi-tickets-list-view" class="pixxi-offcanvas-view">
				<div class="pixxi-offcanvas-header">
					<div class="pixxi-offcanvas-title-wrapper">
						<h2 class="pixxi-offcanvas-title">Support Tickets</h2>
					</div>
					<button type="button" class="pixxi-offcanvas-close" id="pixxi-support-offcanvas-close">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div class="pixxi-offcanvas-body">
					<!-- Create New Ticket Form -->
					<div id="pixxi-new-ticket-form-wrapper" class="pixxi-new-ticket-wrapper">
						<button type="button" class="pixxi-btn pixxi-btn-primary pixxi-btn-block" id="pixxi-toggle-new-ticket-form">
							<span class="dashicons dashicons-plus-alt"></span> Create New Ticket
						</button>
						<div id="pixxi-new-ticket-form" style="display: none;">
				<form id="pixxi-support-form" onsubmit="return false;">
					<div class="pixxi-form-group">
						<label for="support_name" class="pixxi-form-label">Name <span class="pixxi-required">*</span></label>
						<input type="text" id="support_name" name="name" class="pixxi-form-input" value="<?php echo esc_attr(wp_get_current_user()->display_name); ?>" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_site_url" class="pixxi-form-label">Site URL <span class="pixxi-required">*</span></label>
						<input type="url" id="support_site_url" name="site_url" class="pixxi-form-input" value="<?php echo esc_url(home_url()); ?>" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_subject" class="pixxi-form-label">Subject <span class="pixxi-required">*</span></label>
						<input type="text" id="support_subject" name="subject" class="pixxi-form-input" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_email" class="pixxi-form-label">Email <span class="pixxi-required">*</span></label>
						<input type="email" id="support_email" name="email" class="pixxi-form-input" value="<?php echo esc_attr(wp_get_current_user()->user_email); ?>" required>
					</div>
					<div class="pixxi-form-group">
						<label for="support_issue" class="pixxi-form-label">Issue Description <span class="pixxi-required">*</span></label>
						<textarea id="support_issue" name="issue_description" class="pixxi-form-textarea" rows="5" placeholder="Please describe your issue in detail..." required></textarea>
					</div>
					<div class="pixxi-form-group">
						<label for="support_login_creds" class="pixxi-form-label">Login Credentials (Optional)</label>
						<textarea id="support_login_creds" name="login_creds" class="pixxi-form-textarea" rows="3" placeholder="If you want us to access your site, please provide login credentials here"></textarea>
					</div>
								<input type="hidden" id="support_wp_version" name="wp_version" value="<?php echo esc_attr(get_bloginfo('version')); ?>">
								<?php
								// Get license key/domain
								$opt = get_option('pixxi_opt_val');
								if(!$opt) {
									$opt = get_option('lw_opt_val');
								}
								$pixxi_option = json_decode($opt, true);
								$license_domain = !empty($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : '';
								?>
								<input type="hidden" id="support_license_key" name="license_key" value="<?php echo esc_attr($license_domain); ?>">
								<div class="pixxi-form-actions">
									<button type="button" class="pixxi-btn pixxi-btn-secondary" id="pixxi-cancel-new-ticket">Cancel</button>
									<button type="submit" class="pixxi-btn pixxi-btn-primary" id="pixxi-support-submit">
										<span class="dashicons dashicons-sos"></span> Create Ticket
									</button>
								</div>
							</form>
						</div>
					</div>
					
					<!-- Tickets List -->
					<div id="pixxi-tickets-list" class="pixxi-tickets-list">
						<!-- Tickets will be loaded here -->
						<div class="pixxi-loading-tickets">
							<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span>
							Loading tickets...
						</div>
					</div>
				</div>
			</div>
			
			<!-- Ticket Detail View -->
			<div id="pixxi-ticket-detail-view" class="pixxi-offcanvas-view" style="display: none;">
				<div class="pixxi-offcanvas-header">
					<div class="pixxi-offcanvas-title-wrapper">
						<button type="button" class="pixxi-back-to-list" id="pixxi-back-to-tickets">
							<span class="dashicons dashicons-arrow-left-alt"></span>
						</button>
						<h2 class="pixxi-offcanvas-title">Ticket Details</h2>
						<span id="pixxi-ticket-number" class="pixxi-ticket-number"></span>
						<span id="pixxi-ticket-status" class="pixxi-ticket-status"></span>
					</div>
					<button type="button" class="pixxi-offcanvas-close" id="pixxi-support-offcanvas-close-detail">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div class="pixxi-offcanvas-body">
					<div id="pixxi-support-comments" class="pixxi-comments-list">
						<!-- Comments will be loaded here -->
					</div>
					<div class="pixxi-comment-input-wrapper">
						<form id="pixxi-comment-form">
							<div class="pixxi-form-group">
								<textarea id="pixxi-comment-message" name="message" class="pixxi-form-textarea" rows="3" placeholder="Type your message here..." required></textarea>
							</div>
							<input type="hidden" id="pixxi-comment-ticket-id" name="ticket_id" value="">
							<button type="submit" class="pixxi-btn pixxi-btn-primary pixxi-btn-block">
								<span class="dashicons dashicons-email-alt"></span> Send Message
							</button>
						</form>
					</div>
				</div>
			</div>
		</div>
	</div>
<?php
}

// Function to extract all field paths from API data
function pixxi_extract_field_paths($data, $prefix = '', $fields = array()) {
	if(is_array($data)) {
		foreach($data as $key => $value) {
			$field_path = $prefix ? $prefix . '.' . $key : $key;
			
			if(is_array($value) && !empty($value) && !isset($value[0])) {
				// Nested object - recurse
				$fields = pixxi_extract_field_paths($value, $field_path, $fields);
			} elseif(is_array($value) && isset($value[0])) {
				// Array - add the array field itself
				$fields[$field_path] = array(
					'type' => 'array',
					'path' => $field_path,
					'example' => is_string($value[0]) ? $value[0] : (is_numeric($value[0]) ? $value[0] : 'Array')
				);
				
				// If first element is an object, extract its fields too
				if(isset($value[0]) && is_array($value[0]) && !isset($value[0][0])) {
					$fields = pixxi_extract_field_paths($value[0], $field_path . '[0]', $fields);
				}
			} else {
				// Simple value
				$fields[$field_path] = array(
					'type' => gettype($value),
					'path' => $field_path,
					'example' => is_string($value) ? (strlen($value) > 50 ? substr($value, 0, 50) . '...' : $value) : (is_numeric($value) ? $value : (is_bool($value) ? ($value ? 'true' : 'false') : ''))
				);
			}
		}
	}
	return $fields;
}

// Function to get API field list from sample data
function pixxi_get_api_field_list() {
	global $wpdb;
	
	// Try to get a sample property from the database
	$sample = $wpdb->get_row("SELECT data FROM " . PIXXI_TABLE . " LIMIT 1", ARRAY_A);
	
	if($sample && !empty($sample['data'])) {
		$property_data = json_decode($sample['data'], true);
		if($property_data && is_array($property_data)) {
			$fields = pixxi_extract_field_paths($property_data);
			ksort($fields);
			return $fields;
		}
	}
	
	// If no data in database, return common fields based on API documentation
	return array(
		'id' => array('type' => 'integer', 'path' => 'id', 'example' => '6591'),
		'propertyId' => array('type' => 'string', 'path' => 'propertyId', 'example' => '100310106591'),
		'name' => array('type' => 'string', 'path' => 'name', 'example' => 'Sama Yas by Aldar'),
		'title' => array('type' => 'string', 'path' => 'title', 'example' => 'Property Title'),
		'price' => array('type' => 'integer', 'path' => 'price', 'example' => '1900000'),
		'description' => array('type' => 'string', 'path' => 'description', 'example' => 'Property description'),
		'regionName' => array('type' => 'string', 'path' => 'regionName', 'example' => 'Yas Island'),
		'cityName' => array('type' => 'string', 'path' => 'cityName', 'example' => 'Abu Dhabi'),
		'developerName' => array('type' => 'string', 'path' => 'developerName', 'example' => 'Aldar'),
		'bedRoomNum' => array('type' => 'integer', 'path' => 'bedRoomNum', 'example' => '3'),
		'size' => array('type' => 'integer', 'path' => 'size', 'example' => '2500'),
		'plotSize' => array('type' => 'integer', 'path' => 'plotSize', 'example' => '5000'),
		'photos' => array('type' => 'array', 'path' => 'photos', 'example' => 'Array'),
		'rentParam.washRoom' => array('type' => 'integer', 'path' => 'rentParam.washRoom', 'example' => '2'),
		'rentParam.parking' => array('type' => 'integer', 'path' => 'rentParam.parking', 'example' => '1'),
		'rentParam.acFee' => array('type' => 'double', 'path' => 'rentParam.acFee', 'example' => '8.0'),
		'sellParam.washRoom' => array('type' => 'integer', 'path' => 'sellParam.washRoom', 'example' => '3'),
		'sellParam.parking' => array('type' => 'integer', 'path' => 'sellParam.parking', 'example' => '2'),
		'newParam.handoverTime' => array('type' => 'string', 'path' => 'newParam.handoverTime', 'example' => '2027-06-01 00:00:00'),
		'newParam.totalUnits' => array('type' => 'integer', 'path' => 'newParam.totalUnits', 'example' => '500'),
		'newParam.minSize' => array('type' => 'integer', 'path' => 'newParam.minSize', 'example' => '1065'),
		'newParam.maxSize' => array('type' => 'integer', 'path' => 'newParam.maxSize', 'example' => '3143'),
		'agentName' => array('type' => 'string', 'path' => 'agentName', 'example' => 'Frank Test'),
		'agentPhone' => array('type' => 'string', 'path' => 'agentPhone', 'example' => '+971503312037'),
		'agentEmail' => array('type' => 'string', 'path' => 'agentEmail', 'example' => 'franktest@pixxicrm.ae'),
	);
}

// Function to render field list
function pixxi_render_field_list($fields, $level = 0) {
	if(empty($fields)) {
		return;
	}
	
	$output = '<ul class="pixxi-field-list' . ($level > 0 ? ' pixxi-field-list-nested' : '') . '">';
	
	foreach($fields as $field_path => $field_info) {
		if(!is_array($field_info)) {
			continue;
		}
		
		$path = isset($field_info['path']) ? $field_info['path'] : $field_path;
		$type = isset($field_info['type']) ? $field_info['type'] : 'unknown';
		$example = isset($field_info['example']) ? $field_info['example'] : '';
		
		// Truncate long examples
		if(strlen($example) > 30) {
			$example = substr($example, 0, 30) . '...';
		}
		
		$output .= '<li class="pixxi-field-item" data-field-path="' . esc_attr($path) . '">';
		$output .= '<div class="pixxi-field-item-content">';
		$output .= '<div class="pixxi-field-main">';
		$output .= '<code class="pixxi-field-path">' . esc_html($path) . '</code>';
		$output .= '<span class="pixxi-field-type">' . esc_html($type) . '</span>';
		$output .= '</div>';
		if(!empty($example) && $type != 'array') {
			$output .= '<div class="pixxi-field-example" title="Example: ' . esc_attr($field_info['example']) . '">' . esc_html($example) . '</div>';
		}
		$output .= '<button type="button" class="pixxi-field-copy" title="Copy to clipboard">
			<span class="dashicons dashicons-clipboard"></span>
		</button>';
		$output .= '</div>';
		$output .= '</li>';
	}
	
	$output .= '</ul>';
	
	echo $output;
}

// Subscriptions Page
function pixxi_subscriptions_page(){
	global $wpdb;
	
	// Check user capabilities
	if (!current_user_can('manage_options')) {
		wp_die(__('You do not have sufficient permissions to access this page.'));
	}
	
	// Handle Stripe checkout success redirect
	if (isset($_GET['pixxi_license']) && $_GET['pixxi_license'] === 'success' && !empty($_GET['session_id'])) {
		$session_id = sanitize_text_field($_GET['session_id']);
		
		// Verify session and activate license via API
		if (class_exists('Pixxi_License_API_Client')) {
			$api_client = new Pixxi_License_API_Client();
			$verify_response = $api_client->verify_checkout_session($session_id);
			
			if ($verify_response && isset($verify_response['success']) && $verify_response['success']) {
				// Clear license cache to refresh status
				$domain = $api_client->get_current_domain();
				$cache_key = 'pixxi_license_validation_' . md5($domain);
				delete_transient($cache_key);
				delete_transient('pixxi_license_plans');
				
				// Show success message
				$success_message = isset($verify_response['message']) ? $verify_response['message'] : 'Subscription activated successfully!';
				add_action('admin_notices', function() use ($success_message) {
					echo '<div class="notice notice-success is-dismissible"><p><strong>' . esc_html($success_message) . '</strong> Your license has been activated. Please refresh the page to see your updated subscription status.</p></div>';
				});
			} else {
				$error_msg = isset($verify_response['error']) ? $verify_response['error'] : 'Unknown error';
				error_log('Pixxi License: Failed to verify checkout session - ' . $error_msg);
				
				// Show info message - webhook will handle it
				add_action('admin_notices', function() {
					echo '<div class="notice notice-info is-dismissible"><p><strong>Payment received!</strong> Your subscription is being processed. Please wait a moment and refresh the page.</p></div>';
				});
			}
		}
	}
	
	// Get current property count
	$property_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}postmeta WHERE meta_key='_xm_api_sku'");
	
	// Get current license status (domain-based auto-detection)
	$current_plan = null;
	$current_plan_name = null;
	$license_data = null;
	$plans_data = array('plans' => array());
	
	if (class_exists('Pixxi_License_API_Client')) {
		$api_client = new Pixxi_License_API_Client();
		$license_data = $api_client->validate_license();
		$plans_data = $api_client->get_pricing_plans();
		
		if ($license_data['valid']) {
			$current_plan = $license_data['plan_id'];
			// Find plan name
			foreach ($plans_data['plans'] as $plan) {
				if ($plan['id'] === $current_plan) {
					$current_plan_name = $plan['name'];
					break;
				}
			}
		}
	}
	
	$plans = isset($plans_data['plans']) ? $plans_data['plans'] : array();
	
	// Format expiration date
	$expires_date = null;
	if ($license_data && isset($license_data['expires_at']) && !empty($license_data['expires_at'])) {
		$expires_timestamp = strtotime($license_data['expires_at']);
		$expires_date = date('F j, Y', $expires_timestamp);
	}
	
	// Get max properties for current plan
	$max_properties = 'Unlimited';
	if ($license_data && isset($license_data['limits']['max_properties'])) {
		$max_props = $license_data['limits']['max_properties'];
		if ($max_props == -1) {
			$max_properties = 'Unlimited';
		} else {
			$max_properties = number_format($max_props);
		}
	}
	
	// Get subscription page URL - point to devmizan.com pricing page
	$subscription_url = 'https://devmizan.com';
	
	// Handle cancelled redirect
	if (isset($_GET['pixxi_license']) && $_GET['pixxi_license'] === 'cancelled') {
		add_action('admin_notices', function() {
			echo '<div class="notice notice-warning is-dismissible"><p><strong>Subscription cancelled.</strong> You cancelled the checkout process.</p></div>';
		});
	}
	
	?>
	<div class="wrap pixxi-subscriptions-page">
		<?php if (empty($plans)): ?>
		<div class="pixxi-info-box" style="background: rgba(239, 68, 68, 0.1); border-left-color: #EF4444; color: #991B1B; margin-bottom: 24px; padding: 12px 16px; border-radius: 6px;">
			<strong>Unable to load subscription plans.</strong> Please check your connection to the License Manager at <a href="https://devmizan.com" target="_blank">devmizan.com</a> and try again.
			<br><br>
			<p>If you're trying to subscribe, please visit the subscription page directly:</p>
			<a href="<?php echo esc_url($subscription_url); ?>" target="_blank" class="button button-primary">Visit Subscription Page</a>
			<button type="button" id="pixxi-retry-plans" class="button button-secondary" style="margin-left: 10px;">Retry Loading Plans</button>
		</div>
		<?php else: ?>
		
		<!-- Current Subscription Section -->
		<?php if ($license_data && $license_data['valid'] && $current_plan_name): ?>
		<div class="pixxi-current-subscription">
			<h2 class="pixxi-section-title">Current Subscription</h2>
			<div class="pixxi-current-subscription-card">
				<div class="pixxi-current-plan-main">
					<div class="pixxi-current-plan-name"><?php echo esc_html($current_plan_name); ?> Plan</div>
					<div class="pixxi-current-plan-details">
						<span class="pixxi-status-badge-active">Status: ACTIVE</span>
						<?php if ($expires_date): ?>
							<span class="pixxi-expires-text">Expires: <?php echo esc_html($expires_date); ?></span>
						<?php endif; ?>
						<span class="pixxi-properties-text">Properties: <?php echo number_format($property_count); ?> imported (<?php echo esc_html($max_properties); ?>)</span>
					</div>
				</div>
				<div class="pixxi-current-plan-actions">
					<button type="button" class="pixxi-btn-refresh" id="pixxi-refresh-status">
						<span class="dashicons dashicons-update"></span>
						Refresh Status
					</button>
					<button type="button" class="pixxi-btn-cancel" id="pixxi-cancel-subscription">
						<span class="dashicons dashicons-no-alt"></span>
						Cancel subscription
					</button>
				</div>
			</div>
		</div>
		<?php endif; ?>
		
		<!-- Subscription Plans Section -->
		<div class="pixxi-subscription-plans-section">
			<h2 class="pixxi-section-title">Subscription Plans</h2>
			<p class="pixxi-plans-description">Choose from our professionally managed subscription plans. All plans include automatic updates and premium support.</p>
			
			<!-- Billing Toggle -->
			<div class="pixxi-billing-toggle-wrapper">
				<div class="pixxi-billing-toggle">
					<button type="button" class="pixxi-billing-option active" data-billing="monthly">Monthly</button>
					<button type="button" class="pixxi-billing-option" data-billing="yearly">
						Yearly
						<span class="pixxi-save-badge">2 months free</span>
					</button>
				</div>
			</div>
			
			<!-- Plans Grid -->
			<div class="pixxi-plans-grid">
				<?php foreach ($plans as $plan): 
					$is_current_plan = ($current_plan === $plan['id']);
					$features = isset($plan['features']) ? $plan['features'] : array();
					$limits = isset($plan['limits']) ? $plan['limits'] : array();
					$max_props = isset($limits['max_properties']) ? $limits['max_properties'] : 0;
					$property_limit_text = $max_props == -1 ? 'Unlimited Properties' : number_format($max_props) . ' Properties';
					
					// Get feature list
					$feature_list = array();
					if (isset($features['feature_list']) && is_array($features['feature_list'])) {
						$feature_list = $features['feature_list'];
					}
					
					// Build feature list based on plan
					$plan_features = array();
					if ($plan['id'] === 'free' || $plan['id'] === 'basic') {
						$plan_features[] = 'Up to ' . ($max_props == -1 ? 'Unlimited' : number_format($max_props)) . ' Properties';
						$plan_features[] = 'Core XML Import';
						$plan_features[] = 'Manual Imports';
					} elseif ($plan['id'] === 'starter' || $plan['id'] === 'pro') {
						$plan_features[] = 'Up to ' . ($max_props == -1 ? 'Unlimited' : number_format($max_props)) . ' Properties';
						$plan_features[] = 'Scheduled Imports (Hourly/Daily/Weekly)';
						$plan_features[] = 'Batch Processing';
					} elseif ($plan['id'] === 'standard' || $plan['id'] === 'enterprise') {
						$plan_features[] = 'Up to ' . ($max_props == -1 ? 'Unlimited' : number_format($max_props)) . ' Properties';
						$plan_features[] = 'Everything in ' . ($plan['id'] === 'standard' ? 'Basic' : 'Standard');
						$plan_features[] = 'REST API Access';
					} else {
						// Premium/Unlimited
						$plan_features[] = 'Up to ' . ($max_props == -1 ? 'Unlimited' : number_format($max_props)) . ' Properties';
						$plan_features[] = 'Everything in ' . ($plan['id'] === 'unlimited' ? 'Premium' : 'Standard');
						if ($plan['id'] === 'unlimited') {
							$plan_features[] = 'Hourly Auto-Sync';
						} else {
							$plan_features[] = 'Advanced Webhooks';
						}
					}
					
					// Merge with custom features
					if (!empty($feature_list)) {
						$plan_features = array_merge($plan_features, $feature_list);
					}
				?>
				<?php 
				$is_popular_plan = isset($plan['is_popular']) && $plan['is_popular'];
				?>
				<div class="pixxi-plan-card <?php echo $is_current_plan ? 'pixxi-plan-current' : ''; ?> <?php echo $is_popular_plan ? 'pixxi-plan-popular' : ''; ?>">
					<?php if ($is_popular_plan): ?>
						<div class="pixxi-plan-popular-badge">Popular</div>
					<?php endif; ?>
					<div class="pixxi-plan-card-header">
						<h3 class="pixxi-plan-name"><?php echo esc_html($plan['name']); ?></h3>
						<span class="pixxi-plan-property-badge"><?php echo esc_html($property_limit_text); ?></span>
					</div>
					
					<div class="pixxi-plan-card-pricing">
						<span class="pixxi-plan-price-monthly">$<?php echo number_format((float) $plan['monthly_price'], 2); ?> <span class="pixxi-price-period">/month</span></span>
						<span class="pixxi-plan-price-yearly" style="display: none;">$<?php echo number_format((float) $plan['yearly_price'], 2); ?> <span class="pixxi-price-period">/month</span></span>
					</div>
					
					<div class="pixxi-plan-card-features">
						<?php foreach ($plan_features as $feature): ?>
						<div class="pixxi-plan-feature-item">
							<span class="dashicons dashicons-yes-alt"></span>
							<span><?php echo esc_html($feature); ?></span>
						</div>
						<?php endforeach; ?>
					</div>
					
					<div class="pixxi-plan-card-actions">
						<?php if ($is_current_plan): ?>
							<button type="button" class="button button-secondary" disabled style="width: 100%;">
								Current Plan
							</button>
						<?php else: ?>
							<?php
							// Get current domain for subscription
							$current_domain = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field($_SERVER['HTTP_HOST']) : '';
							if (empty($current_domain)) {
								$current_domain = parse_url(home_url(), PHP_URL_HOST);
							}
							// Remove port if present
							if (strpos($current_domain, ':') !== false) {
								$current_domain = explode(':', $current_domain)[0];
							}
							?>
							<button type="button" 
									class="button button-primary pixxi-subscribe-btn" 
									style="width: 100%; text-align: center; display: block;"
									data-plan-id="<?php echo esc_attr($plan['id']); ?>"
									data-domain="<?php echo esc_attr($current_domain); ?>"
									data-billing-period="monthly">
								Subscribe Now
							</button>
						<?php endif; ?>
					</div>
				</div>
				<?php endforeach; ?>
			</div>
		</div>
		
		<?php endif; ?>
	</div>
	<?php
}

// Activate License Page (Function kept for backward compatibility but not linked in menu)
/* 
function pixxi_activate_license_page(){
	global $wpdb;
	
	// Check user capabilities
	if (!current_user_can('manage_options')) {
		wp_die(__('You do not have sufficient permissions to access this page.'));
	}
	
	// Get current license status
	$license_data = null;
	$license_key = get_option('pixxi_license_key', '');
	
	if (class_exists('Pixxi_License_API_Client')) {
		$api_client = new Pixxi_License_API_Client();
		$license_data = $api_client->validate_license();
	}
	
	$is_valid = $license_data && isset($license_data['valid']) && $license_data['valid'];
	
	// Get property count
	$property_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}postmeta WHERE meta_key='_xm_api_sku'");
	
	?>
	<div class="wrap pixxi-activate-license-page">
		<h1>Activate License</h1>
		
		<?php if ($is_valid): ?>
			<!-- Active License Section -->
			<div class="pixxi-license-active-card" style="background: #fff; border-left: 4px solid #10B981; padding: 20px; border-radius: 8px; margin-top: 20px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
				<h2 style="margin-top: 0; color: #10B981; font-size: 20px;">
					<span class="dashicons dashicons-yes-alt" style="font-size: 24px; vertical-align: middle;"></span>
					License Active
				</h2>
				
				<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-top: 16px;">
					<div>
						<strong style="color: #475569;">Plan:</strong>
						<p style="margin: 4px 0 0; font-size: 16px; color: #0F172A;"><?php echo esc_html(ucfirst($license_data['plan_id'])); ?></p>
					</div>
					<div>
						<strong style="color: #475569;">Status:</strong>
						<p style="margin: 4px 0 0;">
							<span style="display: inline-block; background: #10B981; color: white; padding: 4px 12px; border-radius: 4px; font-size: 14px; font-weight: 500;">
								<?php echo esc_html(strtoupper($license_data['status'])); ?>
							</span>
						</p>
					</div>
					<?php if (!empty($license_data['expires_at'])): ?>
					<div>
						<strong style="color: #475569;">Expires:</strong>
						<p style="margin: 4px 0 0; font-size: 16px; color: #0F172A;">
							<?php echo esc_html(date('F j, Y', strtotime($license_data['expires_at']))); ?>
						</p>
					</div>
					<?php endif; ?>
					<div>
						<strong style="color: #475569;">Properties:</strong>
						<p style="margin: 4px 0 0; font-size: 16px; color: #0F172A;">
							<?php 
							$max_props = isset($license_data['limits']['max_properties']) ? $license_data['limits']['max_properties'] : 0;
							$max_display = $max_props == -1 ? 'Unlimited' : number_format($max_props);
							echo number_format($property_count) . ' / ' . esc_html($max_display);
							?>
						</p>
					</div>
				</div>
				
				<?php if (!empty($license_key)): ?>
				<div style="margin-top: 20px; padding-top: 16px; border-top: 1px solid #E2E8F0;">
					<strong style="color: #475569;">License Key:</strong>
					<p style="margin: 4px 0 0; font-family: monospace; font-size: 14px; color: #0F172A;"><?php echo esc_html($license_key); ?></p>
				</div>
				<?php endif; ?>
				
				<div style="margin-top: 20px;">
					<button type="button" class="button button-secondary" id="pixxi-refresh-license" style="margin-right: 8px;">
						<span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
						Refresh License Status
					</button>
					<button type="button" class="button" id="pixxi-deactivate-license" style="color: #EF4444;">
						<span class="dashicons dashicons-no" style="vertical-align: middle;"></span>
						Deactivate License
					</button>
				</div>
			</div>
		<?php else: ?>
			<!-- License Activation Form -->
			<div class="pixxi-license-inactive-card" style="background: #fff; border-left: 4px solid #EF4444; padding: 20px; border-radius: 8px; margin-top: 20px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
				<h2 style="margin-top: 0; color: #EF4444; font-size: 20px;">
					<span class="dashicons dashicons-warning" style="font-size: 24px; vertical-align: middle;"></span>
					License Not Activated
				</h2>
				
				<p style="color: #475569; font-size: 15px; line-height: 1.6;">
					Please enter your license key to activate the Pixxi Property Importer. You can obtain a license key from your administrator or purchase a subscription.
				</p>
				
				<form id="pixxi-license-activation-form" style="margin-top: 20px;">
					<table class="form-table" role="presentation">
						<tbody>
							<tr>
								<th scope="row">
									<label for="pixxi_license_key_input">License Key</label>
								</th>
								<td>
									<input type="text" 
										   id="pixxi_license_key_input" 
										   name="license_key" 
										   class="regular-text" 
										   placeholder="PIXXI-XXXX-XXXX-XXXX"
										   pattern="PIXXI-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}"
										   style="font-family: monospace; text-transform: uppercase; letter-spacing: 1px;"
										   required>
									<p class="description">
										Enter your license key in the format: PIXXI-XXXX-XXXX-XXXX
									</p>
								</td>
							</tr>
						</tbody>
					</table>
					
					<p class="submit">
						<button type="submit" class="button button-primary button-large" id="pixxi-activate-license-btn">
							<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span>
							Activate License
						</button>
					</p>
				</form>
				
				<div id="pixxi-license-activation-message" style="margin-top: 16px;"></div>
			</div>
		<?php endif; ?>
		
		<div style="margin-top: 24px; padding: 16px; background: #F8FAFC; border-radius: 8px;">
			<h3 style="margin-top: 0; font-size: 16px; color: #0F172A;">Need Help?</h3>
			<p style="color: #475569; line-height: 1.6; margin-bottom: 0;">
				If you need assistance with license activation or have questions about your subscription, 
				please contact support at <a href="https://devmizan.com" target="_blank">devmizan.com</a>.
			</p>
		</div>
	</div>
	
	<script type="text/javascript">
	jQuery(document).ready(function($) {
		// Refresh license status
		$('#pixxi-refresh-license').on('click', function() {
			location.reload();
		});
		
		// Deactivate license
		$('#pixxi-deactivate-license').on('click', function() {
			if (confirm('Are you sure you want to deactivate this license?')) {
				$.ajax({
					url: ajaxurl,
					type: 'POST',
					data: {
						action: 'pixxi_deactivate_license',
						nonce: '<?php echo wp_create_nonce("pixxi_license_nonce"); ?>'
					},
					success: function(response) {
						if (response.success) {
							location.reload();
						} else {
							alert('Error: ' + (response.data || 'Failed to deactivate license'));
						}
					},
					error: function() {
						alert('Failed to deactivate license. Please try again.');
					}
				});
			}
		});
		
		// Handle license activation form
		$('#pixxi-license-activation-form').on('submit', function(e) {
			e.preventDefault();
			
			var $form = $(this);
			var $btn = $('#pixxi-activate-license-btn');
			var $msg = $('#pixxi-license-activation-message');
			var licenseKey = $('#pixxi_license_key_input').val().trim().toUpperCase();
			
			// Validate format
			var licensePattern = /^PIXXI-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;
			if (!licensePattern.test(licenseKey)) {
				$msg.html('<div class="notice notice-error"><p>Invalid license key format. Please use format: PIXXI-XXXX-XXXX-XXXX</p></div>');
				return;
			}
			
			// Disable button and show loading
			$btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin" style="vertical-align: middle;"></span> Activating...');
			$msg.html('');
			
			// Send AJAX request
			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'pixxi_activate_license',
					license_key: licenseKey,
					nonce: '<?php echo wp_create_nonce("pixxi_license_nonce"); ?>'
				},
				success: function(response) {
					if (response.success) {
						$msg.html('<div class="notice notice-success"><p><strong>Success!</strong> ' + response.data.message + '</p></div>');
						setTimeout(function() {
							location.reload();
						}, 1500);
					} else {
						$msg.html('<div class="notice notice-error"><p><strong>Error:</strong> ' + (response.data || 'Failed to activate license') + '</p></div>');
						$btn.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span> Activate License');
					}
				},
				error: function() {
					$msg.html('<div class="notice notice-error"><p><strong>Error:</strong> Failed to connect to license server. Please try again.</p></div>');
					$btn.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span> Activate License');
				}
			});
		});
		
		// Auto-format license key input
		$('#pixxi_license_key_input').on('input', function() {
			var val = $(this).val().toUpperCase().replace(/[^A-Z0-9-]/g, '');
			$(this).val(val);
		});
	});
	</script>
	
	<style>
	.dashicons.spin {
		animation: spin 1s linear infinite;
	}
	@keyframes spin {
		from { transform: rotate(0deg); }
		to { transform: rotate(360deg); }
	}
	</style>
	<?php
}
*/


register_deactivation_hook( __FILE__, 'pixxi_deactivate_plugin' );

function pixxi_deactivate_plugin() {
  //  wp_clear_scheduled_hook( 'lw_up_cron_hook' );
  // Clear our cron
  pixxi_deactivate_cron();
}

//Delete media attached to an custom post type
add_action( 'before_delete_post', 'pixxi_delete_all_attached_media' );
function pixxi_delete_all_attached_media( $post_id ) {

  if( get_post_type($post_id) == "property") {
    $attachments = get_attached_media( '', $post_id );

    foreach ($attachments as $attachment) {
      wp_delete_attachment( $attachment->ID, 'true' );
    }
  }
}

// Helper function to check API status
function pixxi_check_api_status() {
	$opt = get_option('pixxi_opt_val');
	// Migration: Check for old option name
	if(!$opt) {
		$opt = get_option('lw_opt_val');
		if($opt) {
			update_option('pixxi_opt_val', $opt);
		}
	}
	$pixxi_option = json_decode($opt, true);
	
	if(!is_array($pixxi_option)) {
		return array('status' => 'offline', 'last_check' => 0);
	}
	
	$api_url = !empty($pixxi_option['pixxi_apiurl']) ? trim($pixxi_option['pixxi_apiurl']) : (!empty($pixxi_option['lw_apiurl']) ? trim($pixxi_option['lw_apiurl']) : '');
	$api_token = !empty($pixxi_option['pixxi_token']) ? $pixxi_option['pixxi_token'] : (!empty($pixxi_option['lw_token']) ? $pixxi_option['lw_token'] : '');
	
	if(empty($api_url) || empty($api_token)) {
		return array('status' => 'offline', 'last_check' => 0);
	}
	
	// Get last check time
	$last_check = get_option('pixxi_api_last_check', 0);
	$last_status = get_option('pixxi_api_status', 'offline');
	
	// Check if we need to update (check every 5 minutes)
	$current_time = time();
	if(($current_time - $last_check) < 300) {
		// Return cached status if checked within last 5 minutes
		return array('status' => $last_status, 'last_check' => $last_check);
	}
	
	// Perform actual API check
	$curl = curl_init();
	curl_setopt_array($curl, array(
		CURLOPT_URL => $api_url,
		CURLOPT_RETURNTRANSFER => true,
		CURLOPT_TIMEOUT => 10,
		CURLOPT_CONNECTTIMEOUT => 5,
		CURLOPT_HTTPHEADER => array(
			'X-PIXXI-TOKEN: ' . $api_token,
			'Content-Type: application/json'
		),
	));
	
	curl_exec($curl);
	$httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
	$curl_error = curl_error($curl);
	curl_close($curl);
	
	$status = 'offline';
	if(!$curl_error && ($httpcode == 200 || $httpcode == 400 || $httpcode == 401)) {
		// 200 = success, 400/401 = API responding (just auth/request issue)
		$status = 'online';
	}
	
	// Update last check time and status
	update_option('pixxi_api_last_check', $current_time);
	update_option('pixxi_api_status', $status);
	
	return array('status' => $status, 'last_check' => $current_time);
}

// Helper function to add import history
function pixxi_add_import_history($type = 'Manual', $action = 'Download', $status = 'Success') {
	$history = get_option('pixxi_import_history');
	// Migration: Check for old option name
	if(!$history) {
		$history = get_option('lw_import_history');
		if($history) {
			update_option('pixxi_import_history', $history);
		}
	}
	// Handle both string and array formats
	if(is_array($history)) {
		$history_data = $history;
	} elseif(is_string($history) && !empty($history)) {
		$history_data = json_decode($history, true);
		if(!is_array($history_data)) {
			$history_data = array();
		}
	} else {
		$history_data = array();
	}
	
	// Add new entry
	$history_data[] = array(
		'date' => time(),
		'type' => $type,
		'action' => $action,
		'status' => $status
	);
	
	// Keep only last 50 entries
	$history_data = array_slice($history_data, -50);
	
	update_option('pixxi_import_history', json_encode($history_data));
}

// Clean output buffers before AJAX handlers - run early
add_action('init', 'pixxi_clean_output_buffers_early', 1);
add_action('admin_init', 'pixxi_clean_output_buffers', 1);

function pixxi_clean_output_buffers_early() {
	if (defined('DOING_AJAX') && DOING_AJAX) {
		// Clean any existing output buffers
		while (ob_get_level()) {
			ob_end_clean();
		}
		// Start a fresh buffer
		ob_start();
	}
}

function pixxi_clean_output_buffers() {
	if (defined('DOING_AJAX') && DOING_AJAX) {
		// Clean any existing output buffers
		while (ob_get_level()) {
			ob_end_clean();
		}
	}
}

// AJAX handlers for dashboard
add_action('wp_ajax_pixxi_download_properties', 'pixxi_ajax_download_properties');
add_action('wp_ajax_pixxi_update_properties', 'pixxi_ajax_update_properties');
add_action('wp_ajax_pixxi_full_sync', 'pixxi_ajax_full_sync');
add_action('wp_ajax_pixxi_get_stats', 'pixxi_ajax_get_stats');
add_action('wp_ajax_pixxi_pause_sync', 'pixxi_ajax_pause_sync');
add_action('wp_ajax_pixxi_stop_sync', 'pixxi_ajax_stop_sync');
add_action('wp_ajax_pixxi_resume_sync', 'pixxi_ajax_resume_sync');
add_action('wp_ajax_pixxi_refresh_license', 'pixxi_ajax_refresh_license');
add_action('wp_ajax_pixxi_cancel_subscription', 'pixxi_ajax_cancel_subscription');
add_action('wp_ajax_pixxi_activate_license', 'pixxi_ajax_activate_license');
add_action('wp_ajax_pixxi_deactivate_license', 'pixxi_ajax_deactivate_license');
add_action('wp_ajax_pixxi_create_support_ticket', 'pixxi_ajax_create_support_ticket');
add_action('wp_ajax_pixxi_get_ticket_comments', 'pixxi_ajax_get_ticket_comments');
add_action('wp_ajax_pixxi_add_ticket_comment', 'pixxi_ajax_add_ticket_comment');
add_action('wp_ajax_pixxi_get_all_tickets', 'pixxi_ajax_get_all_tickets');
add_action('wp_ajax_pixxi_get_unread_count', 'pixxi_ajax_get_unread_count');
add_action('wp_ajax_pixxi_mark_comments_read', 'pixxi_ajax_mark_comments_read');
add_action('wp_ajax_pixxi_sync_single_property', 'pixxi_ajax_sync_single_property');
add_action('wp_ajax_nopriv_pixxi_support_webhook', 'pixxi_ajax_support_webhook');
add_action('rest_api_init', 'pixxi_register_support_webhook_route');

function pixxi_ajax_download_properties() {
	// Clean any existing output and BOM characters
	if (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	// Update download timestamp
	file_put_contents(PIXXI_DOWN, time());
	
	// Add to history
	pixxi_add_import_history('Manual', 'Download', 'Success');
	
	// Initialize sync data
	$data = array(
		'action' => 'delete',
		'chunk' => 0,
		'size' => 1000
	);
		update_option('pixxi_api_info', json_encode($data));
	
	// Call the download function (it handles its own output buffering)
	$pixxi = new pixxi_property_class();
	$result = $pixxi->get_curl_data();
	
	// Discard any remaining output including BOM
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	// Check for errors
	if(isset($result['error'])) {
		$error_message = $result['error'];
		if(isset($result['http_code'])) {
			$error_message .= ' (HTTP ' . $result['http_code'] . ')';
		}
		wp_send_json_error(array('message' => $error_message));
		return;
	}
	
	wp_send_json_success(array('message' => isset($result['message']) ? $result['message'] : 'Download completed successfully'));
}

function pixxi_ajax_update_properties() {
	// Set error handler to catch fatal errors
	set_error_handler(function($errno, $errstr, $errfile, $errline) {
		error_log("PHP Error [$errno]: $errstr in $errfile on line $errline");
		return false; // Let PHP handle it normally
	});
	
	// Clean any existing output and BOM characters
	if (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	try {
		check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
		if(!current_user_can('manage_options')) {
			ob_end_clean();
			wp_send_json_error('Unauthorized');
			return;
		}
		
		file_put_contents(PIXXI_UPDATE, time());
		
		// Add to history
		pixxi_add_import_history('Manual', 'Update', 'Success');
		$data = get_option('pixxi_api_info');
		// Migration: Check for old option name
		if(!$data) {
			$data = get_option('lw_api_info');
			if($data) {
				update_option('pixxi_api_info', $data);
			}
		}
		$lw = json_decode($data, true);
		
		// If no sync data exists, initialize it
		if(!$lw || !is_array($lw)) {
			$lw = array(
				'action' => 'delete',
				'chunk' => 0,
				'size' => 1000
			);
			update_option('pixxi_api_info', json_encode($lw));
		}
		
		$processed = 0;
		$max_iterations = 100; // Prevent infinite loops
		$last_error = '';
		
		// Process in chunks until complete or paused
		while(isset($lw['action']) && $lw['action'] != 'stop' && $lw['action'] != 'pause' && $processed < $max_iterations) {
			// Start output buffering for update_products which has echo statements
			ob_start();
			
			try {
				$pixxi_class = new pixxi_property_class();
				$pixxi_class->update_products($lw);
			} catch (Exception $e) {
				$last_error = $e->getMessage();
				error_log('Property update error: ' . $last_error . ' | Trace: ' . $e->getTraceAsString());
			} catch (Error $e) {
				$last_error = $e->getMessage();
				error_log('Property update fatal error: ' . $last_error . ' | Trace: ' . $e->getTraceAsString());
			}
			
			// Discard all output from update_products (including any echo statements)
			while (ob_get_level()) {
				ob_end_clean();
			}
			
			// Get updated sync data
			$data = get_option('pixxi_api_info');
		// Migration: Check for old option name
		if(!$data) {
			$data = get_option('lw_api_info');
			if($data) {
				update_option('pixxi_api_info', $data);
			}
		}
			$lw = json_decode($data, true);
			
			if(!$lw || !is_array($lw) || !isset($lw['action']) || $lw['action'] == 'stop' || $lw['action'] == 'pause') {
				break;
			}
			
			$processed++;
		}
		
		// Discard any remaining output including BOM
		while (ob_get_level()) {
			ob_end_clean();
		}
		
		// Check final state
		$data = get_option('pixxi_api_info');
		// Migration: Check for old option name
		if(!$data) {
			$data = get_option('lw_api_info');
			if($data) {
				update_option('pixxi_api_info', $data);
			}
		}
		$lw = json_decode($data, true);
		
		if($lw && is_array($lw) && isset($lw['action']) && $lw['action'] == 'pause') {
			wp_send_json_success(array('message' => 'Update process paused', 'paused' => true));
		} elseif($lw && is_array($lw) && isset($lw['action']) && $lw['action'] != 'stop') {
			wp_send_json_success(array('message' => 'Update process continuing...', 'continue' => true));
		} elseif(!empty($last_error)) {
			wp_send_json_error(array('message' => 'Update completed with errors. Check error log. Last error: ' . substr($last_error, 0, 200)));
		} else {
			wp_send_json_success(array('message' => 'Update process completed'));
		}
		
	} catch (Exception $e) {
		while (ob_get_level()) {
			ob_end_clean();
		}
		error_log('AJAX update error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString());
		wp_send_json_error(array('message' => 'Error: ' . $e->getMessage()));
	} catch (Error $e) {
		while (ob_get_level()) {
			ob_end_clean();
		}
		error_log('AJAX update fatal error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString());
		wp_send_json_error(array('message' => 'Fatal error: ' . $e->getMessage()));
	} finally {
		restore_error_handler();
	}
}

function pixxi_ajax_full_sync() {
	// Clean any existing output and BOM characters
	if (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	// Start with download
	file_put_contents(PIXXI_DOWN, time());
	
	// Add to history
	pixxi_add_import_history('Manual', 'Full Sync', 'Success');
	
	$data = array(
		'action' => 'delete',
		'chunk' => 0,
		'size' => 1000
	);
		update_option('pixxi_api_info', json_encode($data));
	
	// Start output buffering for get_curl_data
	ob_start();
	$pixxi = new pixxi_property_class();
	$result = $pixxi->get_curl_data();
	
	// Discard any output including BOM
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	// Check for errors
	if(isset($result['error'])) {
		wp_send_json_error(array('message' => $result['error']));
		return;
	}
	
	wp_send_json_success(array('message' => isset($result['message']) ? $result['message'] : 'Full sync started successfully'));
}

function pixxi_ajax_get_stats() {
	// Clean any existing output and BOM characters - be very aggressive
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	// Prevent any output
	@ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		while (ob_get_level()) {
			ob_end_clean();
		}
		wp_send_json_error('Unauthorized');
		return;
	}
	
	global $wpdb;
	$total_properties = (int)$wpdb->get_var("SELECT COUNT(*) FROM " . PIXXI_TABLE);
	$total_synced = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}postmeta WHERE meta_key='_xm_api_sku'");
	$total_pending = $total_properties - $total_synced;
	
	$last_download = file_exists(PIXXI_DOWN) ? (int)file_get_contents(PIXXI_DOWN) : 0;
	$last_update = file_exists(PIXXI_UPDATE) ? (int)file_get_contents(PIXXI_UPDATE) : 0;
	$last_sync = max($last_download, $last_update);
	
	$api_info = get_option('pixxi_api_info');
	// Migration: Check for old option name
	if(!$api_info) {
		$api_info = get_option('lw_api_info');
		if($api_info) {
			update_option('pixxi_api_info', $api_info);
		}
	}
	$api_data = $api_info ? json_decode($api_info, true) : array('action' => 'stop', 'chunk' => 0);
	
	if(!is_array($api_data)) {
		$api_data = array('action' => 'stop', 'chunk' => 0);
	}
	
	// Discard any output including BOM - be very aggressive
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	// Send clean JSON response
	wp_send_json_success(array(
		'total_properties' => $total_properties,
		'total_synced' => $total_synced,
		'total_pending' => $total_pending,
		'last_sync' => $last_sync ? human_time_diff($last_sync, current_time('timestamp')) . ' ago' : 'Never',
		'current_action' => isset($api_data['action']) ? $api_data['action'] : 'stop',
		'current_chunk' => isset($api_data['chunk']) ? $api_data['chunk'] : 0
	));
}

function pixxi_ajax_pause_sync() {
	// Clean any existing output
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	$data = get_option('lw_api_info');
	$lw = json_decode($data, true);
	
	if($lw && is_array($lw) && isset($lw['action']) && $lw['action'] != 'stop' && $lw['action'] != 'pause') {
		// Save current action before pausing
		if(in_array($lw['action'], array('delete', 'update', 'insert'))) {
			$lw['last_action'] = $lw['action'];
		}
		$lw['action'] = 'pause';
		update_option('pixxi_api_info', json_encode($lw));
	}
	
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	wp_send_json_success(array('message' => 'Sync paused'));
}

function pixxi_ajax_stop_sync() {
	// Clean any existing output
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	$data = get_option('lw_api_info');
	$lw = json_decode($data, true);
	
	if($lw && is_array($lw)) {
		$lw['action'] = 'stop';
		$lw['chunk'] = 0;
		unset($lw['last_action']);
		update_option('pixxi_api_info', json_encode($lw));
	}
	
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	wp_send_json_success(array('message' => 'Sync stopped'));
}

function pixxi_ajax_refresh_license() {
	// Clean any existing output
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	// Clear license cache
	if (class_exists('Pixxi_License_API_Client')) {
		$api_client = new Pixxi_License_API_Client();
		$domain = $api_client->get_current_domain();
		
		// Clear transients
		$cache_key = 'pixxi_license_validation_' . md5($domain);
		delete_transient($cache_key);
		
		// Also clear legacy cache if exists
		$license_key = get_option('pixxi_license_key', '');
		if (!empty($license_key)) {
			$cache_key_legacy = 'pixxi_license_validation_' . md5($license_key . $domain);
			delete_transient($cache_key_legacy);
		}
		
		// Clear plans cache
		delete_transient('pixxi_license_plans');
	}
	
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	wp_send_json_success(array('message' => 'License cache cleared'));
}

function pixxi_ajax_cancel_subscription() {
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if (!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	$license_key = get_option('pixxi_license_key', '');
	if (empty($license_key)) {
		ob_end_clean();
		wp_send_json_error(array('message' => __('No license key. Activate your license first.', 'pixxi-property-importer')));
		return;
	}
	
	if (!class_exists('Pixxi_License_API_Client')) {
		ob_end_clean();
		wp_send_json_error(array('message' => __('License client unavailable.', 'pixxi-property-importer')));
		return;
	}
	
	$client = new Pixxi_License_API_Client();
	$result = $client->cancel_subscription();
	
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	if (is_wp_error($result)) {
		wp_send_json_error(array('message' => $result->get_error_message()));
		return;
	}
	
	wp_send_json_success(array('message' => __('Subscription cancelled. Your license has been deactivated.', 'pixxi-property-importer')));
}

// AJAX Handler: Activate License
function pixxi_ajax_activate_license() {
	// Clean any existing output
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_license_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	$license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
	
	// Validate license key format
	if (empty($license_key)) {
		ob_end_clean();
		wp_send_json_error('License key is required');
		return;
	}
	
	// Check format: PIXXI-XXXX-XXXX-XXXX
	if (!preg_match('/^PIXXI-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $license_key)) {
		ob_end_clean();
		wp_send_json_error('Invalid license key format. Please use format: PIXXI-XXXX-XXXX-XXXX');
		return;
	}
	
	// Validate with API
	if (!class_exists('Pixxi_License_API_Client')) {
		ob_end_clean();
		wp_send_json_error('License API client not available');
		return;
	}
	
	// Store license key temporarily
	update_option('pixxi_license_key', $license_key);
	
	// Create API client with the new license key
	$api_client = new Pixxi_License_API_Client(null, $license_key);
	$domain = $api_client->get_current_domain();
	
	// Clear any existing cache
	$cache_key = 'pixxi_license_validation_' . md5($domain);
	delete_transient($cache_key);
	$cache_key_legacy = 'pixxi_license_validation_' . md5($license_key . $domain);
	delete_transient($cache_key_legacy);
	
	// Validate the license
	$validation_result = $api_client->validate_license();
	
	if (!$validation_result || !isset($validation_result['valid'])) {
		// Remove invalid license key
		delete_option('pixxi_license_key');
		ob_end_clean();
		wp_send_json_error('Failed to validate license. Please check your connection and try again.');
		return;
	}
	
	if (!$validation_result['valid']) {
		// Remove invalid license key
		delete_option('pixxi_license_key');
		$error_msg = isset($validation_result['error']) ? $validation_result['error'] : 'Invalid license key';
		ob_end_clean();
		wp_send_json_error($error_msg);
		return;
	}
	
	// License is valid - keep it stored
	// Clear plans cache to refresh
	delete_transient('pixxi_license_plans');
	
	$plan_name = isset($validation_result['plan_id']) ? ucfirst($validation_result['plan_id']) : 'Unknown';
	
	ob_end_clean();
	wp_send_json_success(array(
		'message' => 'License activated successfully! Your ' . $plan_name . ' plan is now active.',
		'plan_id' => $validation_result['plan_id'],
		'status' => $validation_result['status']
	));
}

// AJAX Handler: Deactivate License
function pixxi_ajax_deactivate_license() {
	// Clean any existing output
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_license_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	// Remove license key from options
	delete_option('pixxi_license_key');
	
	// Clear all license caches
	if (class_exists('Pixxi_License_API_Client')) {
		$api_client = new Pixxi_License_API_Client();
		$domain = $api_client->get_current_domain();
		
		$cache_key = 'pixxi_license_validation_' . md5($domain);
		delete_transient($cache_key);
		
		delete_transient('pixxi_license_plans');
	}
	
	ob_end_clean();
	wp_send_json_success(array('message' => 'License deactivated successfully'));
}

function pixxi_ajax_resume_sync() {
	// Clean any existing output
	while (ob_get_level()) {
		ob_end_clean();
	}
	ob_start();
	
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		ob_end_clean();
		wp_send_json_error('Unauthorized');
		return;
	}
	
	$data = get_option('lw_api_info');
	$lw = json_decode($data, true);
	
	if($lw && is_array($lw) && isset($lw['action']) && $lw['action'] == 'pause') {
		// Resume from the last action before pause
		if(isset($lw['last_action']) && !empty($lw['last_action']) && in_array($lw['last_action'], array('delete', 'update', 'insert'))) {
			$lw['action'] = $lw['last_action'];
		} else {
			// Try to determine the correct action based on current state
			global $wpdb;
			$pixxi_class = new pixxi_property_class();
			$wp_data = $pixxi_class->get_save_sku();
			$api_data = $pixxi_class->get_products_data();
			
			if(!is_array($wp_data)) $wp_data = array();
			if(!is_array($api_data)) $api_data = array();
			
			$wp_sku = array_keys($wp_data);
			$api_sku = array_keys($api_data);
			$delete_sku = array_diff($wp_sku, $api_sku);
			$update_sku = array_intersect($wp_sku, $api_sku);
			$insert_sku = array_diff($api_sku, $wp_sku);
			
			if(!empty($delete_sku)) {
				$lw['action'] = 'delete';
			} elseif(!empty($update_sku)) {
				$lw['action'] = 'update';
			} elseif(!empty($insert_sku)) {
				$lw['action'] = 'insert';
			} else {
				$lw['action'] = 'update'; // Default
			}
		}
		unset($lw['last_action']);
		update_option('pixxi_api_info', json_encode($lw));
		
		// Update timestamp
		file_put_contents(PIXXI_UPDATE, time());
		
		// Add to history
		pixxi_add_import_history('Manual', 'Update', 'Success');
	}
	
	while (ob_get_level()) {
		ob_end_clean();
	}
	
	wp_send_json_success(array('message' => 'Sync resumed'));
}

// Cron scheduling functions
function pixxi_schedule_cron($enabled, $interval, $minute = 5) {
	// Clear existing cron
	wp_clear_scheduled_hook('pixxi_auto_sync_cron');
	
	if($enabled == 1) {
		$timestamp = time();
		
		if($interval == 'minute') {
			// Custom minute interval - use wp_schedule_event with hourly and then reschedule
			// For minute intervals, we'll use a custom approach
			$recurrence = 'pixxi_minute_' . $minute;
			// Add custom cron schedule
			add_filter('cron_schedules', 'pixxi_add_cron_schedules');
			wp_schedule_event($timestamp, $recurrence, 'pixxi_auto_sync_cron');
		} elseif($interval == 'hourly') {
			wp_schedule_event($timestamp, 'hourly', 'pixxi_auto_sync_cron');
		} elseif($interval == 'daily') {
			wp_schedule_event($timestamp, 'daily', 'pixxi_auto_sync_cron');
		} elseif($interval == 'weekly') {
			wp_schedule_event($timestamp, 'weekly', 'pixxi_auto_sync_cron');
		}
	}
}

function pixxi_add_cron_schedules($schedules) {
	$opt = get_option('lw_opt_val');
	if($opt) {
		$pixxi_option = json_decode($opt, true);
		if(is_array($pixxi_option) && isset($pixxi_option['pixxi_cron_minute'])) {
			$minute = (int)$pixxi_option['pixxi_cron_minute'];
			if($minute >= 1 && $minute <= 60) {
				$schedules['pixxi_minute_' . $minute] = array(
					'interval' => $minute * 60, // Convert minutes to seconds
					'display' => sprintf(__('Every %d Minutes', 'pixxi-property-importer'), $minute)
				);
			}
		}
	}
	
	return $schedules;
}

// Add custom cron schedules filter
add_filter('cron_schedules', 'pixxi_add_cron_schedules');

// Register cron hook
add_action('pixxi_auto_sync_cron', 'pixxi_run_auto_sync');

function pixxi_run_auto_sync() {
	// Check if sync is already running (paused or in progress)
	$api_info = get_option('pixxi_api_info');
	// Migration: Check for old option name
	if(!$api_info) {
		$api_info = get_option('lw_api_info');
		if($api_info) {
			update_option('pixxi_api_info', $api_info);
		}
	}
	$api_data = $api_info ? json_decode($api_info, true) : array('action' => 'stop');
	
	// Don't start if already running or paused
	if(isset($api_data['action']) && $api_data['action'] != 'stop') {
		error_log('Pixxi Auto Sync: Skipping - sync already in progress (action: ' . $api_data['action'] . ')');
		return;
	}
	
	// Initialize sync process
	file_put_contents(PIXXI_UPDATE, time());
	
	// Add to history
	pixxi_add_import_history('Automatic', 'Update', 'Success');
	
	$data = get_option('lw_api_info');
	$lw = json_decode($data, true);
	
	// If no sync data exists, initialize it
	if(!$lw || !is_array($lw)) {
		$lw = array(
			'action' => 'delete',
			'chunk' => 0,
			'size' => 1000
		);
		update_option('pixxi_api_info', json_encode($lw));
	}
	
	// Start the sync process
	try {
		$pixxi_class = new pixxi_property_class();
		$pixxi_class->update_products($lw);
	} catch (Exception $e) {
		error_log('Pixxi Auto Sync Error: ' . $e->getMessage());
	} catch (Error $e) {
		error_log('Pixxi Auto Sync Fatal Error: ' . $e->getMessage());
	}
}

function pixxi_get_next_cron_run() {
	$next_run = wp_next_scheduled('pixxi_auto_sync_cron');
	if($next_run) {
		return date('Y-m-d H:i:s', $next_run) . ' (' . human_time_diff($next_run, current_time('timestamp')) . ' from now)';
	}
	return false;
}

// Schedule cron on plugin activation (called from pixxi_plugin_create_db)
function pixxi_activate_cron() {
	$opt = get_option('lw_opt_val');
	if($opt) {
		$pixxi_option = json_decode($opt, true);
		if(is_array($pixxi_option) && isset($pixxi_option['pixxi_cron_enabled']) && $pixxi_option['pixxi_cron_enabled'] == 1) {
			$interval = isset($pixxi_option['pixxi_cron_interval']) ? $pixxi_option['pixxi_cron_interval'] : 'hourly';
			$minute = isset($pixxi_option['pixxi_cron_minute']) ? (int)$pixxi_option['pixxi_cron_minute'] : 5;
			pixxi_schedule_cron(1, $interval, $minute);
		}
	}
}

// Clear cron on plugin deactivation
function pixxi_deactivate_cron() {
	wp_clear_scheduled_hook('pixxi_auto_sync_cron');
}

// Helper function to ensure support tables exist
function pixxi_ensure_support_tables() {
	global $wpdb;
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	
	// Check if tables exist
	$tickets_table_exists = $wpdb->get_var("SHOW TABLES LIKE '$tickets_table'");
	
	// Check if subject and email columns exist, add them if they don't
	if ($tickets_table_exists) {
		$columns = $wpdb->get_col("SHOW COLUMNS FROM $tickets_table");
		if (!in_array('subject', $columns)) {
			$wpdb->query("ALTER TABLE $tickets_table ADD COLUMN subject varchar(500) AFTER site_url");
		}
		if (!in_array('email', $columns)) {
			$wpdb->query("ALTER TABLE $tickets_table ADD COLUMN email varchar(255) AFTER subject");
		}
	}
	
	// Check if comments table exists and add read_at column if needed
	$comments_table_exists = $wpdb->get_var("SHOW TABLES LIKE '$comments_table'");
	if ($comments_table_exists) {
		$comment_columns = $wpdb->get_col("SHOW COLUMNS FROM $comments_table");
		if (!in_array('read_at', $comment_columns)) {
			$wpdb->query("ALTER TABLE $comments_table ADD COLUMN read_at datetime DEFAULT NULL AFTER created_at");
		}
	}
	
	if (!$tickets_table_exists) {
		$charset_collate = $wpdb->get_charset_collate();
		
		// Create support tickets table
		$support_tickets_sql = "CREATE TABLE IF NOT EXISTS $tickets_table (
			ticket_id int(11) NOT NULL AUTO_INCREMENT,
			ticket_number varchar(50) NOT NULL,
			user_name varchar(255) NOT NULL,
			site_url varchar(500) NOT NULL,
			subject varchar(500),
			email varchar(255),
			login_creds text,
			issue_description text NOT NULL,
			wp_version varchar(50),
			license_key varchar(255),
			status varchar(50) DEFAULT 'open',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (ticket_id),
			UNIQUE KEY ticket_number (ticket_number)
		) $charset_collate;";
		
		// Create support comments table
		$support_comments_sql = "CREATE TABLE IF NOT EXISTS $comments_table (
			comment_id int(11) NOT NULL AUTO_INCREMENT,
			ticket_id int(11) NOT NULL,
			user_type varchar(20) NOT NULL,
			message text NOT NULL,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			read_at datetime DEFAULT NULL,
			PRIMARY KEY (comment_id),
			KEY ticket_id (ticket_id),
			KEY read_at (read_at)
		) $charset_collate;";
		
		require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
		dbDelta($support_tickets_sql);
		dbDelta($support_comments_sql);
	}
}

// Support Ticket AJAX Handlers
function pixxi_ajax_create_support_ticket() {
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	
	// Get and sanitize form data
	$name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
	$site_url = isset($_POST['site_url']) ? esc_url_raw($_POST['site_url']) : '';
	$subject = isset($_POST['subject']) ? sanitize_text_field($_POST['subject']) : '';
	$email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
	$login_creds = isset($_POST['login_creds']) ? sanitize_textarea_field($_POST['login_creds']) : '';
	$issue_description = isset($_POST['issue_description']) ? sanitize_textarea_field($_POST['issue_description']) : '';
	$wp_version = isset($_POST['wp_version']) ? sanitize_text_field($_POST['wp_version']) : '';
	$license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
	
	// Validate required fields
	if(empty($name) || empty($site_url) || empty($subject) || empty($email) || empty($issue_description)) {
		wp_send_json_error(array('message' => 'Please fill in all required fields'));
		return;
	}
	
	// Validate email format
	if(!is_email($email)) {
		wp_send_json_error(array('message' => 'Please provide a valid email address'));
		return;
	}
	
	// Generate unique ticket number
	$ticket_number = 'PIXXI-' . date('Ymd') . '-' . strtoupper(wp_generate_password(6, false));
	
	// Encrypt login credentials if provided
	$encrypted_creds = '';
	if(!empty($login_creds)) {
		$encrypted_creds = base64_encode($login_creds); // Simple encoding, can be improved with proper encryption
	}
	
	// Insert ticket into database
	$result = $wpdb->insert(
		$tickets_table,
		array(
			'ticket_number' => $ticket_number,
			'user_name' => $name,
			'site_url' => $site_url,
			'subject' => $subject,
			'email' => $email,
			'login_creds' => $encrypted_creds,
			'issue_description' => $issue_description,
			'wp_version' => $wp_version,
			'license_key' => $license_key,
			'status' => 'open',
			'created_at' => current_time('mysql'),
			'updated_at' => current_time('mysql')
		),
		array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
	);
	
	if($result === false) {
		wp_send_json_error(array('message' => 'Failed to create ticket. Please try again.'));
		return;
	}
	
	$ticket_id = $wpdb->insert_id;
	
	// Send ticket to devmizan.com API
	$api_data = array(
		'ticket_number' => $ticket_number,
		'ticket_id' => $ticket_id,
		'user_name' => $name,
		'site_url' => $site_url,
		'subject' => $subject,
		'email' => $email,
		'login_creds' => $encrypted_creds,
		'issue_description' => $issue_description,
		'wp_version' => $wp_version,
		'license_key' => $license_key
	);
	
	$api_response = wp_remote_post('https://devmizan.com/wp-json/pixxi-license/v1/support/create-ticket', array(
		'body' => json_encode($api_data),
		'headers' => array(
			'Content-Type' => 'application/json'
		),
		'timeout' => 30
	));
	
	// Check API response and log errors
	if (is_wp_error($api_response)) {
		error_log('Pixxi Support: Failed to send ticket to devmizan.com - ' . $api_response->get_error_message());
	} else {
		$response_code = wp_remote_retrieve_response_code($api_response);
		$response_body = wp_remote_retrieve_body($api_response);
		
		if ($response_code !== 200) {
			error_log('Pixxi Support: API returned error code ' . $response_code . ' - ' . $response_body);
		} else {
			$api_result = json_decode($response_body, true);
			if (isset($api_result['success']) && $api_result['success']) {
				error_log('Pixxi Support: Ticket successfully sent to devmizan.com - Ticket #' . $ticket_number);
			} else {
				error_log('Pixxi Support: API returned success=false - ' . $response_body);
			}
		}
	}
	
	// Add issue_description as the first comment/message
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	$wpdb->insert(
		$comments_table,
		array(
			'ticket_id' => $ticket_id,
			'user_type' => 'user',
			'message' => $issue_description,
			'created_at' => current_time('mysql')
		),
		array('%d', '%s', '%s', '%s')
	);
	
	// Send email notification to admin
	pixxi_send_support_email('admin', 'new_ticket', array(
		'ticket_number' => $ticket_number,
		'user_name' => $name,
		'email' => $email,
		'subject' => $subject,
		'site_url' => $site_url,
		'issue_description' => $issue_description
	));
	
	wp_send_json_success(array(
		'ticket_id' => $ticket_id,
		'ticket_number' => $ticket_number,
		'message' => 'Support ticket created successfully!'
	));
}

function pixxi_ajax_get_ticket_comments() {
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
	
	if(empty($ticket_id)) {
		wp_send_json_error(array('message' => 'Invalid ticket ID'));
		return;
	}
	
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	
	// Get ticket info (including subject)
	$ticket = $wpdb->get_row($wpdb->prepare(
		"SELECT ticket_number, status, subject, issue_description FROM $tickets_table WHERE ticket_id = %d",
		$ticket_id
	), ARRAY_A);
	
	if(!$ticket) {
		wp_send_json_error(array('message' => 'Ticket not found'));
		return;
	}
	
	// Get comments
	$comments = $wpdb->get_results($wpdb->prepare(
		"SELECT * FROM $comments_table WHERE ticket_id = %d ORDER BY created_at ASC",
		$ticket_id
	), ARRAY_A);
	
	wp_send_json_success(array(
		'ticket_number' => $ticket['ticket_number'],
		'status' => $ticket['status'],
		'subject' => $ticket['subject'] ? $ticket['subject'] : '',
		'issue_description' => $ticket['issue_description'] ? $ticket['issue_description'] : '',
		'comments' => $comments
	));
}

function pixxi_ajax_get_all_tickets() {
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	
	// Get all tickets with comment count (including subject field and unread status)
	$tickets = $wpdb->get_results(
		"SELECT t.ticket_id, t.ticket_number, t.user_name, t.site_url, t.subject, t.email, t.issue_description, t.status, t.created_at, t.updated_at, 
			(SELECT COUNT(*) FROM $comments_table c WHERE c.ticket_id = t.ticket_id) as comment_count,
			(SELECT MAX(created_at) FROM $comments_table c WHERE c.ticket_id = t.ticket_id) as last_comment_at,
			(SELECT COUNT(*) FROM $comments_table c WHERE c.ticket_id = t.ticket_id AND c.user_type = 'admin' AND c.read_at IS NULL) as has_unread
		FROM $tickets_table t 
		ORDER BY t.updated_at DESC, t.created_at DESC",
		ARRAY_A
	);
	
	wp_send_json_success(array(
		'tickets' => $tickets
	));
}

function pixxi_ajax_add_ticket_comment() {
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
	$message = isset($_POST['message']) ? sanitize_textarea_field($_POST['message']) : '';
	
	if(empty($ticket_id) || empty($message)) {
		wp_send_json_error(array('message' => 'Ticket ID and message are required'));
		return;
	}
	
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	
	// Verify ticket exists
	$ticket = $wpdb->get_row($wpdb->prepare(
		"SELECT ticket_number FROM $tickets_table WHERE ticket_id = %d",
		$ticket_id
	), ARRAY_A);
	
	if(!$ticket) {
		wp_send_json_error(array('message' => 'Ticket not found'));
		return;
	}
	
	// Insert comment
	$result = $wpdb->insert(
		$wpdb->prefix . 'pixxi_support_comments',
		array(
			'ticket_id' => $ticket_id,
			'user_type' => 'user',
			'message' => $message,
			'created_at' => current_time('mysql')
		),
		array('%d', '%s', '%s', '%s')
	);
	
	if($result === false) {
		wp_send_json_error(array('message' => 'Failed to add comment'));
		return;
	}
	
	// Send comment to devmizan.com API
	// Use ticket_number instead of ticket_id since ticket_id differs between sites
	$api_data = array(
		'ticket_number' => $ticket['ticket_number'],
		'user_type' => 'user',
		'message' => $message
	);
	
	$api_response = wp_remote_post('https://devmizan.com/wp-json/pixxi-license/v1/support/add-comment', array(
		'body' => json_encode($api_data),
		'headers' => array(
			'Content-Type' => 'application/json'
		),
		'timeout' => 30
	));
	
	// Check API response and log errors
	if (is_wp_error($api_response)) {
		error_log('Pixxi Support: Failed to send comment to devmizan.com - ' . $api_response->get_error_message());
	} else {
		$response_code = wp_remote_retrieve_response_code($api_response);
		$response_body = wp_remote_retrieve_body($api_response);
		
		if ($response_code !== 200) {
			error_log('Pixxi Support: API returned error code ' . $response_code . ' for comment - ' . $response_body);
		} else {
			$api_result = json_decode($response_body, true);
			if (isset($api_result['success']) && $api_result['success']) {
				error_log('Pixxi Support: Comment successfully sent to devmizan.com - Ticket #' . $ticket['ticket_number']);
			} else {
				error_log('Pixxi Support: API returned success=false for comment - ' . $response_body);
			}
		}
	}
	
	// Send email notification to admin
	global $wpdb;
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	$ticket_info = $wpdb->get_row($wpdb->prepare(
		"SELECT user_name, site_url FROM $tickets_table WHERE ticket_id = %d",
		$ticket_id
	), ARRAY_A);
	
	pixxi_send_support_email('admin', 'new_comment', array(
		'ticket_number' => $ticket['ticket_number'],
		'user_name' => $ticket_info['user_name'] ?? '',
		'site_url' => $ticket_info['site_url'] ?? '',
		'message' => $message
	));
	
	// Get updated comments
	$comments = $wpdb->get_results($wpdb->prepare(
		"SELECT * FROM {$wpdb->prefix}pixxi_support_comments WHERE ticket_id = %d ORDER BY created_at ASC",
		$ticket_id
	), ARRAY_A);
	
	wp_send_json_success(array(
		'comments' => $comments,
		'message' => 'Comment added successfully'
	));
}

function pixxi_ajax_support_webhook() {
	// Webhook endpoint for admin comments from devmizan.com
	// This should be secured with an API key or token
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$ticket_number = isset($_POST['ticket_number']) ? sanitize_text_field($_POST['ticket_number']) : '';
	$message = isset($_POST['message']) ? sanitize_textarea_field($_POST['message']) : '';
	
	if(empty($ticket_number) || empty($message)) {
		wp_send_json_error(array('message' => 'Invalid data'));
		return;
	}
	
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	
	// Get ticket by ticket number
	$ticket = $wpdb->get_row($wpdb->prepare(
		"SELECT ticket_id, user_name, site_url FROM $tickets_table WHERE ticket_number = %s",
		$ticket_number
	), ARRAY_A);
	
	if(!$ticket) {
		wp_send_json_error(array('message' => 'Ticket not found'));
		return;
	}
	
	// Insert admin comment
	$result = $wpdb->insert(
		$wpdb->prefix . 'pixxi_support_comments',
		array(
			'ticket_id' => $ticket['ticket_id'],
			'user_type' => 'admin',
			'message' => $message,
			'created_at' => current_time('mysql')
		),
		array('%d', '%s', '%s', '%s')
	);
	
	if($result === false) {
		wp_send_json_error(array('message' => 'Failed to add comment'));
		return;
	}
	
	// Send email notification to user
	pixxi_send_support_email('user', 'admin_response', array(
		'ticket_number' => $ticket_number,
		'user_name' => $ticket['user_name'],
		'site_url' => $ticket['site_url'],
		'message' => $message
	));
	
	wp_send_json_success(array('message' => 'Comment added successfully'));
}

// Register REST API route for support webhook
function pixxi_register_support_webhook_route() {
	register_rest_route('pixxi-license/v1', '/support/webhook', array(
		'methods' => 'POST',
		'callback' => 'pixxi_rest_support_webhook',
		'permission_callback' => '__return_true',
	));
}

function pixxi_rest_support_webhook($request) {
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$params = $request->get_json_params();
	
	$ticket_number = isset($params['ticket_number']) ? sanitize_text_field($params['ticket_number']) : '';
	$message = isset($params['message']) ? sanitize_textarea_field($params['message']) : '';
	
	if(empty($ticket_number) || empty($message)) {
		return new WP_Error('invalid_data', 'Ticket number and message are required', array('status' => 400));
	}
	
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	
	// Get ticket by ticket number
	$ticket = $wpdb->get_row($wpdb->prepare(
		"SELECT ticket_id, user_name, site_url FROM $tickets_table WHERE ticket_number = %s",
		$ticket_number
	), ARRAY_A);
	
	if(!$ticket) {
		return new WP_Error('ticket_not_found', 'Ticket not found', array('status' => 404));
	}
	
	// Insert admin comment
	$result = $wpdb->insert(
		$wpdb->prefix . 'pixxi_support_comments',
		array(
			'ticket_id' => $ticket['ticket_id'],
			'user_type' => 'admin',
			'message' => $message,
			'created_at' => current_time('mysql')
		),
		array('%d', '%s', '%s', '%s')
	);
	
	if($result === false) {
		return new WP_Error('db_error', 'Failed to add comment', array('status' => 500));
	}
	
	// Update ticket updated_at
	$wpdb->update(
		$tickets_table,
		array('updated_at' => current_time('mysql')),
		array('ticket_id' => $ticket['ticket_id'])
	);
	
	// Send email notification to user
	pixxi_send_support_email('user', 'admin_response', array(
		'ticket_number' => $ticket_number,
		'user_name' => $ticket['user_name'],
		'site_url' => $ticket['site_url'],
		'message' => $message
	));
	
	return rest_ensure_response(array(
		'success' => true,
		'message' => 'Comment added successfully'
	));
}

function pixxi_ajax_get_unread_count() {
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	
	// Count unread admin comments (admin comments that haven't been read)
	$unread_count = $wpdb->get_var(
		"SELECT COUNT(*) FROM $comments_table c
		INNER JOIN $tickets_table t ON c.ticket_id = t.ticket_id
		WHERE c.user_type = 'admin' AND c.read_at IS NULL"
	);
	
	wp_send_json_success(array(
		'unread_count' => (int) $unread_count
	));
}

function pixxi_ajax_mark_comments_read() {
	check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
	if(!current_user_can('manage_options')) {
		wp_send_json_error(array('message' => 'Unauthorized'));
		return;
	}
	
	// Ensure tables exist
	pixxi_ensure_support_tables();
	
	global $wpdb;
	$ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
	
	if(empty($ticket_id)) {
		wp_send_json_error(array('message' => 'Invalid ticket ID'));
		return;
	}
	
	$comments_table = $wpdb->prefix . 'pixxi_support_comments';
	
	// Mark all admin comments for this ticket as read
	$wpdb->query($wpdb->prepare(
		"UPDATE $comments_table SET read_at = %s WHERE ticket_id = %d AND user_type = 'admin' AND read_at IS NULL",
		current_time('mysql'),
		$ticket_id
	));
	
	wp_send_json_success(array('message' => 'Comments marked as read'));
}

// AJAX Handler: Sync Single Property
function pixxi_ajax_sync_single_property() {
	// Clean output buffers
	if (ob_get_level()) {
		ob_end_clean();
	}
	
	// Set proper headers
	header('Content-Type: application/json');
	
	try {
		check_ajax_referer('pixxi_dashboard_nonce', 'nonce');
		if(!current_user_can('manage_options')) {
			wp_send_json_error(array('message' => 'Unauthorized'));
			return;
		}
		
		$sku = isset($_POST['sku']) ? sanitize_text_field($_POST['sku']) : '';
		
		if(empty($sku)) {
			wp_send_json_error(array('message' => 'Invalid SKU'));
			return;
		}
		
		global $wpdb;
		
		// Get property data from database
		$property = $wpdb->get_row($wpdb->prepare(
			"SELECT * FROM " . PIXXI_TABLE . " WHERE sku = %s",
			$sku
		), ARRAY_A);
		
		if(!$property || empty($property['data'])) {
			wp_send_json_error(array('message' => 'Property data not found'));
			return;
		}
		
		$pd = json_decode($property['data'], true);
		if(!$pd || !isset($pd['id'])) {
			wp_send_json_error(array('message' => 'Invalid property data'));
			return;
		}
		
		// Initialize property class - use correct class name
		$pixxi_property = new pixxi_property_class();
		
		// Check if property already exists
		$existing_posts = get_posts(array(
			'post_type' => 'property',
			'meta_key' => '_xm_api_sku',
			'meta_value' => $sku,
			'posts_per_page' => 1,
			'fields' => 'ids',
		));
		
		$post_id = !empty($existing_posts) ? $existing_posts[0] : null;
		
		if($post_id) {
			// Update existing property
			// Save full API data
			update_post_meta($post_id, 'pixi_import_data', json_encode($pd));
			
			// Apply all field mappings
			$pixxi_property->apply_field_mappings($post_id, $pd);
			
			// Apply taxonomy mappings
			$pixxi_property->apply_taxonomy_mappings($post_id, $pd);
			
			// Geolocation
			$coords = $pixxi_property->extract_coordinates_from_api($pd);
			if($coords) {
				update_post_meta($post_id, 'houzez_geolocation_lat', $coords['lat']);
				update_post_meta($post_id, 'houzez_geolocation_long', $coords['lng']);
				$pixxi_property->set_houzez_map_data($post_id, $pd, $coords);
			} else {
				update_post_meta($post_id, 'fave_property_map', 0);
			}
			
			// Update post title and content if needed
			$title = !empty($pd['name']) ? $pd['name'] : (isset($pd['title']) ? $pd['title'] : '');
			$desc = !empty($pd['description']) ? $pd['description'] : '';
			
			if(!empty($title)) {
				wp_update_post(array(
					'ID' => $post_id,
					'post_title' => $title,
					'post_content' => $desc
				));
			}
			
			wp_send_json_success(array(
				'message' => 'Property updated successfully',
				'post_id' => $post_id,
				'action' => 'updated'
			));
		} else {
			// Insert new property
			$api_data = array($sku => $property['data']);
			$insert_sku = array($sku);
			
			$result = $pixxi_property->insert_sku($insert_sku, $api_data);
			
			if(isset($result['error'])) {
				wp_send_json_error(array('message' => $result['error']));
				return;
			}
			
			// Get the newly created post ID
			$new_posts = get_posts(array(
				'post_type' => 'property',
				'meta_key' => '_xm_api_sku',
				'meta_value' => $sku,
				'posts_per_page' => 1,
				'fields' => 'ids',
			));
			
			$new_post_id = !empty($new_posts) ? $new_posts[0] : null;
			
			wp_send_json_success(array(
				'message' => 'Property synced successfully',
				'post_id' => $new_post_id,
				'action' => 'created'
			));
		}
	} catch(Exception $e) {
		wp_send_json_error(array(
			'message' => 'Error syncing property: ' . $e->getMessage(),
			'file' => $e->getFile(),
			'line' => $e->getLine()
		));
	} catch(Error $e) {
		wp_send_json_error(array(
			'message' => 'Fatal error: ' . $e->getMessage(),
			'file' => $e->getFile(),
			'line' => $e->getLine()
		));
	}
}

// Email notification function
function pixxi_send_support_email($recipient, $type, $data) {
	$to = '';
	$subject = '';
	$message = '';
	
	if($recipient === 'admin') {
		// Always send admin emails to admin@devmizan.com
		$to = 'admin@devmizan.com';
		
		if($type === 'new_ticket') {
			$subject = 'New Support Ticket #' . $data['ticket_number'];
			$message = "A new support ticket has been created:\n\n";
			$message .= "Ticket Number: " . $data['ticket_number'] . "\n";
			$message .= "User Name: " . $data['user_name'] . "\n";
			$message .= "Email: " . (isset($data['email']) ? $data['email'] : 'N/A') . "\n";
			$message .= "Subject: " . (isset($data['subject']) ? $data['subject'] : 'N/A') . "\n";
			$message .= "Site URL: " . $data['site_url'] . "\n";
			$message .= "Issue Description:\n" . $data['issue_description'] . "\n\n";
			$message .= "View ticket: https://devmizan.com/wp-admin/admin.php?page=pixxi-license-support&ticket=" . $data['ticket_number'];
		} elseif($type === 'new_comment') {
			$subject = 'New Comment on Ticket #' . $data['ticket_number'];
			$message = "A new comment has been added to ticket #" . $data['ticket_number'] . ":\n\n";
			$message .= "User: " . (isset($data['user_name']) ? $data['user_name'] : 'Customer') . "\n";
			$message .= "Site: " . (isset($data['site_url']) ? $data['site_url'] : 'N/A') . "\n\n";
			$message .= "Comment:\n" . $data['message'] . "\n\n";
			$message .= "View ticket: https://devmizan.com/wp-admin/admin.php?page=pixxi-license-support&ticket=" . $data['ticket_number'];
		}
	} elseif($recipient === 'user') {
		// Get user email from ticket data
		global $wpdb;
		$tickets_table = $wpdb->prefix . 'pixxi_support_tickets';
		$ticket = $wpdb->get_row($wpdb->prepare(
			"SELECT email, user_name, site_url FROM $tickets_table WHERE ticket_number = %s",
			$data['ticket_number']
		), ARRAY_A);
		
		// Use email from ticket if available, otherwise try WordPress user
		if(!empty($ticket['email'])) {
			$to = $ticket['email'];
		} else {
			// Try to get email from WordPress user
			$user = get_user_by('login', $ticket['user_name']);
			if($user) {
				$to = $user->user_email;
			} else {
				// Fallback to admin email
				$to = get_option('admin_email');
			}
		}
		
		if($type === 'admin_response') {
			$subject = 'Response to Support Ticket #' . $data['ticket_number'];
			$message = "Hello " . $data['user_name'] . ",\n\n";
			$message .= "We have responded to your support ticket #" . $data['ticket_number'] . ":\n\n";
			$message .= $data['message'] . "\n\n";
			$message .= "View your ticket in the dashboard: " . admin_url('admin.php?page=pixxi-property-importer');
		}
	}
	
	if(!empty($to) && !empty($subject) && !empty($message)) {
		$headers = array('Content-Type: text/html; charset=UTF-8');
		wp_mail($to, $subject, nl2br(esc_html($message)), $headers);
	}
}

// Function to display all Pixxi data
function pixxi_view_all_data_page() {
	global $wpdb;
	
	// Pagination
	$per_page = 20;
	$current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
	$offset = ($current_page - 1) * $per_page;
	
	// Search functionality
	$search_query = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
	$where_clause = '';
	if (!empty($search_query)) {
		$where_clause = $wpdb->prepare("WHERE data LIKE %s", '%' . $wpdb->esc_like($search_query) . '%');
	}
	
	// Get total count
	$total_query = "SELECT COUNT(*) FROM " . PIXXI_TABLE . " " . $where_clause;
	$total_items = $wpdb->get_var($total_query);
	$total_pages = ceil($total_items / $per_page);
	
	// Get properties
	$query = "SELECT * FROM " . PIXXI_TABLE . " " . $where_clause . " ORDER BY updated DESC LIMIT %d OFFSET %d";
	$properties = $wpdb->get_results($wpdb->prepare($query, $per_page, $offset), ARRAY_A);
	
	// Count properties by listing type
	$off_plan_count = 0;
	$sell_count = 0;
	$rent_count = 0;
	
	// Get all properties to count by type (not just paginated ones)
	$all_properties_query = "SELECT data FROM " . PIXXI_TABLE . " " . $where_clause;
	$all_properties = $wpdb->get_results($all_properties_query, ARRAY_A);
	
	foreach($all_properties as $prop) {
		if(empty($prop['data'])) continue;
		
		$data = json_decode($prop['data'], true);
		if(!$data || !isset($data['listingType'])) continue;
		
		$listing_type = $data['listingType'];
		if($listing_type === 'NEW') {
			$off_plan_count++;
		} elseif($listing_type === 'SELL') {
			$sell_count++;
		} elseif($listing_type === 'RENT') {
			$rent_count++;
		}
	}
	
	?>
	<div class="wrap pixxi-dashboard-wrapper">
		<div class="pixxi-dashboard-header">
			<div class="pixxi-header-content">
				<div class="pixxi-header-left">
					<h1 class="pixxi-dashboard-title">View All Pixxi Data</h1>
					<p class="pixxi-dashboard-subtitle">Browse all properties fetched from Pixxi API</p>
				</div>
				<div class="pixxi-header-actions">
					<a href="<?php echo admin_url('admin.php?page=pixxi-property-importer'); ?>" class="pixxi-header-btn pixxi-btn-secondary">
						<span class="dashicons dashicons-arrow-left-alt"></span>
						Back to Dashboard
					</a>
				</div>
			</div>
		</div>
		
		<!-- Property Type Statistics Cards -->
		<div class="pixxi-stats-grid" style="margin-bottom: 24px;">
			<div class="pixxi-stat-card">
				<div class="pixxi-stat-icon" style="background: #F1F5F9;">
					<span class="dashicons dashicons-building" style="color: #64748B;"></span>
				</div>
				<div class="pixxi-stat-content">
					<div class="pixxi-stat-value"><?php echo number_format($total_items); ?></div>
					<div class="pixxi-stat-label">Total Properties</div>
				</div>
			</div>
			
			<div class="pixxi-stat-card">
				<div class="pixxi-stat-icon" style="background: #F0FDF4;">
					<span class="dashicons dashicons-building" style="color: #10B981;"></span>
				</div>
				<div class="pixxi-stat-content">
					<div class="pixxi-stat-value"><?php echo number_format($off_plan_count); ?></div>
					<div class="pixxi-stat-label">Off Plan Properties</div>
				</div>
			</div>
			
			<div class="pixxi-stat-card">
				<div class="pixxi-stat-icon" style="background: #EFF6FF;">
					<span class="dashicons dashicons-cart" style="color: #3B82F6;"></span>
				</div>
				<div class="pixxi-stat-content">
					<div class="pixxi-stat-value"><?php echo number_format($sell_count); ?></div>
					<div class="pixxi-stat-label">Sell Properties</div>
				</div>
			</div>
			
			<div class="pixxi-stat-card">
				<div class="pixxi-stat-icon" style="background: #FEF3C7;">
					<span class="dashicons dashicons-admin-home" style="color: #F59E0B;"></span>
				</div>
				<div class="pixxi-stat-content">
					<div class="pixxi-stat-value"><?php echo number_format($rent_count); ?></div>
					<div class="pixxi-stat-label">Rent Properties</div>
				</div>
			</div>
		</div>
		
		<div class="pixxi-card">
			<div class="pixxi-card-header">
				<h2 class="pixxi-card-title">Properties Data</h2>
				<div class="pixxi-card-actions">
					<span class="pixxi-badge pixxi-badge-info">Total: <?php echo number_format($total_items); ?> properties</span>
				</div>
			</div>
			
			<div class="pixxi-card-body">
				<!-- Search Form -->
				<div class="pixxi-search-box" style="margin-bottom: 24px;">
					<form method="get" action="" style="display: flex; gap: 12px; align-items: center;">
						<input type="hidden" name="page" value="pixxi-view-all-data">
						<input 
							type="text" 
							name="s" 
							value="<?php echo esc_attr($search_query); ?>" 
							placeholder="Search by ID, name, property ID, city, developer..." 
							class="pixxi-input"
							style="flex: 1; max-width: 500px;"
						>
						<button type="submit" class="pixxi-header-btn pixxi-btn-primary">
							<span class="dashicons dashicons-search"></span>
							Search
						</button>
						<?php if (!empty($search_query)): ?>
							<a href="<?php echo admin_url('admin.php?page=pixxi-view-all-data'); ?>" class="pixxi-header-btn pixxi-btn-secondary">
								Clear
							</a>
						<?php endif; ?>
					</form>
				</div>
				
				<?php if (empty($properties)): ?>
					<div class="pixxi-empty-state">
						<span class="dashicons dashicons-database" style="font-size: 48px; color: #94A3B8; margin-bottom: 16px;"></span>
						<h3 style="color: #0F172A; margin: 0 0 8px 0;">No properties found</h3>
						<p style="color: #475569; margin: 0;">
							<?php if (!empty($search_query)): ?>
								No properties match your search criteria. Try a different search term.
							<?php else: ?>
								No properties have been fetched from Pixxi yet. Go to the dashboard to download properties.
							<?php endif; ?>
						</p>
					</div>
				<?php else: ?>
					<!-- Properties Table -->
					<div class="pixxi-table-wrapper">
						<table class="pixxi-table">
							<thead>
								<tr>
									<th style="width: 80px; min-width: 80px;">ID</th>
									<th style="width: 120px; min-width: 120px;">Property ID</th>
									<th style="width: auto; min-width: 200px;">Name/Title</th>
									<th style="width: 150px; min-width: 150px;">City</th>
									<th style="width: 150px; min-width: 150px;">Developer</th>
									<th style="width: 120px; min-width: 120px;">Price</th>
									<th style="width: 100px; min-width: 100px;">Status</th>
									<th style="width: 120px; min-width: 120px;">Updated</th>
									<th style="width: 100px; min-width: 100px;">Actions</th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ($properties as $property): 
									$data = json_decode($property['data'], true);
									if (!$data) continue;
									
									$property_id = isset($data['id']) ? $data['id'] : '';
									$property_pid = isset($data['propertyId']) ? $data['propertyId'] : '';
									$name = isset($data['name']) ? $data['name'] : (isset($data['title']) ? $data['title'] : 'N/A');
									$city = isset($data['cityName']) ? $data['cityName'] : (isset($data['city']) ? $data['city'] : 'N/A');
									$developer = isset($data['developer']) ? $data['developer'] : (isset($data['developerName']) ? $data['developerName'] : 'N/A');
									$price = isset($data['price']) ? number_format($data['price']) : 'N/A';
									$status = isset($data['status']) ? $data['status'] : (isset($data['Status']) ? $data['Status'] : 'N/A');
									$listing_type = isset($data['listingType']) ? $data['listingType'] : '';
									
									// Handle updated timestamp - can be integer timestamp or datetime string
									$updated = 'N/A';
									if (!empty($property['updated'])) {
										$updated_timestamp = null;
										
										// Check if it's already a timestamp (integer)
										if (is_numeric($property['updated']) && $property['updated'] > 0) {
											$updated_timestamp = (int)$property['updated'];
										} 
										// Try to parse as datetime string
										elseif (is_string($property['updated']) && $property['updated'] !== '0000-00-00 00:00:00') {
											$updated_timestamp = strtotime($property['updated']);
										}
										
										// Format if we have a valid timestamp
										if ($updated_timestamp && $updated_timestamp > 0) {
											$updated = date('Y-m-d H:i', $updated_timestamp);
										}
									}
									
									$synced = $property['pid'] > 0 ? true : false;
								?>
									<tr>
										<td>
											<strong><?php echo esc_html($property_id); ?></strong>
										</td>
										<td>
											<code style="font-size: 12px; background: #F1F5F9; padding: 4px 8px; border-radius: 4px;">
												<?php echo esc_html($property_pid); ?>
											</code>
										</td>
										<td>
											<div style="font-weight: 500; color: #0F172A; margin-bottom: 4px;">
												<?php echo esc_html(wp_trim_words($name, 8)); ?>
											</div>
											<?php if ($synced): ?>
												<span class="pixxi-badge pixxi-badge-success" style="font-size: 11px;">
													Synced (PID: <?php echo $property['pid']; ?>)
												</span>
											<?php else: ?>
												<span class="pixxi-badge pixxi-badge-warning" style="font-size: 11px;">
													Not Synced
												</span>
											<?php endif; ?>
										</td>
										<td><?php echo esc_html($city); ?></td>
										<td><?php echo esc_html($developer); ?></td>
										<td>
											<?php if ($price !== 'N/A'): ?>
												<strong style="color: #10B981;">AED <?php echo esc_html($price); ?></strong>
											<?php else: ?>
												<span style="color: #94A3B8;">N/A</span>
											<?php endif; ?>
										</td>
										<td>
											<?php if ($listing_type): ?>
												<span class="pixxi-badge pixxi-badge-info"><?php echo esc_html($listing_type); ?></span>
											<?php else: ?>
												<span style="color: #94A3B8;"><?php echo esc_html($status); ?></span>
											<?php endif; ?>
										</td>
										<td style="color: #475569; font-size: 13px;">
											<?php echo esc_html($updated); ?>
										</td>
										<td>
											<div style="display: flex; gap: 8px; align-items: center;">
												<button 
													type="button" 
													class="pixxi-sync-property-btn" 
													data-property-id="<?php echo esc_attr($property['id']); ?>"
													data-sku="<?php echo esc_attr($property['sku']); ?>"
													style="padding: 6px 12px; font-size: 12px; background: #10B981; color: white; border: none; border-radius: 6px; cursor: pointer; transition: all 0.2s;"
													onmouseover="this.style.background='#059669'"
													onmouseout="this.style.background='#10B981'"
												>
													Sync
												</button>
												<button 
													type="button" 
													class="pixxi-view-json-btn" 
													data-property-id="<?php echo esc_attr($property['id']); ?>"
													style="padding: 6px 12px; font-size: 12px; background: #6366F1; color: white; border: none; border-radius: 6px; cursor: pointer; transition: all 0.2s;"
													onmouseover="this.style.background='#4F46E5'"
													onmouseout="this.style.background='#6366F1'"
												>
													View JSON
												</button>
											</div>
										</td>
									</tr>
									<!-- Hidden JSON Data Row -->
									<tr class="pixxi-json-row" id="json-row-<?php echo esc_attr($property['id']); ?>" style="display: none;">
										<td colspan="9" style="padding: 0; border-top: 2px solid #E2E8F0;">
											<div style="background: #F8FAFC; border-left: 1px solid #E2E8F0; border-right: 1px solid #E2E8F0; border-bottom: 1px solid #E2E8F0; border-radius: 0 0 8px 8px; padding: 16px; margin: 0;">
												<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 12px;">
													<h4 style="margin: 0; color: #0F172A; font-size: 14px; font-weight: 600;">Full JSON Data</h4>
													<button 
														type="button" 
														class="pixxi-close-json-btn" 
														data-property-id="<?php echo esc_attr($property['id']); ?>"
														style="padding: 4px 8px; font-size: 12px; background: #EF4444; color: white; border: none; border-radius: 4px; cursor: pointer;"
													>
														Close
													</button>
												</div>
												<pre style="background: #FFFFFF; border: 1px solid #E2E8F0; border-radius: 6px; padding: 16px; overflow-x: auto; max-height: 500px; overflow-y: auto; font-family: 'JetBrains Mono', monospace; font-size: 12px; line-height: 1.6; color: #0F172A; margin: 0;"><?php echo esc_html(json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)); ?></pre>
											</div>
										</td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
					</div>
					
					<!-- Pagination -->
					<?php if ($total_pages > 1): ?>
						<div class="pixxi-pagination" style="margin-top: 24px; display: flex; justify-content: center; align-items: center; gap: 8px;">
							<?php
							$base_url = admin_url('admin.php?page=pixxi-view-all-data');
							if (!empty($search_query)) {
								$base_url .= '&s=' . urlencode($search_query);
							}
							
							// Previous button
							if ($current_page > 1):
								$prev_url = $base_url . '&paged=' . ($current_page - 1);
							?>
								<a href="<?php echo esc_url($prev_url); ?>" class="pixxi-header-btn pixxi-btn-secondary" style="padding: 8px 16px;">
									<span class="dashicons dashicons-arrow-left-alt2"></span>
									Previous
								</a>
							<?php endif; ?>
							
							<span style="color: #475569; padding: 0 16px;">
								Page <?php echo $current_page; ?> of <?php echo $total_pages; ?>
							</span>
							
							<?php
							// Next button
							if ($current_page < $total_pages):
								$next_url = $base_url . '&paged=' . ($current_page + 1);
							?>
								<a href="<?php echo esc_url($next_url); ?>" class="pixxi-header-btn pixxi-btn-primary" style="padding: 8px 16px;">
									Next
									<span class="dashicons dashicons-arrow-right-alt2"></span>
								</a>
							<?php endif; ?>
						</div>
					<?php endif; ?>
				<?php endif; ?>
			</div>
		</div>
	</div>
	
	<script>
	jQuery(document).ready(function($) {
		// Toggle JSON view
		$('.pixxi-view-json-btn').on('click', function() {
			var propertyId = $(this).data('property-id');
			$('#json-row-' + propertyId).slideDown(200);
			$(this).text('Hide JSON').css('background', '#EF4444');
		});
		
		// Close JSON view
		$('.pixxi-close-json-btn').on('click', function() {
			var propertyId = $(this).data('property-id');
			$('#json-row-' + propertyId).slideUp(200);
			$('.pixxi-view-json-btn[data-property-id="' + propertyId + '"]').text('View JSON').css('background', '#6366F1');
		});
		
		// Sync single property
		$('.pixxi-sync-property-btn').on('click', function() {
			var $button = $(this);
			var sku = $button.data('sku');
			var propertyId = $button.data('property-id');
			
			if(!sku) {
				alert('Invalid property SKU');
				return;
			}
			
			// Disable button and show loading state
			$button.prop('disabled', true);
			var originalText = $button.text();
			$button.text('Syncing...').css('background', '#6B7280');
			
			// Send AJAX request
			$.ajax({
				url: '<?php echo admin_url('admin-ajax.php'); ?>',
				type: 'POST',
				data: {
					action: 'pixxi_sync_single_property',
					sku: sku,
					nonce: '<?php echo wp_create_nonce('pixxi_dashboard_nonce'); ?>'
				},
				success: function(response) {
					if(response.success) {
						$button.text('Synced!').css('background', '#10B981');
						setTimeout(function() {
							$button.text(originalText).css('background', '#10B981').prop('disabled', false);
						}, 2000);
						
						// Show success message
						if(response.data && response.data.message) {
							alert(response.data.message);
						}
						
						// Reload page after 2 seconds to show updated status
						setTimeout(function() {
							location.reload();
						}, 2000);
					} else {
						$button.text(originalText).css('background', '#10B981').prop('disabled', false);
						alert(response.data && response.data.message ? response.data.message : 'Error syncing property');
					}
				},
				error: function(xhr, status, error) {
					$button.text(originalText).css('background', '#10B981').prop('disabled', false);
					alert('Error syncing property: ' + error);
				}
			});
		});
	});
	</script>
	
	<style>
	.pixxi-empty-state {
		text-align: center;
		padding: 64px 24px;
		background: #F8FAFC;
		border-radius: 12px;
		border: 1px dashed #E2E8F0;
	}
	
	.pixxi-table-wrapper {
		overflow-x: auto;
		border-radius: 8px;
		border: 1px solid #E2E8F0;
		position: relative;
		min-width: 100%;
	}
	
	.pixxi-table {
		width: 100%;
		border-collapse: collapse;
		background: #FFFFFF;
		table-layout: fixed;
		min-width: 1200px;
	}
	
	.pixxi-table thead {
		background: #F8FAFC;
		border-bottom: 2px solid #E2E8F0;
	}
	
	.pixxi-table th {
		padding: 16px;
		text-align: left;
		font-weight: 600;
		font-size: 13px;
		color: #0F172A;
		text-transform: uppercase;
		letter-spacing: 0.5px;
		background: #F8FAFC;
		white-space: nowrap;
	}
	
	.pixxi-table td {
		padding: 16px;
		border-bottom: 1px solid #E2E8F0;
		font-size: 14px;
		color: #475569;
	}
	
	.pixxi-table tbody tr:hover {
		background: #F8FAFC;
		transition: background 0.15s ease;
	}
	
	.pixxi-table tbody tr:last-child td {
		border-bottom: none;
	}
	
	.pixxi-table .pixxi-json-row td {
		padding: 0;
		background: #F8FAFC;
	}
	
	.pixxi-table .pixxi-json-row:hover {
		background: transparent;
	}
	
	.pixxi-badge {
		display: inline-block;
		padding: 4px 10px;
		border-radius: 6px;
		font-size: 12px;
		font-weight: 500;
		line-height: 1.4;
	}
	
	.pixxi-badge-success {
		background: #D1FAE5;
		color: #065F46;
	}
	
	.pixxi-badge-warning {
		background: #FEF3C7;
		color: #92400E;
	}
	
	.pixxi-badge-info {
		background: #DBEAFE;
		color: #1E40AF;
	}
	
	.pixxi-input {
		padding: 10px 16px;
		border: 1px solid #E2E8F0;
		border-radius: 6px;
		font-size: 14px;
		font-family: 'Inter', sans-serif;
		color: #0F172A;
		transition: all 0.2s;
	}
	
	.pixxi-input:focus {
		outline: none;
		border-color: #6366F1;
		box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
	}
	</style>
	<?php
}
?>